
//
// 1defdsim.cpp - a program so simulate a CV in a one dimensional system
//
// Assuming diffusion ocefficents for 'O' and 'R' are the same
//
// i=0 ELECTRODE ---> i=numx BULK SOLUTION
//
// Last Updated J.L.A. Thursday 17-09-1999

#include <iostream.h>
#include <fstream.h>
#include <functional>
#include <math.h>
#include <string.h>
#include <iomanip.h>

#define LAMDA 0.25				// lamda value
#define DELTA_O 1.0
#define MAXNUMX 1000			// maximum number of points along X
#define INITCONCO 1.0			// initial 'O' concentration
#define INITCONCR 0.0			// initial 'R' concentration

double mapping[] = {10, 19, 21, 25, 50, 59, 61, 65, 1E10};	// output concentration profiles at these dimensionless times

// Profiling
void main(void);
void map_the_conc(int lowpos, int highpos, double timepos, int timecount, bool test,
				  double *parray1, double *parray2, double dx);

void main(void)
{
	// user variables
	double Ztop;	// dimensionless distance to top of simulation domain
	int numx;		// number of points along domain
	double EndTime;	// Dimensionless End Time
	double Ehigh;	// Starting (upper potential)
	double Elow;	// Switching potential
	double Sr;		// Scan Rate

	// calculated variables
	double dx;		// dimensionless step size - distance
	double dt;		// dimensionless step size - time
	double de;		// dimensionless step size - potential
	int numt;		// number of time steps
	double expde;	// exponential of dimensionless step size - potential

	// calculation parameters
	double co1[MAXNUMX], co2[MAXNUMX];	// two arrays of species 'O' concentrations 
	double cr1[MAXNUMX], cr2[MAXNUMX];	// two arrays of species 'R' concentrations 
	double *pnewo, *poldo;				// pointers to arrays for 'O' species
	double *pnewr, *poldr;				// pointers to arrays for 'R' species

	// other variables
	int i, j;				// integer counters for loops
	double potential;		// current dimensionless potential
	double exppotential;	// exponential of current dimensionless potential
	bool scanup;			// are we scanning in positive potential direction
	double concgrad;		// concentration gradient at electrode (proportional to current)
	double timepos;			// current dimensionless time		
	int map_count;			// integer counter of number of concentration profiles produced
	ofstream fileDAT;		// output stream used for current values

stupidinput:
	// Welcome and Variable Input
	cout << endl << "ID Cyclic Voltammetry Simulation Program" << endl
		<< "Version 1.0" << endl << endl;
	cout << "Please input the following dimensionless parameters" << endl;
	cout << "Ztop        : "; cin >> Ztop;
	cout << "numx        : "; cin >> numx;
	cout << "End Time    : "; cin >> EndTime;
	cout << "Ehigh       : "; cin >> Ehigh;
	cout << "Elow        : "; cin >> Elow;
	cout << "Scan Rate   : "; cin >> Sr;

	// Basic Data Validation
	if ((numx>MAXNUMX)||(Ehigh<=Elow)||(Sr>(Ehigh-Elow))||(Ztop<0.1)||(numx<2)) 
	{
		cout << endl << "Stupid Input Values!" << endl;
		goto stupidinput;
	}

	dx = Ztop/(double)numx;
	dt = LAMDA * dx * dx;
	numt = (int)(EndTime / dt);
	de = Sr*dt;
	expde = exp(Sr*dt);

	cout << endl << "Starting Simulation" << endl;

	// Open and Setup Current Output Stream
	// opens current datafile
	fileDAT.open("current.dat");
	fileDAT.setf(ios::fixed, ios::floatfield);	// decimal notation
	fileDAT.setf(ios::showpoint);				// decimal point always shown
	fileDAT << setprecision(6);					// set number of decimal places

	// set pointers for first timestep
	pnewo = co1; poldo = co2;
	pnewr = cr1; poldr = cr2;

	// Initial Conditions
	for (i=0; i<=numx; i++)
	{
		pnewo[i] = INITCONCO; poldo[i] = INITCONCO;
		pnewr[i] = INITCONCR; poldr[i] = INITCONCR;
	}

	potential = Ehigh;
	exppotential = exp(potential);	
	scanup = false;
	timepos = 0.0;
	map_count = 0;

	// Start of time loop
	for (j=1; j<=numt; j++)
	{
		timepos = timepos + dt;

		// are we scanning upwards or downwards in potential?
		if (!scanup)
		{
			// if we are scanning downwards, lower the potential until we hit Elow
			potential = potential - de;
			exppotential = exppotential/expde;
			if (potential<Elow)
			{
				potential = potential + (2.0*de);
				scanup = true;
				exppotential = exppotential*expde*expde;
			}
		}
		if (scanup)
		{
			// if we are scanning upwards, raise the potential until we hit Ehigh
			potential = potential + de;
			exppotential = exppotential*expde;
			if (potential>Ehigh)
			{
				potential = potential + (2.0*de);
				exppotential = exppotential/expde/expde;
				scanup = true;
			}
		}

		// boundary conditions

		// electrode
		pnewo[0] = (poldr[1] + (DELTA_O*poldo[1]))/(DELTA_O + (1.0/exppotential));
		pnewr[0] = pnewo[0] / exppotential;

		//bulk solution
		pnewo[numx] = INITCONCO;
		pnewr[numx] = INITCONCR;

		for (i=1; i<=numx-1; i++)
		{
			pnewo[i] = (DELTA_O*LAMDA*poldo[i+1]) + ((1.0-(2.0*LAMDA*DELTA_O))*poldo[i]) + (LAMDA*DELTA_O*poldo[i-1]); 
			pnewr[i] = (LAMDA*poldr[i+1]) + ((1.0-(2.0*LAMDA))*poldr[i]) + (LAMDA*poldr[i-1]);
		}

		// Calculate and Output Concentration Gradient
		if ((j%20)==0)
		{
			concgrad = (pnewr[1] - pnewr[0]) / dx; 
			fileDAT << timepos << "\t" << potential << "\t" << concgrad << endl;
		}

		// Print Concentration Profiles if necessary
		if (timepos>=mapping[map_count])     
		{
			map_the_conc(0,numx,timepos,map_count,false,pnewo,pnewr,dx); 
			map_count = map_count+1;
		}

		// swap arrays before start again at top of the loop
		// arrays chosen depends on if odd or even timestep
		if ((j%2)!=0)
		{
			pnewo = co2; poldo = co1;
			pnewr = cr2; poldr = cr1;
		}
		else
		{
			pnewo = co1; poldo = co2;
			pnewr = cr1; poldr = cr2;
		}

	}

	// output final concentration profile
	map_the_conc(0,numx,999.9,map_count,true,pnewo,pnewr,dx);

	// Close Current data output stream
	fileDAT.close();
}

void map_the_conc(int lowpos, int highpos, double timepos, int timecount, bool test, double *parray1, double *parray2, double dx)
{

	// prints out a list of concentrations (range requested) from two arrays

	// Input Variables
	// lowpos	- lowest concentration cell to output
	// highpos	- highest concentration cell to output
	// timepos	- current time
	// timecount- number of previous lists produced
	// test		- is this the last cocentration printout?
	// parray1	- pointer to array 1
	// parray1	- pointer to array 2
	// dx		- dimensionless distance between concentration points

	// local variables
	ofstream mapOUT;				// output stream
	int i;							// integer counter	
	char concfile[] ={"xmap.smap"}; // set generic filename

	// set filename to appropriate string
	concfile[0]=(char)(timecount+65);
	if (test==true) { strcpy(concfile,"FILE.smap"); }
	
	// setup output stream
	mapOUT.open(concfile);
	mapOUT.setf(ios::fixed, ios::floatfield);
	mapOUT.setf(ios::showpoint);
	mapOUT << setprecision(8) << timepos << endl;

	for (i=lowpos; i<=highpos; i++)
    {
		mapOUT << (double)i*dx << "\t" << parray1[i] << "\t" << parray2[i] << endl;
    }

	mapOUT.close();
	return;
}

