

// WINCUSIMPOS.cpp - A copper(II)/(I)/(0) Cyclic Voltrammetry Simulation
//              
//				A microelectrode response close to a substrate
//				undergoing potential ramps is simulated.         
// 
//				Simulation for MS WINDOWS 32 bit
//
//				Grid Shape:
//
//				/////////////SUBSTRATE(CV)///////////////
//				/////////////////////////////////////////
//				|			Cu(0)						|
//				S										|
//				Y										|
//				M										|
//				M			Cu(I), Cu(II), Cl-			|
//				|			solution					|
//				A										|
//				X										B	
//				I										|
//				S										U
//				|										|
//				|___________...............				L
//				|			...............				|
//				| electrode	... glass .....				K
//				|			...............				|
//		   ^	|	fixed	...............				|
//		Z  |	| potential	...............				|
//				|			...............				|
//				|			...............				|
//				|			...............				|
//              BULK
//			
//					R ->
//
//				This program implements ADI in two dimensions using
//				fully implicit boundary conditions and expanding
//				space and time grids
//
//				Version 2. Implements a better approximation of the 
//				numcleation loop for having a potential for instantaneous
//				nucleation when plating. We then strip until all copper(0)
//				has been oxidised. 
//
//				J.L.Amphlett

// microdisc being held at +0.5V vs SCE

// Last Updated : Wednesday 27 January 1999
// Last Windows Update : Tuesday 24 August 1999

// Includes and Definitions

#include <iostream.h>
#include <functional>
#include <math.h>
#include <string.h>
#include <fstream.h>
#include <iomanip.h>
#include <time.h>

#undef MAX_X
#undef MAX_Y
#undef M_PI
#undef idcode
#define MAX_X 1000
#define MAX_Y 1000
#define M_PI 3.141592654
#define idcode -1

// Prototypes
int main(void);								// main control subrountine
double welcome(void);						// welcomes user+input initial variables+calculate grid/time/potential parameters
int ferrors(int e_code);					// turn simulation error codes into user error messages
void clearbulk(int a, int b, int c, int d);	// clear the matrix square (with bulk concentrations) defined by a,b,c,d
void id_block(int a,int b,int c,int d);     // id's the microdisc with a marker concentration (idcode)
void adi_init(double dt);					// calculation of time independent ADI parameters
void adi_calc(double e_esub, double e_esub2, double e_eincr, double d_t);	// the concentration calculation sub-routine (needs exp(substrate potential),exp(sub potential relative to cu(0)) + its half step increment + time step size)
inline void perp(int a,int b,int c);		// set vertical boundary bulk values 
inline void parral(int a, int b, int c);	// set horizontal boundary bulk values
inline void electrode_p(int a,int b,int c);	// set radial electrode values (+0.5V)
inline void electrode_z(int a,int b,int c); // set radial electrode values (+0.0V)
inline double calc_tipcurrent_z(void);		// returns the dimensionless tip current (+0.0V)
inline double calc_tipcurrent_p(void);		// returns the dimensionless tip current (+0.5V)
inline double calc_subcurrent(double d_t);	// returns the substrate current (normalised wrt steady state microelectrode current in the bulk) - needs timestep size for plating/stripping current
inline void map_the_conc(int bl, int br, int lb, int lt, double timepos, int timecount, bool test); // prints out cocentration profile from boundaries, marks with timepos, map number and a bool to see if its the final map (1=YES)
inline double calc_cucoverage(void);		// returns Cu 0 coverage on the substrate

// Global Constants & Variables
const char *version[] = {"cuSIM_DOS rv 2.1b"};	// program version number
const char *mdisc_stat[] = {"held at +0.5V vs SCE"};	// what the microelectrode is doing
const double mapping[] = {73.0,85.0,89.0,89.5,92.0,103.0,109.0,118.0,133.0,1E10};			// array of times to print out concentration profiles IMP:last number large to avoid problems
const double e2 = -19.7;					// dimensionless Eo of Cu(I)/Cu(0) plating on Pt
const double nucl_loop = 4.0;				// stripping 1/2 wave potential (forms nucleation loop)
const double eta = 1.0;						// term of uncertainty in conservation of flux equation, units sec cm^-1
 
double delta;								// ration of CU(I):Cu(II) complex diffusion coefficients
double cob1, cob2;							// initial concentrations of CuII and CuI complexes
double einit, efinal, sr;					// dimensionless initial, final potentials and scanrate
double dx,dy;								// step change in virtual X and Y
double var_a, var_b;						// expansion grid coefficents
double de;									// change in dimensionless potential (per full ADI iteration)
int scanit;									// iterations reuired for 1 scan (right to left)
int ktmax;									// total number of simulation iterations (equal time teps)
int numd,numg,numbulk,numx,numxtot;			// i direction grid parameters
int numy1,numy,numytot,numdepth;			// j direction grid parameters

double eidisc,ejgsd;						// estimated grid size parameters used in welcome and adi_init
double dz_b1, depth, dz_m1, dz_m2, midz;	// calculated real space grid size parameters		

double cu0array1[MAX_X+2];					// copper 0 array along substrate
double cu0array2[MAX_X+2];
double cu1matrix1[MAX_X+2][MAX_Y+1];		// copper I concentration old and new matrices		
double cu1matrix2[MAX_X+2][MAX_Y+1];
double cu2matrix1[MAX_X+2][MAX_Y+1];		// copper II concentration old and new matrices
double cu2matrix2[MAX_X+2][MAX_Y+1];
double *pnew0;								// pointer to new Cu 0 concentration array
double *pold0;								// pointer to old Cu 0 concentration array
double (*pnew1)[MAX_Y+1];					// pointer to new Cu I concentration matrix
double (*pold1)[MAX_Y+1];					// pointer to old Cu I concentration matrix
double (*pnew2)[MAX_Y+1];					// pointer to new Cu II concentration matrix
double (*pold2)[MAX_Y+1];					// pointer to old Cu II concentration matrix

double dr_m1;								// smallest change in dr (electrode/glass boundary)
double posr[MAX_X+2],posz[MAX_Y+1];			// arrays with real position of points (adi_init)
double a2[MAX_X+2],a3[MAX_X+2],cofx[MAX_X+2];	// ADI radial calculation coefficients
double a_1[MAX_X+2];							// inward ADI radial calculation coefficient
double c2[MAX_Y+1],c3[MAX_Y+1],cofy[MAX_Y+1];	// ADI axial calculation coefficients
bool plate;									// do we consider plating?

// Main 

int main(void)
{

	double dt;							// time step size (per full ADI iteration)
	ofstream fp_current;				// output steam for current data
	double timepos,end_time;			// position in time and simulation end time
	double potential,epot,ede;			// potential applied to substrate, its exponential, its exponential of change
	double epot2b,epot2f;				// the exp potential relative to the copper plating (backward scan) and stripping (forward scan) potential
	double epot2;						// the cu(0)/cu(1) exp potential passed to adi_calc (either epot2b pr epot2f - determined by the direction of scan) 
	double ede2,epot_incr;				// 1/2 time step exponential change, exponential potential factor to send to adi_calc
	int kt;								// iteration counter
	double ifctr,ifctrc;				// parameters to determine if current should be calculated
	double dlctip,dlcsub;				// dimensionless tip and substrate current (wrt the steady state microelectrode current in the bulk)
	double cu0cov;						// dimensionless cu0 coverage (calculated from calc_cucoverage)
	int map_count;						// counting the number of concentration maps printed

	int oldoutperc = -1;				// value to used give fraction of simulation complete to user
	clock_t begint,markt;				// clock ticks, used to estimate finishing time
	double duration;					// total estimated duration to end of simulation in seconds
	int hours, mins;					// printout of estimated duration to end of simulation

	dt = welcome();						// greets users and determines grid size and time / potential scales

	fp_current.open("current.dat");		// open current data file
	
	fp_current.setf(ios::fixed, ios::floatfield);	// decimal notation specified
	fp_current.setf(ios::showpoint);				// decimal point always shown even for whole numbers

	fp_current << setprecision(6);		// set number of decimal places + output time
    			
	fp_current << " time \t E \t Itip \t Isub \t Cu0 cov." << endl;

	cout << "\n\n **** Running Simulation **** \n" << numxtot << endl << numytot << endl;

	// initialising simulation t=0
	pold1=cu1matrix1;							//initialise array/matrix pointers
	pold2=cu2matrix1;
	pold0=cu0array1;
	pnew1=cu1matrix2;
	pnew2=cu2matrix2;
	pnew0=cu0array2;

	clearbulk(0,numxtot+1,0,numytot);			// clear the array (requires pointers to be set)
	id_block(0,numdepth-1,0,numx);				// identify the area of microelectrode assembly

	// starting conditions
	timepos = 0.0;								// dimensionless time = 0				
	map_count = 0;								// map count set to zero
	kt = 0;										// iterations = 0
	potential = einit;							// starting potential set
	ede = exp(de);	ede2 = exp(de/2.0);			// Computation time saving 
	epot = exp(potential);						//	exponential parameters set (for potential)
	epot2b = epot/exp(e2);
	epot2f = epot2b/exp(nucl_loop);
	end_time = dt*ktmax;							
	ifctrc = end_time/500.0;
	ifctr = ifctrc;
	plate = false;								// we don't consider plating initially

	// we half the lenth of time step for double step ADI
	dt = dt/2;
	
	adi_init(dt);								// calculate ADI parameters that depend upon the size of the grid/time step size

	// time loop
	begint = clock();
	while (timepos < end_time)
	{
		timepos = timepos + (2.0*dt);			// alter time position	
		kt = kt++;								// new iteration

		// printout estimated time to finish. only works with equally spaced time steps!
		if (kt == 100) 
		{
			markt = clock();
			duration = (end_time/timepos)*((double)(markt - begint)/CLOCKS_PER_SEC);
			hours = (int)(duration/3600.0); duration = duration - ((double)hours*3600.0);
			mins = (int)(duration/60.0); duration = duration - ((double)mins*60.0);
			cout << endl << "Time To Finish: " << hours << " hrs " << mins << " mins " 
				<< duration << " secs " << endl << flush;
		}

		// which direction are we scanning in?
		
		if ((((kt-1)/scanit)%2)==0)                 // 1 = forwards (+ve) 0 = backwards (-ve)
		{
			// backward scan
			potential = potential - de;				// sort out potential			
			epot = epot / ede;						// calculate potential relative to cuI/cuII
			epot2b = epot2b / ede;					// calculate potential relative to plating
			epot2f = epot2f / ede;					// calculate potential relative to stripping
			epot2 = epot2b;							// consider plating in the backward scan
			epot_incr = (1.0 / ede2);				// work out half step change in potential
			if ((!plate)&&(potential<=e2))
			{ 
				plate = true; cout << "PLATING********";
			}
		}
		else 
		{
			potential = potential + de;				// sort out potential	
			epot = epot * ede;						// calculate potential relative to cuI/cuII
			epot2b = epot2b * ede;					// calculate potential relative to plating
			epot2f = epot2f * ede;					// calculate potential relative to stripping
			epot2 = epot2f;							// consider stripping in the forward scan
			epot_incr = (ede2);						// work out half step change in potential
		}
		// cout << " " << kt << " " << potential << " ";
		adi_calc(epot,epot2,epot_incr,dt);			

		// Saves out concentration profiles if required
		if (timepos>=mapping[map_count])
		{
			map_the_conc(0,numxtot+1,0,numytot,timepos,map_count,0);
			map_count += 1;
		}

		// Calculates currents and saves them to file if required
		if (timepos>=ifctr)
		{
			dlctip = calc_tipcurrent_p();
			// dlctip = calc_tipcurrent_z();
			dlcsub = calc_subcurrent(dt);
			cu0cov = calc_cucoverage();
			if (plate)
				fp_current << timepos << "\t" << potential << "\t" << dlctip << "\t" << dlcsub << "\t" << cu0cov
					<< "\t" << epot << "\t" << epot2f << endl;
			else
				fp_current << timepos << "\t" << potential << "\t" << dlctip << "\t" << dlcsub << "\t" << cu0cov << endl;

			ifctr=ifctr+ifctrc;
		}
	} // time loop

	// prints out final concentration map
	map_the_conc(0,numxtot+1,0,numytot,timepos,map_count,1);

	cout << "\n\n******** SIMULATION END *********\n\n";

	fp_current.close();

	return 0;
} // main

// ****************

double welcome(void)
{
	double ts;								// returning time step size 
	double approxefinal,approxeinit;		// approximate finishing and starting dimensionless potentials
	double eiglass,eibulk,ejdepth;			// effective lengths for simulation boundary edges
	int temp1,temp2;						// temp integers to help in calculations
	double topz;							// real space tip / substrate gap

	cout << "\n\n\nADI Copper System CV Simulation\n"
		 << "version : " << *version << "\n"; 
	cout << "microelectrode status : " << *mdisc_stat << "\n\n";

	cout << "This simulation looks at the microelectrode reponse\n"
		 << "as the potential of a nearby substrate is cycled\n\n";

startagain:

	cout << "Please input these SIMULATION parameters:\n\n"
		 << "eidisc  : "; cin >> eidisc; 
	cout << "eiglass : "; cin >> eiglass;
	cout << "ejgsd   : "; cin >> ejgsd;
	cout << "eibulk  : "; cin >> eibulk;
	cout << "ejdepth : "; cin >> ejdepth;
	cout << "\nnumd   : "; cin >> numd;
	cout << "expansion coefficient A : "; cin >> var_a;

	dx = log(1.0 + var_a)/(double)numd;
	dy = dx;

	cout << "\nPlease input these DIMENSIONLESS experimental parameters:\n\n"
		 << "D_Cu(I)/D_Cu(II) : "; cin >> delta; 
	cout << "init Cu(II) conc : "; cin >> cob2;
	cout << "init Cu(I) conc  : "; cin >> cob1;
	cout << "approx E final (-ve)   : "; cin >> approxefinal;
	cout << "approx E initial (+ve) : "; cin >> approxeinit;
	cout << "scanrate         : "; cin >> sr;

	cout << "\nPlease enter a value for 1/dt : ";  cin >> ts;
	cout << "\n\n";

	de = sr * (1.0/(double)ts);		// calculate step in potential from dt

	if ((approxefinal>0) || (approxeinit<0) || (numd<2) || (sr==0)) { goto startagain; }

	// assuming we want a potential step at E=0, calculate potential range parameters
	temp1 = ((int)(approxeinit/de));
	temp2 = ((int)(-1.0*approxefinal/de));
	scanit = temp1+temp2;
	einit = de*(double)temp1;
	efinal = -1.0*de*(double)temp2;

	cout << "Calculated Einit : " << einit << " Efinal : " << efinal << " de : " << de;
	cout << "\nIterations for 1 complete CV scan : " << 2*scanit;
	cout << "\nIterations in 1 diffusion time    : " << ts;
	cout << "\nPlease input total iterations to be simulated : "; cin >> ktmax;

	// now we know the time-potential-iteration relationship, 
	// the grid size can be calculated
	// grid given in lab book2 p 88

	dr_m1 = (pow((1.0+var_a),1.0/(double)numd)-1.0)/var_a;
	dz_b1 = dr_m1;
	topz=ejgsd/eidisc;
	numg = (int)(log(1.0+(var_a*((eiglass/eidisc)-1.0)))/dx);
	numbulk = (int)(log(1.0+(var_a*((eibulk/eidisc)-1.0)))/dx);
	numdepth = (int)(log(1.0+(var_a*(((ejdepth-ejgsd)/eidisc))))/dx);
	numx = numg + numd;
	numxtot = numd + numbulk;

	numy1 = (int)(log(1.0+(var_a*(ejgsd/(2.0*eidisc))))/dy);
	depth = ((exp(dy*numdepth))-1.0)/var_a;
	midz = ((exp(numy1*dy))-1.0)/var_a;

	var_b = ((exp(dy*numy1))-1.0)/(topz-midz);  
	numy=numy1*2;
	numytot = numdepth + numy;
	dz_m1 = midz - ((exp(dy*(numy1-1))-1.0)/var_a);
	dz_m2 = topz - ((exp(dy*(numy1-1))-1.0)/var_b) - midz;

	cout << "\nChosen Parameters\n";
	cout << "numd:"<<numd<<" numg:"<<numg<<" numxtot:"<<numxtot;
	cout << "\nnumy(gap):"<<numy<<" numytot:"<<numytot<<" var_a:"<<var_a<<" var_b:"<<var_b;
	cout << "\nlargest lamda value :"<<(1.0/(dr_m1*dr_m1*(double)ts));
	cout << "\nDATA INPUT OKAY\n\n";

	return (1.0/(double)ts);	// return initial timestep size

} // welcome

// ****************

int ferrors(int e_code)
{
	// turns error code into user message

	cout << "\n\nFATAL ERROR:";

	switch(e_code)
	{
	case 1: cout << "\nError in creating Current Output File\n\n";
	default: cout << "\nAn unefined error has happened in the simulation\n\n";
	}

	return 0;

} // ferrors

// ****************

void clearbulk(int a, int b, int c, int d)
{
	int i,j;

	for (i = a; i <= b; i++)
    {
    for (j = c; j <= d; j++)
         {
         pold1[i][j] = cob1;
         pold2[i][j] = cob2;
         }
	pold0[i] = 0.0;
    }

return;     
} // clearbulk 

// ****************

void id_block(int a,int b,int c,int d)
{
int i,j;

for (j = a; j<=b; j++)
     {
     for (i = c; i<=d; i++)
            {
            pnew1[i][j] = idcode;
			pnew2[i][j] = idcode;
			pold1[i][j] = idcode;
			pold2[i][j] = idcode;
            }
     }
} // id_block            

// ****************

void adi_init(double dt)
{	
	// calculation of time independent ADI parameters

	int i,j;										// Integer counters
	double lamx,lamy,maxlam;						// General calculation		
	double x2k,xk,y2k,yk,zsumdiv2;					//	variables only
	double r1,r2,z1;								//	applicable to 
	double var_a1,var_a2,var_a3,expc1,expa1,expa2;	//	adi_init

	// constants used to simplify later calculations

	lamx = dt / (dx * dx);
	lamy = lamx;
	maxlam = dt / (dr_m1 * dr_m1);

	var_a1 = (var_a * var_a) / ((1.0 + var_a) * (1.0 + var_a));
	var_a2 = var_a * var_a * (1.0 + var_a) * (1.0 + var_a);
	var_a3 = (var_a + 1.0) * (var_a - 1.0);

	// Radial Loop Calculations

	posr[0] = 0.0;
	expa1 = 1.0;
	expc1 = exp(dx);

	for (i = 1 ; i <= numd-1 ; i++)				// electrode 
		{
		expa1 = expa1 * expc1;
		r1 = (var_a1 * expa1) / (1.0 - (1.0 / expa1));
		r2 = var_a1 * (expa1 * expa1);
		x2k = r2;
		xk = r1 + r2;
		cofx[i] = ((-1.0*lamx*x2k) + (xk*lamx*dx/2.0));
		a2[i] = (1.0 + (x2k*2.0*lamx))/cofx[i];
		a3[i] = -1.0*((x2k*lamx) + (xk*lamx*dx/2.0))/cofx[i]; 
		posr[i] = ((1.0 + var_a)*(1.0 - (1.0 / expa1))) / var_a;
		}          

	i = numd;									// electrode - glass boundary
	expa1 = expa1 * expc1;
	cofx[i] = ((-1.0*maxlam) + (maxlam*dr_m1/2.0));
	a2[i] = (1.0 + (2.0*maxlam))/cofx[i];
	a3[i] = -1.0*((maxlam) + (maxlam*dr_m1/2.0))/cofx[i]; 
	posr[numd] = 1.0;

	for (i = numd+1 ; i <= numxtot+1 ; i++)		// glass and bulk boundary
		{
		expa1 = expa1 * expc1;
		r1 = (var_a2 / expa1) / (expa1 + var_a3);
		r2 = var_a2 / (expa1 * expa1);
		x2k = r2;
		xk = r1 - r2;
		cofx[i] = ((-1.0*lamx*x2k) + (xk*lamx*dx/2.0));
		a2[i] = (1.0 + (x2k*2.0*lamx))/cofx[i];
		a3[i] = -1.0*((x2k*lamx) + (xk*lamx*dx/2.0))/cofx[i]; 
		posr[i] = (expa1 + var_a3) / (var_a * (var_a + 1.0));
		}    
    
	// now calculate inward parameter 
	a_1[numxtot] = a2[numxtot] + a3[numxtot];	// if there is an a3 term here then there is a no flux condition at the radial bulk axis
 
	for ( i = numxtot - 1 ; i >= 1 ; i--)
		{
		a_1[i] = a2[i] - ( a3[i] / a_1[i+1] );
		}

	// Axial Loop Calculations
	posz[0] = 0.0;
	expa1 = exp(numdepth*dy);
	expc1 = exp(-1.0*dy);

	expa2 = exp(numy * dy);

	for (j = 1 ; j <= numdepth-1 ; j++)					// behind electrode assembly
		{
		expa1 = expa1 * expc1;
		z1 = var_a * var_a / (expa1 * expa1);
		y2k = z1;
		yk = z1;
		cofy[j] = ((-1.0*y2k*lamy)+(yk*lamy*dy/2.0));
		c2[j] = (1.0 + (2.0*y2k*lamy)) / cofy[j];
		c3[j] = -1.0*((y2k*lamy)+(yk*lamy*dy/2.0)) / cofy[j];
		posz[j] = depth + ((1.0 - expa1) / var_a);     
		}

	j = numdepth;										// grid join flush with face
	expa1 = expa1 * expc1;
	cofy[j] = -1.0 * dt / (dz_b1 * dz_b1);
	c2[j] = (1.0 + (2.0*dt/(dz_b1*dz_b1)))/cofy[j];
	c3[j] = 1.0;
	posz[j] = depth;

	for (j = (numdepth+1) ; j <= (numdepth+numy1-1) ; j++)	// lower tip - substrate gap
		{
		expa1 = expa1 * expc1;
		z1 = var_a * var_a * (expa1 * expa1);
		y2k = z1;
		yk = -1.0 * z1;
		cofy[j] = ((-1.0*y2k*lamy)+(yk*lamy*dy/2.0));
		c2[j] = (1.0 + (2.0*y2k*lamy)) / cofy[j];
		c3[j] = -1.0*((y2k*lamy)+(yk*lamy*dy/2.0)) / cofy[j];
		posz[j] = depth + (((1.0/expa1) - 1.0) / var_a);     
		}

	j = numy1 + numdepth;									// mid gap grid join	
	expa1 = expa1 * expc1;
	zsumdiv2 = (dz_m1 + dz_m2)/2.0;
	cofy[j] = -1.0 * dt / (dz_m1 * zsumdiv2);
	c2[j] = (1.0 + (dt/(dz_m2*zsumdiv2)) + (dt/(dz_m1*zsumdiv2)))/cofy[j];
	c3[j] = (-1.0 * dt / (dz_m2 * zsumdiv2))/cofy[j];
	posz[j] = midz + depth;

	expa1 = expa1 * expa2;
	for (j = (numy1+1+numdepth) ; j <= (numy+numdepth) ; j++)	// upper tip - substrate gap
		{
		expa1 = expa1 * expc1;
		z1 = var_b * var_b / (expa1 * expa1);
		y2k = z1;
		yk = z1;
		cofy[j] = ((-1.0*y2k*lamy)+(yk*lamy*dy/2.0));
		c2[j] = (1.0 + (2.0*y2k*lamy)) / cofy[j];
		c3[j] = -1.0*((y2k*lamy)+(yk*lamy*dy/2.0))/cofy[j];
		posz[j] = (depth + ((double)ejgsd/(double)eidisc)) + ((1.0 - expa1) / var_b);
		}

// note: because the upper boundary condition depends on the potential
// being applied to the substrate, it is better to calculate all downward 
// (towards electrode) ADI parameters in the main ADI sub-routine (adi_calc). 

return;

} // adi_init

// ****************

void adi_calc(double e_esub, double e_esub2, double e_eincr, double d_t)
{
	int i,j,int_limit;							// interger counters

	double cu1_b4[MAX_X+2], cu1_b_1[MAX_X+2];	// copper I ADI parameters (radial implicit)
	double cu2_b4[MAX_X+2], cu2_b_1[MAX_X+2];	// copper II ADI parameters (radial implicit)

	double gamma2[MAX_Y+1], gamma3[MAX_Y+1], delta4[MAX_Y+1]; // copper I/II ADI parameters (axial implicit)
	double alpha2[MAX_Y+1], alpha3[MAX_Y+1], betab4[MAX_Y+1]; // copper I/II ADI parameters (axial implicit)
	double cu1_d4[MAX_Y+1], cu2_d4[MAX_Y+1];	// copper I/II ADI parameters (axial implicit)

	double gamma2plate[MAX_Y+1], alpha2plate[MAX_Y+1]; // plating axial implicit bits

	double denom1[MAX_Y+1],td1,td2;				// temporary ADI calculation variables

	double dz_top;								// gap between j = numytot-1 and j = numytot (on surface of substrate)

	bool pointplate, runout;		// are we plating / do we run out of copper on single i point?

	// ********************************************
	// Implicit calculation in the RADIAL direction
	// ********************************************

	dz_top = posz[numytot]-posz[numytot-1];

	// Initialise pointers to arrays
	pold1=cu1matrix1;						
	pold2=cu2matrix1;
	pold0=cu0array1;
	pnew1=cu1matrix2;
	pnew2=cu2matrix2;
	pnew0=cu0array2;

	// set bulk boundary values (not really neccessary)
	// perp(numxtot+1,0,numytot); 
	parral(0,numx+1,numxtot);

//	cout << " e_pot 1-2 : " << e_esub << " e_pot 1-0 : " << e_esub2 << "\n";

	// loop upwards
	for (j = 1; j<=(numy+numdepth-1); j++)
	{	
		// outward (towards bulk) sweep to calculate explicit terms 
		if ( j <= numdepth ) { int_limit = numx+2; } else { int_limit = 1; }         
		for (i = int_limit; i<=numxtot; i++)
		{
			cu1_b4[i] = ((-1.0*pold1[i][j-1]*cofy[j]) + 
				(((-1.0*c2[j]*cofy[j])+2.0)*pold1[i][j]) +
				(-1.0*c3[j]*cofy[j]*pold1[i][j+1]))/cofx[i];
			cu2_b4[i] = ((-1.0*pold2[i][j-1]*cofy[j]) + 
				(((-1.0*c2[j]*cofy[j])+2.0)*pold2[i][j]) +
				(-1.0*c3[j]*cofy[j]*pold2[i][j+1]))/cofx[i];
		}
		
		// inward (towards symmetry axis) sweep to consider implicit bulk BC
		cu1_b_1[numxtot] = cu1_b4[numxtot]; // - (a3[numxtot]*pold1[numxtot+1][j]);
		cu2_b_1[numxtot] = cu2_b4[numxtot]; // - (a3[numxtot]*pold2[numxtot+1][j]);
		for ( i = numxtot-1 ; i>=int_limit; i-- )
		{
			cu1_b_1[i] = cu1_b4[i] - (a3[i]*cu1_b_1[i+1]/a_1[i+1]);
			cu2_b_1[i] = cu2_b4[i] - (a3[i]*cu2_b_1[i+1]/a_1[i+1]);
		}

		// final outward loop to calculate new concentrations
		pnew1[int_limit-1][j] = cu1_b_1[int_limit]/(1.0+a_1[int_limit]);
		pnew2[int_limit-1][j] = cu2_b_1[int_limit]/(1.0+a_1[int_limit]);
		for (i=int_limit ; i<=numxtot ; i++)
		{
			pnew1[i][j] = (cu1_b_1[i] - pnew1[i-1][j])/a_1[i];
			pnew2[i][j] = (cu2_b_1[i] - pnew2[i-1][j])/a_1[i];
		}
	}

	// set remaining boundarys
	electrode_p(numdepth,0,numd);		// electrode
	j = numdepth;						// glass
	for (i=numd+1; i<=numx ; i++) 
	{
		pnew1[i][j] = pnew1[i][j+1];
		pnew2[i][j] = pnew2[i][j+1];
	}
	j = numy+numdepth;					// substrate

	for (i=0; i<=numxtot ; i++) 
	{
		if (plate)
		{
			pnew1[i][j] = e_esub2;
			pnew2[i][j] = e_esub*e_esub2;
			pnew0[i] = pold0[i] + ((d_t/(dz_top*eta))*(pnew2[i][j-1]-pnew2[i][j]+(delta*(pnew1[i][j-1]-pnew1[i][j]))));
			if (pnew0[i]<=0.0) 
			{ 
				pnew0[i]=0.0;
				pnew2[i][j] = ((delta*pnew1[i][j-1]) + pnew2[i][j-1]) / (1.0 + (delta/e_esub));
				pnew1[i][j] = pnew1[i][j-1] + 
					((1/delta)*(((dz_top*eta/d_t)*pold0[i])+pnew2[i][j-1]-pnew2[i][j]));
			}
		}
		if (!plate)
		{
			pnew1[i][j] = ((delta*pnew1[i][j-1]) + pnew2[i][j-1]) / (delta + e_esub);
			pnew2[i][j] = ((delta*pnew1[i][j-1]) + pnew2[i][j-1]) / (1.0 + (delta/e_esub));
			pnew0[i] = 0.0;
		}
	}
	if (pnew1[numxtot][numy+numdepth]==0.0) { plate = false; }

	i= numxtot+1;
	for (j=0; j<=numytot; j++)			// zero flux at radial limit
	{
		pnew1[i][j] = pnew1[i-1][j];
		pnew2[i][j] = pnew2[i-1][j];
	}
	pnew0[numxtot+1] = pnew0[numxtot];

	// *******************************************
	// Implicit calculation in the AXIAL direction
	// *******************************************

	// Initialise pointers to arrays
	pold1=cu1matrix2;						
	pold2=cu2matrix2;
	pold0=cu0array2;
	pnew1=cu1matrix1;
	pnew2=cu2matrix1;
	pnew0=cu0array1;

	e_esub = e_esub * e_eincr;			// change potentials for half time step
	e_esub2 = e_esub2 * e_eincr;

	// plate params     
	gamma2plate[numytot-1] = c2[numytot-1];
	alpha2plate[numytot-1] = c2[numytot-1];
	
	// normal params
	gamma2[numytot-1] = c2[numytot-1]+(c3[numytot-1]/(1.0 + (delta/e_esub)));
	gamma3[numytot-1] = (delta*c3[numytot-1])/(1.0 + (delta/e_esub));
	alpha2[numytot-1] = c2[numytot-1]+((delta*c3[numytot-1])/(e_esub + delta));
	alpha3[numytot-1] = (c3[numytot-1]/(e_esub + delta));

	// loop to calculate all params.
	for (j=numytot-2 ; j>=1 ; j--)
	{
		alpha2plate[j] = c2[j] - (c3[j]/alpha2plate[j+1]); 
		gamma2plate[j] = c2[j] - (c3[j]/gamma2plate[j+1]); 

		denom1[j] = (alpha2[j+1]*gamma2[j+1]) - (alpha3[j+1]*gamma3[j+1]);
		alpha2[j] = c2[j] - ((gamma2[j+1]*c3[j])/denom1[j]); 
		alpha3[j] = ((alpha3[j+1]*c3[j])/denom1[j]);
		gamma2[j] = c2[j] - ((alpha2[j+1]*c3[j])/denom1[j]);
		gamma3[j] = ((gamma3[j+1]*c3[j])/denom1[j]);
	}

	// Now start concentration changing steps 

	// Boundary values - not really necessary
	parral(0,numx+1,numxtot);  

	// Sideways sweep (away from symmetry axis)
	for (i=1; i<=numxtot; i++)
	{
		pointplate = plate;
		runout = false;

		// Are we above microelectrode assembly or bulk solution?
		if ( i <= numx+1 ) { int_limit = numdepth+1; } else { int_limit = 1; } 
	
		// Calculate upwardly the explicit terms
		for (j=int_limit; j<=numytot-1; j++)
		{
			cu1_d4[j] = ((-1.0*cofx[i]*pold1[i-1][j]) +
						(((-1.0*a2[i]*cofx[i])+2.0)*pold1[i][j]) +
						(-1.0*a3[i]*cofx[i]*pold1[i+1][j]))/cofy[j];
			cu2_d4[j] = ((-1.0*cofx[i]*pold2[i-1][j]) +
						(((-1.0*a2[i]*cofx[i])+2.0)*pold2[i][j]) +
						(-1.0*a3[i]*cofx[i]*pold2[i+1][j]))/cofy[j];
		}

tryagain:	// we may have to come back here if we've run out of Cu(0)

		// Calculate downward considering substrate boundary condition
		if (pointplate)
		{
			delta4[numytot-1] = cu2_d4[numytot-1] - (c3[numytot-1]*e_esub2*e_esub);
			betab4[numytot-1] = cu1_d4[numytot-1] - (c3[numytot-1]*e_esub2);
		}
		else 
		{
			if (runout)
			{
				delta4[numytot-1] = cu2_d4[numytot-1] 
					- ((c3[numytot-1] * dz_top*eta*pold0[i]) / (d_t * (1.0+(delta/e_esub))));
				betab4[numytot-1] = cu1_d4[numytot-1]
					- ((c3[numytot-1] * dz_top*eta*pold0[i]) / (d_t * (delta + e_esub)));
			}
			else
			{
				delta4[numytot-1] = cu2_d4[numytot-1];
				betab4[numytot-1] = cu1_d4[numytot-1];
			}
		}
	
		if (pointplate)
		{
			for (j=numytot-2 ; j>=int_limit ; j--) 
			{
				betab4[j] = cu1_d4[j] - (c3[j]*betab4[j+1]/alpha2plate[j+1]);
				delta4[j] = cu2_d4[j] - (c3[j]*delta4[j+1]/gamma2plate[j+1]);
			}
		}
		else
		{
			for (j=numytot-2 ; j>=int_limit ; j--) 
			{
				betab4[j] = cu1_d4[j] +((c3[j]*((alpha3[j+1]*delta4[j+1])-(betab4[j+1]*gamma2[j+1])))/denom1[j]);
				delta4[j] = cu2_d4[j] +((c3[j]*((gamma3[j+1]*betab4[j+1])-(delta4[j+1]*alpha2[j+1])))/denom1[j]);
			}
		}

		// Lowest 'j' concentration depends upon boundary type

		// bulk solution already taken care of by 'parral'

		if (pointplate)
		{
			if ((i > numd)&&(i <= numx+1))		// over glass
			{
				pnew2[i][int_limit-1] = (delta4[int_limit]/(1.0 + gamma2plate[int_limit])); 
				pnew1[i][int_limit-1] = (betab4[int_limit]/(1.0 + alpha2plate[int_limit])); 
			}
			// Insert relevent boundary condition for microelectrode voltage and compile		
			else if (i <= numd)						// on top of electrode +0.5V
			{
				pnew1[i][int_limit-1] = 0.0;
				pnew2[i][int_limit-1] = ((betab4[int_limit]*delta)+(delta4[int_limit]))/(gamma2plate[int_limit] - 1.0);
			}

//			else if (i <= numd)						// on top of electrode +0.0V
//			{
//				pnew2[i][int_limit-1] = 0.0;
//				pnew1[i][int_limit-1] = (betab4[int_limit]+(delta4[int_limit]/delta))/(1.0 + gamma2[int_limit]);
//			}
		}
		else
		{
			if ((i > numd)&&(i <= numx+1))		// over glass
			{
				pnew2[i][int_limit-1] = (delta4[int_limit]-(gamma3[int_limit]*betab4[int_limit]/(1.0+alpha2[int_limit])))/(1.0+gamma2[int_limit]-(gamma3[int_limit]*alpha3[int_limit]/(1.0+alpha2[int_limit])));
				pnew1[i][int_limit-1] = (betab4[int_limit]-(alpha3[int_limit]*pnew2[i][int_limit-1]))/(1.0+alpha2[int_limit]); 
			}

			// Insert relevent boundary condition for microelectrode voltage and compile
			else if (i <= numd)						// on top of electrode +0.5V
			{
				pnew1[i][int_limit-1] = 0.0;
				td1 = gamma3[int_limit]*betab4[int_limit]/alpha2[int_limit];
				td2 = ((gamma2[int_limit]*alpha2[int_limit])-(alpha3[int_limit]*gamma3[int_limit]))/(alpha2[int_limit]-(delta*alpha3[int_limit]));
				pnew2[i][int_limit-1] = (delta4[int_limit]-td1+(td2*delta*betab4[int_limit]/alpha2[int_limit]))/(1.0+td2);
			}

//			else if (i <= numd)						// on top of electrode +0.0V
//			{
//				pnew2[i][int_limit-1] = 0.0;
//				td1 = gamma3[int_limit]/alpha2[int_limit];
//				td2 = ((gamma2[int_limit]*alpha2[int_limit])-(alpha3[int_limit]*gamma3[int_limit]))/(alpha2[int_limit]-(delta*alpha3[int_limit]));
//				pnew1[i][int_limit-1] = (delta4[int_limit]-(td1*betab4[int_limit])+(td2*delta*betab4[int_limit]/alpha2[int_limit]))/(td1+((delta*td2*(alpha2[int_limit]+1.0))/alpha2[int_limit]));
//			}
		}

		if (pointplate)
		{
			// Upward loop (toward substrate) calculate the concentrations		  
			for (j=int_limit;j<=numytot-1;j++) 
			{
				pnew2[i][j] = (delta4[j] - pnew2[i][j-1])/gamma2plate[j];
				pnew1[i][j] = (betab4[j] - pnew1[i][j-1])/alpha2plate[j];
			}
		}
		else
		{
			// Upward loop (toward substrate) calculate the concentrations		  
			for (j=int_limit;j<=numytot-1;j++) 
			{
				pnew2[i][j] = (delta4[j]+(gamma3[j]*(pnew1[i][j-1]-betab4[j])/alpha2[j])-pnew2[i][j-1])/(gamma2[j]-(gamma3[j]*alpha3[j]/alpha2[j]));
				pnew1[i][j] = (betab4[j]+(alpha3[j]*(pnew2[i][j-1]-delta4[j])/gamma2[j])-pnew1[i][j-1])/(alpha2[j]-(gamma3[j]*alpha3[j]/gamma2[j]));
			}
		}

		// added afterwards
		j = numytot;
		if (pointplate)
		{
			pnew1[i][j] = e_esub2;
			pnew2[i][j] = e_esub * e_esub2;
			pnew0[i] = pold0[i] + ((d_t/(dz_top*eta))*(pnew2[i][j-1]-pnew2[i][j]+(delta*(pnew1[i][j-1]-pnew1[i][j]))));
			if (pnew0[i]<=0.0) 
			{ 
				pointplate = false; 
				runout = true;
				goto tryagain;
			}
		}
		else
		{
			pnew1[i][j] = ((delta*pnew1[i][j-1]) + pnew2[i][j-1]) / (delta + e_esub);
			pnew2[i][j] = e_esub * pnew1[i][j];
			pnew0[i] = 0.0;	
		}

	} // i loop

	// Symmetry axis
	for (j=numdepth; j<=numytot ; j++) // or numytot-1
	{
		pnew1[0][j] = pnew1[1][j];
		pnew2[0][j] = pnew2[1][j];
		pnew0[0] = pnew0[1];

	}
	// Side of glass
	for (j = 1; j <= numdepth-1 ; j++) 
	{ 
		pnew1[numx+1][j] = pnew1[numx+2][j];
		pnew2[numx+1][j] = pnew2[numx+2][j];
	}

	if (pnew0[numxtot]==0.0) { plate = false; }

	// Zero flux at radial limit
	i = numxtot+1;
	for (j=0; j<=numytot; j++)			
	{
		pnew1[i][j] = pnew1[i-1][j];
		pnew2[i][j] = pnew2[i-1][j];
	}
  	pnew0[numxtot+1] = pnew0[numxtot];

	return;

} // adi_calc

// ****************

inline void perp(int a,int b,int c)
{
	int i,j;		// integer counters

	i = a;
	for (j = b; j<=c; j++)
		{
		pnew1[i][j] = cob1;
		pnew2[i][j] = cob2;
		}
	return;

} // perp

// ****************

inline void parral(int a,int b,int c)
{
	int i,j;		// integer counters

	j = a;
	for (i = b; i<=c; i++)
		{
		pnew1[i][j] = cob1;
		pnew2[i][j] = cob2;
		}
	return;

} // parral

// ****************

inline void electrode_p(int a,int b,int c)
{
	int i,j;		// integer counters

	j = a;
	for (i = b; i<=c; i++)
		{
		pnew1[i][j] = 0.0;
		pnew2[i][j] = pnew2[i][j+1] + (delta*pnew1[i][j+1]); 
		}
	return;

} // electrode_p

// ****************

inline void electrode_z(int a,int b,int c)
{
	int i,j;		// integer counters

	j = a;
	for (i = b; i<=c; i++)
		{
		pnew1[i][j] = pnew1[i][j+1] + (pnew2[i][j+1]/delta);
		pnew2[i][j] = 0.0; 
		}
	return;

} // electrode_z

// ****************

inline double calc_tipcurrent_z(void)
{
	
	double nsum,dlctip;		// summing variable and calculated substrate current
	int i;					// integer counter

	nsum = 0.0;
	for (i = 1; i<=numd ; i++) { nsum += posr[i]*(posr[i]-posr[i-1])*(pnew2[i][numdepth+1]-pnew2[i][numdepth]); }
	dlctip = M_PI * nsum / (2.0 * dz_b1);

	return dlctip;

} // calc_tipcurrent

// ****************

inline double calc_subcurrent(double d_t)
{

	double nsum,isub;		// summing variable and calculated substrate current
	int i;					// integer counter

	nsum = 0.0;				// initialise variables

	for (i = 1; i<=numxtot+1 ; i++) 
	{
		// copper II reduction/oxidation + formation/stripping of Cu0 gives full wave
		nsum -= posr[i]*(posr[i]-posr[i-1])*((1.0*(pnew2[i][numytot-1]-pnew2[i][numytot]))+(((posz[numytot]-posz[numytot-1])/d_t)*(pnew0[i]-pold0[i]))); 
	}

	isub = M_PI * nsum / (2.0 * (posz[numytot]-posz[numytot-1]));

	return isub;
	
} // calc_subcurrent

// ****************

void map_the_conc(int bl, int br, int lb, int lt, double timepos, int timecount, bool test)
{

	ofstream outMAP;					// output file stream

	int i,j,conclimitx,conclimity;		// integer counters and size limiters (for printing lareg maps)
	char concfile[] ={"xmap.smap"};		// the filename mask (x is overwritten)

	concfile[0]=(char)(timecount+65);	// number filenames Amap,Bmap,Cmap etc

	// if its the final map use a special filename
	if (test==1) { strcpy(concfile,"FILE.smap"); }

	// open datafile, setup floating point output preferences and stamp it with the time
	outMAP.open(concfile);

	outMAP.setf(ios::fixed, ios::floatfield);			// decimal notation specified
	outMAP.setf(ios::showpoint);						// decimal point always shown even for whole numbers

	outMAP << setprecision(8) << timepos << "\n";		// set number of decimal places + output time
    
	// limit printout size to 100x100
	conclimitx=numxtot/100;
	conclimity=numytot/100;
	if (conclimitx==0) {conclimitx=1;}
	if (conclimity==0) {conclimity=1;}

	j=lb;

	// loop through and print concentrations of cuI and cuII
	for (; j<=lt;)
    {
		i = bl;
		for (; i<=(br);)
		{
			outMAP<<posr[i]<<"\t"<<posz[j]<<"\t"<<pnew1[i][j]<<"\t"<<pnew2[i][j]<<endl;
			i+=conclimitx;
		}
    j+=conclimity;
    }

	// loop through and print concentrations of cu0
	i = bl;
	for (; i<=(br);)
	{
		outMAP<<posr[i]<<"\t"<<pnew0[i]<<endl;
		i+=conclimitx;
	}

	// close mapping data file
	outMAP.close();

return;
}

inline double calc_cucoverage(void)
{

	double nsum,coverage;		// summing variable and calculated substrate coverage
	int i;						// integer counter

	nsum = 0.0;

	for (i = 1; i<=numxtot+1 ; i++) 
	{
		nsum += posr[i]*(posr[i]-posr[i-1])*pnew0[i]; 
	}

	coverage = 2.0 * M_PI * nsum ;

	return coverage;

}

inline double calc_tipcurrent_p(void)
{
	
	double nsum,dlctip;		// summing variable and calculated substrate current
	int i;					// integer counter

	nsum = 0.0;
	for (i = 1; i<=numd ; i++) { nsum += posr[i]*(posr[i]-posr[i-1])*delta*(pnew1[i][numdepth+1]-pnew1[i][numdepth]); }
	dlctip = M_PI * nsum / (2.0 * dz_b1);

	return dlctip;

} // calc_tipcurrent


