


// wingluox.cpp - A Microelectrode Transient Simulation
//              
//				A microelectrode response at a constant potential 
//              is simulated close to a substrate with immobilised
//              glucose oxidase.        
// 
//				Simulation for MS WINDOWS 32 bit
//
//				Grid Shape:
//
//				/////////////SUBSTRATE(CV)///////////////
//				////ENZ/////ENZ///////ENZ///////ENZ//////
//				|			     						|
//				S										|
//				Y										|
//				M										|
//				M										|
//				|										|
//				A		species O and R					|
//				X										B	
//				I										|
//				S										U
//				|										|
//				|___________...............				L
//				|			...............				|
//				| electrode	... glass .....				K
//				|			...............				|
//		   ^	|	ox   	...............				|
//		Z  |	| potential	...............				|
//				|			...............				|
//				|			...............				|
//				|			...............				|
//              BULK
//			
//					R ->
//
//				This program implements ADI in two dimensions using
//				fully implicit boundary conditions and expanding
//				space and time grids
//				
//				This simulation assumes that substrate concentration (glucose)
//              is >> Km. The standard Michaelis Mentin reaction scheme for
//              mediated enzyme applies. Mediator Informaion: A is oxygen, B
//              is hydrogen peroxide (oxidised back to oxygen back at electrode
//              surface)
//				S + Eox <=> P + Ered
//				Ered + O -> Eox + R
//				At the electrode surface:
//				R - e- -> O 
//
//				Version 1: Approach curves. Note similarity with standard positive
//              feedback scheme.
//
//				J.L.Amphlett

// Last Windows Update: Monday 24 April 2000
// based upon the copper/ec' simulation programs

// Includes and Definitions

#include <iostream.h>
#include <functional>
#include <math.h>
#include <string.h>
#include <fstream.h>
#include <iomanip.h>
#include <time.h>

#undef MAX_X
#undef MAX_Y
#undef M_PI
#undef idcode
#undef APPROX_1
#undef APPROX_2
#undef APPROX_3
#define MAX_X 1000
#define MAX_Y 1000
#define M_PI 3.141592654
#define idcode -1
#define APPROX_1 1
#define APPROX_2 2
#define APPROX_3 3

// Prototypes
int main(void);								// main control subrountine
double welcome(void);						// welcomes user+input initial variables+calculate grid/time/potential parameters
int ferrors(int e_code);					// turn simulation error codes into user error messages
void clearbulk(int a, int b, int c, int d);	// clear the matrix square (with bulk concentrations) defined by a,b,c,d
void id_block(int a,int b,int c,int d);     // id's the microdisc with a marker concentration (idcode)
void adi_init(double dt);					// calculation of time independent ADI parameters
void adi_calc(double d_t, int ts);			// the concentration calculation sub-routine (needs exp(substrate potential),exp(sub potential relative to cu(0)) + its half step increment + time step size)
inline void perp(int a,int b,int c);		// set vertical boundary bulk values 
inline void parral(int a, int b, int c);	// set horizontal boundary bulk values
inline void electrode(int a,int b,int c);	// set radial electrode values using exp(potential)
inline double calc_tipcurrent(void);		// returns the dimensionless tip current
inline double calc_subcurrent(double d_t);	// returns the substrate current (normalised wrt steady state microelectrode current in the bulk) - needs timestep size
inline void map_the_conc(int bl, int br, int lb, int lt, double timepos, int timecount, bool test); // prints out cocentration profile from boundaries, marks with timepos, map number and a bool to see if its the final map (1=YES)

// Global Constants & Variables
const char *version[] = {"microGluOx3 rv 1.1"};	// program version number
const double mapping[] = {0.01,0.02,0.03,1,20,25,30,35,1E10};			// array of times to print out concentration profiles IMP:last number large to avoid problems
const double eta = 1.0;						// term of uncertainty in conservation of flux equation, units sec cm^-1
 
double delta;								// ration of CU(I):Cu(II) complex diffusion coefficients
double cobO, cobR;							// initial concentrations of A and B species
double v_max, k_mo;							// dimensionless maximum rate, rate constant ratio 
double dx,dy;								// step change in virtual X and Y
double var_a, var_b;						// expansion grid coefficents
int scanit;									// iterations reuired for 1 scan (right to left)
int ktmax;									// total number of simulation iterations (equal time teps)
int numd,numg,numbulk,numx,numxtot;			// i direction grid parameters
int numy1,numy,numytot,numdepth;			// j direction grid parameters
double timeend;								// dimensionless end time

double eidisc,ejgsd;						// estimated grid size parameters used in welcome and adi_init
double dz_b1, depth, dz_m1, dz_m2, midz;	// calculated real space grid size parameters		
double dz_top, lamda_top;					// calculated real space grid size parameters		

double alpha, alpha1, alpha2, alpha3;		// calculated enzyme parameters
double beta, beta1, beta2, beta3, beta4;	// calculated enzyme parameters

double Amatrix1[MAX_X+2][MAX_Y+1];			// A concentration old and new matrices		
double Amatrix2[MAX_X+2][MAX_Y+1];
double Bmatrix1[MAX_X+2][MAX_Y+1];			// B concentration old and new matrices
double Bmatrix2[MAX_X+2][MAX_Y+1];

double (*pnewo)[MAX_Y+1];					// pointer to new A concentration matrix
double (*poldo)[MAX_Y+1];					// pointer to old A concentration matrix
double (*pnewr)[MAX_Y+1];					// pointer to new B concentration matrix
double (*poldr)[MAX_Y+1];					// pointer to old B concentration matrix

double dr_m1;								// smallest change in dr (electrode/glass boundary)
double posr[MAX_X+2],posz[MAX_Y+1];			// arrays with real position of points (adi_init)
double a2o[MAX_X+2],a3[MAX_X+2],cofx[MAX_X+2];	// ADI radial calculation coefficients
double a2r[MAX_X+2];						// ADI radial calculation coefficients
double a_1o[MAX_X+2], a_1r[MAX_X+2];		// inward ADI radial calculation coefficient
double c2o[MAX_Y+1],c3[MAX_Y+1],cofy[MAX_Y+1];	// ADI axial calculation coefficients
double c_1r[MAX_Y+1],c_2r[MAX_Y+1];			// ADI parameter (axial implicit)
double c2r[MAX_Y+1],c_1o[MAX_Y+1];			// extra ADI parameters for second species

int approx_level = 0;

// Main 

int main(void)
{
	double dt;							// time step size (per full ADI iteration)
	double dt_old,dt_ref;				// previous and original timestep's dt value
	ofstream fp_current;				// output steam for current data
	double timepos,end_time;			// position in time and simulation end time
	int kt;								// iteration counter
	double ifctr,ifctrc;				// parameters to determine if current should be calculated
	double dlctip;						// dimensionless tip and substrate current (wrt the steady state microelectrode current in the bulk)
	int map_count;						// counting the number of concentration maps printed
	int oldoutperc = -1;				// value to used give fraction of simulation complete to user
	clock_t begint,markt;				// clock ticks, used to estimate finishing time
	double duration;					// total estimated duration to end of simulation in seconds
	int hours, mins;					// printout of estimated duration to end of simulation

	dt = welcome();						// greets users and determines grid size and time / potential scales

	fp_current.open("current.dat");		// open current data file
	
	fp_current.setf(ios::fixed, ios::floatfield);	// decimal notation specified
	fp_current.setf(ios::showpoint);				// decimal point always shown even for whole numbers

	fp_current << setprecision(6);		// set number of decimal places + output time
    			
	fp_current << " time \t E \t Itip \t Isub \t Cu0 cov." << endl;

	cout << "\n\n **** Running Simulation **** \n\n";

	// initialising simulation t=0
	poldo=Amatrix1;							//initialise array/matrix pointers
	poldr=Bmatrix1;
	pnewo=Amatrix2;
	pnewr=Bmatrix2;

	clearbulk(0,numxtot+1,0,numytot);			// clear the array (requires pointers to be set)
	id_block(0,numdepth-1,0,numx);				// identify the area of microelectrode assembly

	// starting conditions
	timepos = 0.0;								// dimensionless time = 0				
	map_count = 0;								// map count set to zero
	kt = 0;										// iterations = 0
	end_time = timeend;
	ifctrc = end_time/500.0;
	ifctr = ifctrc;

	// we half the lenth of time step for double step ADI
	dt = dt/2;
	
	adi_init(dt);								// calculate ADI parameters that depend upon the size of the grid/time step size
    dt_ref = dt;								// stores initial size of dt for later usage.

	// time loop
	begint = clock();
	while (timepos < end_time)
	{
		timepos = timepos + (2.0*dt);			// alter time position	
		kt = kt++;								// new iteration

	    if (dt!=dt_old)
		{
			adi_init(dt);
		}

		// printout estimated time to finish. only works with equally spaced time steps!
		if (kt == 100) 
		{
			markt = clock();
			duration = (end_time/timepos)*((double)(markt - begint)/CLOCKS_PER_SEC);
			hours = (int)(duration/3600.0); duration = duration - ((double)hours*3600.0);
			mins = (int)(duration/60.0); duration = duration - ((double)mins*60.0);
			cout << endl << "Time To Finish if using equal steps: " << hours << " hrs " << mins << " mins " 
				<< duration << " secs " << endl << flush;
		}

		adi_calc(dt,kt);			

		// Saves out concentration profiles if required
		if (timepos>=mapping[map_count])
		{
			map_the_conc(0,numxtot+1,0,numytot,timepos,map_count,0);
			map_count += 1;
		}

		// Calculates currents and saves them to file if required
		if (timepos>=ifctr)
		{
			dlctip = calc_tipcurrent();
			fp_current << timepos << "\t" << dlctip << endl;
			if ((int)(timepos*10/end_time) != oldoutperc) { oldoutperc = (int)(timepos*10/end_time); cout << oldoutperc; cout << flush; } 
			ifctr=ifctr+ifctrc;
		}

		dt_old = dt;
      
		/*do we require a new value of dt?*/ 
		dt = dt_ref * (((int)timepos)+1);
	
	} // time loop

	// prints out final concentration map
	map_the_conc(0,numxtot+1,0,numytot,timepos,map_count,1);

	cout << "alpha: " << alpha << " beta: " << beta << " dz_top: " << dz_top << endl;
	cout << "\n\n******** SIMULATION END *********\n\n";

	fp_current.close();

	return 0;
} // main

// ****************

double welcome(void)
{
	double ts;								// returning time step size 
	double eiglass,eibulk,ejdepth;			// effective lengths for simulation boundary edges
	double topz;							// real space tip / substrate gap

	cout << "\n\n\nADI Microelectrode CV Simulation\n"
		 << "version : " << *version << "\n"; 

	cout << "This simulation looks at the transient microelectrode reponse" << endl
		 << "We consider it in proximity of" << endl
		 << "an immobilised enzyme substrate" << endl << endl;

	cout << "Please input these SIMULATION parameters:\n\n"
		 << "eidisc  : "; cin >> eidisc; 
	cout << "eiglass : "; cin >> eiglass;
	cout << "ejgsd   : "; cin >> ejgsd;
	cout << "eibulk  : "; cin >> eibulk;
	cout << "ejdepth : "; cin >> ejdepth;
	cout << "\nnumd   : "; cin >> numd;
	cout << "expansion coefficient A : "; cin >> var_a;

	dx = log(1.0 + var_a)/(double)numd;
	dy = dx;

	cout << "\nPlease input these DIMENSIONLESS experimental parameters:\n\n"
		 << "D(R)/D(O) : "; cin >> delta; 
	cout << "init O conc : "; cin >> cobO;
	cout << "Vmax	      : "; cin >> v_max;
	cout << "Kmo          : "; cin >> k_mo;

	cout << "\nPlease enter an initial value for 1/dt : ";  cin >> ts;
	cout << "Please enter dimensionless end time : ";  cin >> timeend;
	cout << "Please enter approximation level (1-3) : ";  cin >> approx_level;
	
	cout << "\n\n";

	// the grid size can now be calculated
	// grid given in lab book2 p 88

	dr_m1 = (pow((1.0+var_a),1.0/(double)numd)-1.0)/var_a;
	dz_b1 = dr_m1;
	topz=ejgsd/eidisc;
	numg = (int)(log(1.0+(var_a*((eiglass/eidisc)-1.0)))/dx);
	numbulk = (int)(log(1.0+(var_a*((eibulk/eidisc)-1.0)))/dx);
	numdepth = (int)(log(1.0+(var_a*(((ejdepth-ejgsd)/eidisc))))/dx);
	numx = numg + numd;
	numxtot = numd + numbulk;

	numy1 = (int)(log(1.0+(var_a*(ejgsd/(2.0*eidisc))))/dy);
	depth = ((exp(dy*numdepth))-1.0)/var_a;
	midz = ((exp(numy1*dy))-1.0)/var_a;

	var_b = ((exp(dy*numy1))-1.0)/(topz-midz);  
	numy=numy1*2;
	numytot = numdepth + numy;
	dz_m1 = midz - ((exp(dy*(numy1-1))-1.0)/var_a);
	dz_m2 = topz - ((exp(dy*(numy1-1))-1.0)/var_b) - midz;

	cout << "\nChosen Parameters\n";
	cout << "numd:"<<numd<<" numg:"<<numg<<" numxtot:"<<numxtot<<" numytot:"<< numytot;
	cout << "\nnumy(gap):"<<numy<<" numytot:"<<numytot<<" var_a:"<<var_a<<" var_b:"<<var_b;
	cout << "\nlargest lamda value :"<<(1.0/(dr_m1*dr_m1*(double)ts));
	cout << "\nDATA INPUT OKAY\n\n";

	return (1.0/(double)ts);	// return initial timestep size

} // welcome

// ****************

int ferrors(int e_code)
{
	// turns error code into user message

	cout << "\n\nFATAL ERROR:";

	switch(e_code)
	{
	case 1: cout << "\nError in creating Current Output File\n\n";
	default: cout << "\nAn undefined error has happened in the simulation\n\n";
	}

	return 0;

} // ferrors

// ****************

void clearbulk(int a, int b, int c, int d)
{
	int i,j;

	for (i = a; i <= b; i++)
    {
		for (j = c; j <= d; j++)
        {
			poldo[i][j] = cobO;
			poldr[i][j] = cobR;
        }
    }

return;     
} // clearbulk 

// ****************

void id_block(int a,int b,int c,int d)
{
int i,j;

for (j = a; j<=b; j++)
     {
     for (i = c; i<=d; i++)
            {
            pnewo[i][j] = idcode;
			pnewr[i][j] = idcode;
			poldo[i][j] = idcode;
			poldr[i][j] = idcode;
            }
     }
} // id_block            

// ****************

void adi_init(double dt)
{	
	// calculation of time independent ADI parameters

	int i,j;										// Integer counters
	double lamx,lamy,maxlam;						// General calculation		
	double x2k,xk,y2k,yk,zsumdiv2;					//	variables only
	double r1,r2,z1;								//	applicable to 
	double var_a1,var_a2,var_a3,expc1,expa1,expa2;	//	adi_init

	// constants used to simplify later calculations

	lamx = dt / (dx * dx);
	lamy = lamx;
	maxlam = dt / (dr_m1 * dr_m1);

	var_a1 = (var_a * var_a) / ((1.0 + var_a) * (1.0 + var_a));
	var_a2 = var_a * var_a * (1.0 + var_a) * (1.0 + var_a);
	var_a3 = (var_a + 1.0) * (var_a - 1.0);

	// Radial Loop Calculations

	posr[0] = 0.0;
	expa1 = 1.0;
	expc1 = exp(dx);

	for (i = 1 ; i <= numd-1 ; i++)				// electrode 
	{
		expa1 = expa1 * expc1;
		r1 = (var_a1 * expa1) / (1.0 - (1.0 / expa1));
		r2 = var_a1 * (expa1 * expa1);
		x2k = r2;
		xk = r1 + r2;
		cofx[i] = ((-1.0*lamx*x2k) + (xk*lamx*dx/2.0));
		a2o[i] = (1.0 + (x2k*2.0*lamx))/cofx[i];
		a3[i] = -1.0*((x2k*lamx) + (xk*lamx*dx/2.0))/cofx[i];
		posr[i] = ((1.0 + var_a)*(1.0 - (1.0 / expa1))) / var_a;
	}          

	i = numd;									// electrode - glass boundary
	expa1 = expa1 * expc1;
	cofx[i] = ((-1.0*maxlam) + (maxlam*dr_m1/2.0));
	a2o[i] = (1.0 + (2.0*maxlam))/cofx[i];
	a3[i] = -1.0*((maxlam) + (maxlam*dr_m1/2.0))/cofx[i];
	posr[numd] = 1.0;

	for (i = numd+1 ; i <= numxtot+1 ; i++)		// glass and bulk boundary
	{
		expa1 = expa1 * expc1;
		r1 = (var_a2 / expa1) / (expa1 + var_a3);
		r2 = var_a2 / (expa1 * expa1);
		x2k = r2;
		xk = r1 - r2;
		cofx[i] = ((-1.0*lamx*x2k) + (xk*lamx*dx/2.0));
		a2o[i] = (1.0 + (x2k*2.0*lamx))/cofx[i];
		a3[i] = -1.0*((x2k*lamx) + (xk*lamx*dx/2.0))/cofx[i];
		posr[i] = (expa1 + var_a3) / (var_a * (var_a + 1.0));
	}    
    
	// now calculate inward parameter 
	a_1o[numxtot] = a2o[numxtot] + a3[numxtot];	// if there is an a3 term here then there is a no flux condition at the radial bulk axis

	for ( i = numxtot - 1 ; i >= 1 ; i--)
	{
		a_1o[i] = a2o[i] - ( a3[i] / a_1o[i+1] );
	}

	// Axial Loop Calculations
	posz[0] = 0.0;
	expa1 = exp(numdepth*dy);
	expc1 = exp(-1.0*dy);
	expa2 = exp(numy * dy);

	for (j = 1 ; j <= numdepth-1 ; j++)					// behind electrode assembly
	{
		expa1 = expa1 * expc1;
		z1 = var_a * var_a / (expa1 * expa1);
		y2k = z1;
		yk = z1;
		cofy[j] = ((-1.0*y2k*lamy)+(yk*lamy*dy/2.0));
		c2o[j] = (1.0 + (2.0*y2k*lamy)) / cofy[j];
		c3[j] = -1.0*((y2k*lamy)+(yk*lamy*dy/2.0)) / cofy[j];
		posz[j] = depth + ((1.0 - expa1) / var_a);     
	}

	j = numdepth;										// grid join flush with face
	expa1 = expa1 * expc1;
	cofy[j] = -1.0 * dt / (dz_b1 * dz_b1);
	c2o[j] = (1.0 + (2.0*dt/(dz_b1*dz_b1)))/cofy[j];
	c3[j] = 1.0;
	posz[j] = depth;

	for (j = (numdepth+1) ; j <= (numdepth+numy1-1) ; j++)	// lower tip - substrate gap
	{
		expa1 = expa1 * expc1;
		z1 = var_a * var_a * (expa1 * expa1);
		y2k = z1;
		yk = -1.0 * z1;
		cofy[j] = ((-1.0*y2k*lamy)+(yk*lamy*dy/2.0));
		c2o[j] = (1.0 + (2.0*y2k*lamy)) / cofy[j];
		c3[j] = -1.0*((y2k*lamy)+(yk*lamy*dy/2.0)) / cofy[j];
		posz[j] = depth + (((1.0/expa1) - 1.0) / var_a);     
	}

	j = numy1 + numdepth;									// mid gap grid join	
	expa1 = expa1 * expc1;
	zsumdiv2 = (dz_m1 + dz_m2)/2.0;
	cofy[j] = -1.0 * dt / (dz_m1 * zsumdiv2);
	c2o[j] = (1.0 + (dt/(dz_m2*zsumdiv2)) + (dt/(dz_m1*zsumdiv2)))/cofy[j];
	c3[j] = (-1.0 * dt / (dz_m2 * zsumdiv2))/cofy[j];	
	posz[j] = midz + depth;

	expa1 = expa1 * expa2;
	for (j = (numy1+1+numdepth) ; j <= (numy+numdepth) ; j++)	// upper tip - substrate gap
	{
		expa1 = expa1 * expc1;
		z1 = var_b * var_b / (expa1 * expa1);
		y2k = z1;
		yk = z1;
		cofy[j] = ((-1.0*y2k*lamy)+(yk*lamy*dy/2.0));
		c2o[j] = (1.0 + (2.0*y2k*lamy)) / cofy[j];
		c3[j] = -1.0*((y2k*lamy)+(yk*lamy*dy/2.0))/cofy[j];
		posz[j] = (depth + ((double)ejgsd/(double)eidisc)) + ((1.0 - expa1) / var_b);
	}

	// calculate dt dependent parameters
	dz_top = posz[numy+numdepth]-posz[numy+numdepth-1];   // shouldn't really be here!
	lamda_top = dt / (dz_top*dz_top);                     // lamda value next to substrate

	// calculate enzyme parameters
	if (approx_level == APPROX_1)
	{
		alpha = 1.0 + ((k_mo*v_max*dz_top) / ((1.0 + k_mo)*(1.0 + k_mo)));
		beta = -1.0 * (v_max*dz_top)/((1.0 + k_mo)*(1.0 + k_mo));
	}
	else if (approx_level == APPROX_2)
	{
		alpha = 1.0;
		beta = -1.0 * dz_top * v_max;
	}
	else if (approx_level == APPROX_3)
	{
		alpha = 1.0 + (v_max*dz_top / k_mo);
		beta = 0.0;
	}
	else 
	{
		cout << "WARNING - not a recognised approximation level, using defaults" << endl;
		alpha = 1.0; beta = 1.0;
	}

	// normal params
	c_1o[numytot-1] = c2o[numytot-1] + (c3[numytot-1]/alpha); // substrate
	c_1r[numytot-1] = c2o[numytot-1]; // system for invalid enzyme driven conc

	// loop to calculate all params.
	for (j=numytot-2 ; j>=1 ; j--)
	{
		c_1o[j] = c2o[j] - (c3[j] / c_1o[j+1]);
		c_1r[j] = c2o[j] - (c3[j] / c_1r[j+1]); // system for invalid enzyme driven conc. looks wrong, but is okay
	}

return;

} // adi_init

// ****************

void adi_calc(double d_t, int ts)
{
	int i,j,int_limit;							// interger counters


	double b4o[MAX_X+2], b_1o[MAX_X+2];		// species A ADI parameters (radial implicit, explicit and implicit terms)

	double d_1o[MAX_Y+1];	// A&B ADI parameters (axial implicit)
	double d4o[MAX_Y+1];		// A and B ADI parameters (axial implicit, explicit terms)


	// ********************************************
	// Implicit calculation in the RADIAL direction
	// ********************************************


	// Initialise pointers to arrays
	poldo=Amatrix1;						
	pnewo=Amatrix2;

	// set bulk boundary values (not really neccessary)
	// perp(numxtot+1,0,numytot); 
	parral(0,numx+1,numxtot);

	// loop upwards
	for (j = 1; j<=(numy+numdepth-1); j++)
	{	
		// outward (towards bulk) sweep to calculate explicit terms 
		if ( j <= numdepth ) { int_limit = numx+2; } else { int_limit = 1; }         
		for (i = int_limit; i<=numxtot; i++)
		{
			b4o[i] = ((-1.0*poldo[i][j-1]*cofy[j]) + 
				(((-1.0*c2o[j]*cofy[j])+2.0)*poldo[i][j]) +
				(-1.0*c3[j]*cofy[j]*poldo[i][j+1]))/cofx[i];
		}
		
		// inward (towards symmetry axis) sweep to consider implicit bulk BC  
		b_1o[numxtot] = b4o[numxtot]; // if 1 term here, no flux at radial limit
		for ( i = numxtot-1 ; i>=int_limit; i-- )
		{
			b_1o[i] = b4o[i] - (a3[i]*b_1o[i+1]/a_1o[i+1]);
		}

		// final outward loop to calculate new concentrations - no flux
		pnewo[int_limit-1][j] = b_1o[int_limit]/(1.0+a_1o[int_limit]);

		for (i=int_limit ; i<=numxtot ; i++)
		{
			pnewo[i][j] = (b_1o[i] - pnewo[i-1][j])/a_1o[i];
		}
	}

	// set remaining boundarys
	electrode(numdepth,0,numd);			// electrode b.c 
	j = numdepth;						// no flux glass
	for (i=numd+1; i<=numx ; i++) 
	{
		pnewo[i][j] = pnewo[i][j+1];
	}
	j = numy+numdepth;					// substrate
	for (i=0; i<=numxtot ; i++) 
	{
		pnewo[i][j] = (pnewo[i][j-1] + beta)/alpha;
		// there is a zeroth order term. So we have to check if less than zero
		if (pnewo[i][j] < 0.0) pnewo[i][j] = 0.0;
	}
	i= numxtot+1;
	for (j=0; j<=numytot; j++)			// zero flux at radial limit
	{
		pnewo[i][j] = pnewo[i-1][j];
	}

	// *******************************************
	// Implicit calculation in the AXIAL direction
	// *******************************************

//	if (ts==1) { map_the_conc(0,numxtot+1,0,numytot,999.0,1,1); }

	// Initialise pointers to arrays
	poldo=Amatrix2;						
	pnewo=Amatrix1;

	// Now start concentration changing steps 

	// Boundary values - not really necessary
	parral(0,numx+1,numxtot);  

	// Sideways sweep (away from symmetry axis)
	for (i=1; i<=numxtot; i++)
	{
		// Are we above microelectrode assembly or bulk solution?
		if ( i <= numx+1 ) { int_limit = numdepth+1; } else { int_limit = 1; } 
	
		// Calculate upwardly the explicit terms
		for (j=int_limit; j<=numytot-1; j++)
		{
			d4o[j] = ((-1.0*cofx[i]*poldo[i-1][j]) +
						(((-1.0*a2o[i]*cofx[i])+2.0)*poldo[i][j]) +
						(-1.0*a3[i]*cofx[i]*poldo[i+1][j]))/cofy[j];
		}

		// ASSUMING THAT THE ZEROTH ORDER ENZYME EQUATION IS VALID..
		// Calculate downward considering substrate boundary condition
		d_1o[numytot-1] = d4o[numytot-1] - (c3[numytot-1]*beta/alpha); // axial limit

		for (j=numytot-2 ; j>=int_limit ; j--) 
		{
			d_1o[j] = d4o[j] - (c3[j]*d_1o[j+1]/c_1o[j+1]);
		}

		// over bulk solution already taken care of (by parallel function)

		if ((i > numd)&&(i <= numx+1))		// over glass - no flux
		{
			pnewo[i][int_limit-1] = d_1o[int_limit]/(1.0 + c_1o[int_limit]);
		}
				
		else if (i <= numd)					// constant potential electrode 
		{
			pnewo[i][int_limit-1] = 1.0;
		}
		
		// loop outward to calculate new concentrations
		for (j=int_limit;j<=numytot-1;j++) 
		{
			pnewo[i][j] = (d_1o[j] - pnewo[i][j-1])/c_1o[j];
		}

		// along substrate
		pnewo[i][numytot] = (pnewo[i][j-1] + beta)/alpha;

		// HOWEVER, IF INVALID CONCENTRATION, WE HAVE TO DO THE WHOLE LOT AGAIN!
		
		if (pnewo[i][numytot] < 0.0)
		{	
			pnewo[i][numytot] = 0.0;
			
			// Calculate downward considering substrate boundary condition
			d_1o[numytot-1] = d4o[numytot-1]; // axial limit

			for (j=numytot-2 ; j>=int_limit ; j--) 
			{
				d_1o[j] = d4o[j] - (c3[j]*d_1o[j+1]/c_1r[j+1]);
			}

			// over bulk solution already taken care of (by parallel function)

			if ((i > numd)&&(i <= numx+1))		// over glass - no flux
			{
				pnewo[i][int_limit-1] = d_1o[int_limit]/(1.0 + c_1r[int_limit]);
			}
				
			else if (i <= numd)					// constant potential electrode 
			{
				pnewo[i][int_limit-1] = 1.0;
			}
		
			// loop outward to calculate new concentrations
			for (j=int_limit;j<=numytot-1;j++) 
			{
				pnewo[i][j] = (d_1o[j] - pnewo[i][j-1])/c_1r[j];
			}

		} // if invalid concentration detected

	} // i loop

	// Symmetry axis
	for (j=numdepth; j<=numytot ; j++) 
	{
		pnewo[0][j] = pnewo[1][j];
	}
	// Side of glass
	for (j = 1; j <= numdepth-1 ; j++) 
	{ 
		pnewo[numx+1][j] = pnewo[numx+2][j];
	}
	// Zero flux at radial limit
	i = numxtot+1;
	for (j=0; j<=numytot; j++)			
	{
		pnewo[i][j] = pnewo[i-1][j];
	}

	return;

} // adi_calc

// ****************

inline void perp(int a,int b,int c)
{
	int i,j;		// integer counters

	i = a;
	for (j = b; j<=c; j++)
	{
		pnewo[i][j] = cobO;
	}
	return;

} // perp

// ****************

inline void parral(int a,int b,int c)
{
	int i,j;		// integer counters

	j = a;
	for (i = b; i<=c; i++)
	{
		pnewo[i][j] = cobO;
	}
	return;

} // parral

// ****************

// ****************

inline void electrode(int a,int b,int c)
{
	int i,j;		// integer counters

	j = a;
	for (i = b; i<=c; i++)
	{
		pnewo[i][j] = 1.0; 
	}
	return;

} // electrode_z

// ****************

// ****************

// ****************

void map_the_conc(int bl, int br, int lb, int lt, double timepos, int timecount, bool test)
{

	ofstream outMAP;					// output file stream

	int i,j,conclimitx,conclimity;		// integer counters and size limiters (for printing lareg maps)
	char concfile[] ={"xmap.smap"};		// the filename mask (x is overwritten)

	concfile[0]=(char)(timecount+65);	// number filenames Amap,Bmap,Cmap etc

	// if its the final map use a special filename
	if (test==1) { strcpy(concfile,"FILE.smap"); }

	// open datafile, setup floating point output preferences and stamp it with the time
	outMAP.open(concfile);

	outMAP.setf(ios::fixed, ios::floatfield);			// decimal notation specified
	outMAP.setf(ios::showpoint);						// decimal point always shown even for whole numbers

	outMAP << setprecision(8) << timepos << "\n";		// set number of decimal places + output time
    
	// limit printout size to 100x100
	conclimitx=numxtot/200;
	conclimity=numytot/200;
	if (conclimitx==0) {conclimitx=1;}
	if (conclimity==0) {conclimity=1;}

	j=lb;

	// loop through and print concentrations of A and B
	for (; j<=lt;)
    {
		i = bl;
		for (; i<=(br);)
		{
			outMAP<<posr[i]<<"\t"<<posz[j]<<"\t"<<pnewo[i][j]<<endl;
			i+=conclimitx;
		}
    j+=conclimity;
    }

	// close mapping data file
	outMAP.close();

return;
}

inline double calc_tipcurrent(void)
{
	
	double nsum,dlctip;		// summing variable and calculated substrate current
	int i;					// integer counter

	nsum = 0.0;
	// for (i = 1; i<=numd ; i++) { nsum += posr[i]*(posr[i]-posr[i-1])*delta*(pnewB[i][numdepth+1]-pnewB[i][numdepth]); }
	for (i = 1; i<=numd ; i++) { nsum += posr[i]*(posr[i]-posr[i-1])*(pnewo[i][numdepth]-pnewo[i][numdepth+1]); }
	dlctip = M_PI * nsum / (2.0 * dz_b1);

	return dlctip;

} // calc_tipcurrent


