


// winmicrocvins.cpp - A Microelectrode Cyclic Voltrammetry Simulation
//              
//				A microelectrode response undergoing potential 
//              ramps is simulated close to a substrate (insulating).        
// 
//				Simulation for MS WINDOWS 32 bit
//
//				Grid Shape:
//
//				/////////////SUBSTRATE(CV)///////////////
//				/////////////////////////////////////////
//				|			     						|
//				S										|
//				Y										|
//				M										|
//				M										|
//				|										|
//				A		species A and B					|
//				X										B	
//				I										|
//				S										U
//				|										|
//				|___________...............				L
//				|			...............				|
//				| electrode	... glass .....				K
//				|			...............				|
//		   ^	|	cycled	...............				|
//		Z  |	| potential	...............				|
//				|			...............				|
//				|			...............				|
//				|			...............				|
//              BULK
//			
//					R ->
//
//				This program implements ADI in two dimensions using
//				fully implicit boundary conditions and expanding
//				space and time grids
//				
//				This simulation simulates a homogeneous EC mechanism:
//				A + e <=> B
//				B -K-> C where K is the dimensionless rate constant for the loss of 'B' 
//
//				Version 1. Effect of proximity of substrate on microelectrode CV 
//
//				J.L.Amphlett

// substrate insulating

// Last Windows Update: Wednesday 24 November 1999
// based upon the copper simulation programs

// Includes and Definitions

#include <iostream.h>
#include <functional>
#include <math.h>
#include <string.h>
#include <fstream.h>
#include <iomanip.h>
#include <time.h>
// #include <afx.h>

#undef MAX_X
#undef MAX_Y
#undef M_PI
#undef idcode
#define MAX_X 1000
#define MAX_Y 1000
#define M_PI 3.141592654
#define idcode -1

// Prototypes
int main(void);								// main control subrountine
double welcome(void);						// welcomes user+input initial variables+calculate grid/time/potential parameters
int ferrors(int e_code);					// turn simulation error codes into user error messages
void clearbulk(int a, int b, int c, int d);	// clear the matrix square (with bulk concentrations) defined by a,b,c,d
void id_block(int a,int b,int c,int d);     // id's the microdisc with a marker concentration (idcode)
void adi_init(double dt);					// calculation of time independent ADI parameters
void adi_calc(double e_epot, double e_eincr, double d_t);	// the concentration calculation sub-routine (needs exp(substrate potential),exp(sub potential relative to cu(0)) + its half step increment + time step size)
inline void perp(int a,int b,int c);		// set vertical boundary bulk values 
inline void parral(int a, int b, int c);	// set horizontal boundary bulk values
inline void electrode(int a,int b,int c, double exp_pot);	// set radial electrode values using exp(potential)
inline double calc_tipcurrent(void);		// returns the dimensionless tip current
inline double calc_subcurrent(double d_t);	// returns the substrate current (normalised wrt steady state microelectrode current in the bulk) - needs timestep size
inline void map_the_conc(int bl, int br, int lb, int lt, double timepos, int timecount, bool test); // prints out cocentration profile from boundaries, marks with timepos, map number and a bool to see if its the final map (1=YES)

// Global Constants & Variables
const char *version[] = {"microCV-EC-ins rv 0.1"};	// program version number
const double mapping[] = {1E10};			// array of times to print out concentration profiles IMP:last number large to avoid problems
const double eta = 1.0;						// term of uncertainty in conservation of flux equation, units sec cm^-1
 
double delta;								// ration of CU(I):Cu(II) complex diffusion coefficients
double cobA, cobB;							// initial concentrations of A and B species
double einit, efinal, sr;					// dimensionless initial, final potentials and scanrate
double dx,dy;								// step change in virtual X and Y
double var_a, var_b;						// expansion grid coefficents
double de;									// change in dimensionless potential (per full ADI iteration)
int scanit;									// iterations reuired for 1 scan (right to left)
int ktmax;									// total number of simulation iterations (equal time teps)
int numd,numg,numbulk,numx,numxtot;			// i direction grid parameters
int numy1,numy,numytot,numdepth;			// j direction grid parameters

double eidisc,ejgsd;						// estimated grid size parameters used in welcome and adi_init
double dz_b1, depth, dz_m1, dz_m2, midz;	// calculated real space grid size parameters		

double Amatrix1[MAX_X+2][MAX_Y+1];			// A concentration old and new matrices		
double Amatrix2[MAX_X+2][MAX_Y+1];
double Bmatrix1[MAX_X+2][MAX_Y+1];			// B concentration old and new matrices
double Bmatrix2[MAX_X+2][MAX_Y+1];

double (*pnewA)[MAX_Y+1];					// pointer to new A concentration matrix
double (*poldA)[MAX_Y+1];					// pointer to old A concentration matrix
double (*pnewB)[MAX_Y+1];					// pointer to new B concentration matrix
double (*poldB)[MAX_Y+1];					// pointer to old B concentration matrix

double dr_m1;								// smallest change in dr (electrode/glass boundary)
double posr[MAX_X+2],posz[MAX_Y+1];			// arrays with real position of points (adi_init)
double a2[MAX_X+2],a3[MAX_X+2],cofx[MAX_X+2];	// ADI radial calculation coefficients
double a_1[MAX_X+2];							// inward ADI radial calculation coefficient
double c2[MAX_Y+1],c3[MAX_Y+1],cofy[MAX_Y+1];	// ADI axial calculation coefficients
double c_1[MAX_Y+1];						// ADI parameter (axial implicit)
double a2b[MAX_X+2],a_1b[MAX_X+2];			// extra ADI parameters for second species
double c2b[MAX_Y+1],c_1b[MAX_Y+1];			// extra ADI parameters for second species
bool plate;									// do we consider plating?
double dim_K;								// dimensionless rate constant for loss of B

// Main 

int main(void)
{
	double dt;							// time step size (per full ADI iteration)
	ofstream fp_current;				// output steam for current data
	double timepos,end_time;			// position in time and simulation end time
	double potential,epot,ede;			// potential applied to substrate, its exponential, its exponential of change
	double ede2,epot_incr;				// 1/2 time step exponential change, exponential potential factor to send to adi_calc
	int kt;								// iteration counter
	double ifctr,ifctrc;				// parameters to determine if current should be calculated
	double dlctip;						// dimensionless tip and substrate current (wrt the steady state microelectrode current in the bulk)
	int map_count;						// counting the number of concentration maps printed
	int oldoutperc = -1;				// value to used give fraction of simulation complete to user
	clock_t begint,markt;				// clock ticks, used to estimate finishing time
	double duration;					// total estimated duration to end of simulation in seconds
	int hours, mins;					// printout of estimated duration to end of simulation

	dt = welcome();						// greets users and determines grid size and time / potential scales

	fp_current.open("current.dat");		// open current data file
	
	fp_current.setf(ios::fixed, ios::floatfield);	// decimal notation specified
	fp_current.setf(ios::showpoint);				// decimal point always shown even for whole numbers

	fp_current << setprecision(6);		// set number of decimal places + output time
    			
	fp_current << " time \t E \t Itip \t Isub \t Cu0 cov." << endl;

	cout << "\n\n **** Running Simulation **** \n\n";

	// initialising simulation t=0
	poldA=Amatrix1;							//initialise array/matrix pointers
	poldB=Bmatrix1;
	pnewA=Amatrix2;
	pnewB=Bmatrix2;

	clearbulk(0,numxtot+1,0,numytot);			// clear the array (requires pointers to be set)
	id_block(0,numdepth-1,0,numx);				// identify the area of microelectrode assembly

	// starting conditions
	timepos = 0.0;								// dimensionless time = 0				
	map_count = 0;								// map count set to zero
	kt = 0;										// iterations = 0
	potential = einit;							// starting potential set
	ede = exp(de);	ede2 = exp(de/2.0);			// Computation time saving 
	epot = exp(potential);						// exponential parameters set (for potential)
	end_time = dt*ktmax;							
	ifctrc = end_time/500.0;
	ifctr = ifctrc;

	// we half the lenth of time step for double step ADI
	dt = dt/2;
	
	adi_init(dt);								// calculate ADI parameters that depend upon the size of the grid/time step size

	// time loop
	begint = clock();
	while (timepos < end_time)
	{
		timepos = timepos + (2.0*dt);			// alter time position	
		kt = kt++;								// new iteration

		// printout estimated time to finish. only works with equally spaced time steps!
		if (kt == 100) 
		{
			markt = clock();
			duration = (end_time/timepos)*((double)(markt - begint)/CLOCKS_PER_SEC);
			hours = (int)(duration/3600.0); duration = duration - ((double)hours*3600.0);
			mins = (int)(duration/60.0); duration = duration - ((double)mins*60.0);
			cout << endl << "Time To Finish: " << hours << " hrs " << mins << " mins " 
				<< duration << " secs " << endl << flush;
		}

		// which direction are we scanning in?
		// IMPORTANT: no point to this test for a simple A<->B
		
		if ((((kt-1)/scanit)%2)==0)                 // 1 = forwards (+ve) 0 = backwards (-ve)
		{
			// backward scan
			potential = potential - de;				// sort out potential			
			epot = epot / ede;						// calculate potential relative to A/B
			epot_incr = (1.0 / ede2);				// work out half step change in potential
		}
		else 
		{
			// forward scan
			potential = potential + de;				// sort out potential	
			epot = epot * ede;						// calculate potential relative to A/B
			epot_incr = (ede2);						// work out half step change in potential
		}
		adi_calc(epot,epot_incr,dt);			

		// Saves out concentration profiles if required
		if (timepos>=mapping[map_count])
		{
			map_the_conc(0,numxtot+1,0,numytot,timepos,map_count,0);
			map_count += 1;
		}

		// Calculates currents and saves them to file if required
		if (timepos>=ifctr)
		{
			dlctip = calc_tipcurrent();
			fp_current << timepos << "\t" << potential << "\t" << dlctip << endl;
			if ((int)(timepos*10/end_time) != oldoutperc) { oldoutperc = (int)(timepos*10/end_time); cout << oldoutperc; cout << flush; } 
			ifctr=ifctr+ifctrc;
		}
	} // time loop

	// prints out final concentration map
	map_the_conc(0,numxtot+1,0,numytot,timepos,map_count,1);

	cout << "\n\n******** SIMULATION END *********\n\n";

	fp_current.close();

	return 0;
} // main

// ****************

double welcome(void)
{
	double ts;								// returning time step size 
	double approxefinal,approxeinit;		// approximate finishing and starting dimensionless potentials
	double eiglass,eibulk,ejdepth;			// effective lengths for simulation boundary edges
	int temp1,temp2;						// temp integers to help in calculations
	double topz;							// real space tip / substrate gap

	cout << "\n\n\nADI Microelectrode CV Simulation\n"
		 << "version : " << *version << "\n"; 

	cout << "This simulation looks at the microelectrode reponse" << endl
		 << "as its potential is cycled. We consider it in proximity of" << endl
		 << "an insulating substrate" << endl << endl;

startagain:

	cout << "Please input these SIMULATION parameters:\n\n"
		 << "eidisc  : "; cin >> eidisc; 
	cout << "eiglass : "; cin >> eiglass;
	cout << "ejgsd   : "; cin >> ejgsd;
	cout << "eibulk  : "; cin >> eibulk;
	cout << "ejdepth : "; cin >> ejdepth;
	cout << "\nnumd   : "; cin >> numd;
	cout << "expansion coefficient A : "; cin >> var_a;

	dx = log(1.0 + var_a)/(double)numd;
	dy = dx;

	cout << "\nPlease input these DIMENSIONLESS experimental parameters:\n\n"
		 << "D(A)/D(B) : "; cin >> delta; 
	cout << "init A conc : "; cin >> cobA;
	cout << "init B conc  : "; cin >> cobB;
	cout << "approx E final (-ve)   : "; cin >> approxefinal;
	cout << "approx E initial (+ve) : "; cin >> approxeinit;
	cout << "scanrate         : "; cin >> sr;
	cout << "dimensionless K  : "; cin >> dim_K;

	cout << "\nPlease enter a value for 1/dt : ";  cin >> ts;
	cout << "\n\n";

	de = sr * (1.0/(double)ts);		// calculate step in potential from dt

	if ((approxefinal>0) || (approxeinit<0) || (numd<2) || (sr==0)) { goto startagain; }

	// assuming we want a potential step at E=0, calculate potential range parameters
	temp1 = ((int)(approxeinit/de));
	temp2 = ((int)(-1.0*approxefinal/de));
	scanit = temp1+temp2;
	einit = de*(double)temp1;
	efinal = -1.0*de*(double)temp2;

	cout << "Calculated Einit : " << einit << " Efinal : " << efinal << " de : " << de;
	cout << "\nIterations for 1 complete CV scan : " << 2*scanit;
	cout << "\nIterations in 1 diffusion time    : " << ts;
	cout << "\nPlease input total iterations to be simulated : "; cin >> ktmax;

	// now we know the time-potential-iteration relationship, 
	// the grid size can be calculated
	// grid given in lab book2 p 88

	dr_m1 = (pow((1.0+var_a),1.0/(double)numd)-1.0)/var_a;
	dz_b1 = dr_m1;
	topz=ejgsd/eidisc;
	numg = (int)(log(1.0+(var_a*((eiglass/eidisc)-1.0)))/dx);
	numbulk = (int)(log(1.0+(var_a*((eibulk/eidisc)-1.0)))/dx);
	numdepth = (int)(log(1.0+(var_a*(((ejdepth-ejgsd)/eidisc))))/dx);
	numx = numg + numd;
	numxtot = numd + numbulk;

	numy1 = (int)(log(1.0+(var_a*(ejgsd/(2.0*eidisc))))/dy);
	depth = ((exp(dy*numdepth))-1.0)/var_a;
	midz = ((exp(numy1*dy))-1.0)/var_a;

	var_b = ((exp(dy*numy1))-1.0)/(topz-midz);  
	numy=numy1*2;
	numytot = numdepth + numy;
	dz_m1 = midz - ((exp(dy*(numy1-1))-1.0)/var_a);
	dz_m2 = topz - ((exp(dy*(numy1-1))-1.0)/var_b) - midz;

	cout << "\nChosen Parameters\n";
	cout << "numd:"<<numd<<" numg:"<<numg<<" numxtot:"<<numxtot;
	cout << "\nnumy(gap):"<<numy<<" numytot:"<<numytot<<" var_a:"<<var_a<<" var_b:"<<var_b;
	cout << "\nlargest lamda value :"<<(1.0/(dr_m1*dr_m1*(double)ts));
	cout << "\nDATA INPUT OKAY\n\n";

	return (1.0/(double)ts);	// return initial timestep size

} // welcome

// ****************

int ferrors(int e_code)
{
	// turns error code into user message

	cout << "\n\nFATAL ERROR:";

	switch(e_code)
	{
	case 1: cout << "\nError in creating Current Output File\n\n";
	default: cout << "\nAn unefined error has happened in the simulation\n\n";
	}

	return 0;

} // ferrors

// ****************

void clearbulk(int a, int b, int c, int d)
{
	int i,j;

	for (i = a; i <= b; i++)
    {
    for (j = c; j <= d; j++)
         {
         poldA[i][j] = cobA;
         poldB[i][j] = cobB;
         }
    }

return;     
} // clearbulk 

// ****************

void id_block(int a,int b,int c,int d)
{
int i,j;

for (j = a; j<=b; j++)
     {
     for (i = c; i<=d; i++)
            {
            pnewA[i][j] = idcode;
			pnewB[i][j] = idcode;
			poldA[i][j] = idcode;
			poldB[i][j] = idcode;
            }
     }
} // id_block            

// ****************

void adi_init(double dt)
{	
	// calculation of time independent ADI parameters

	int i,j;										// Integer counters
	double lamx,lamy,maxlam;						// General calculation		
	double x2k,xk,y2k,yk,zsumdiv2;					//	variables only
	double r1,r2,z1;								//	applicable to 
	double var_a1,var_a2,var_a3,expc1,expa1,expa2;	//	adi_init

	// constants used to simplify later calculations

	lamx = dt / (dx * dx);
	lamy = lamx;
	maxlam = dt / (dr_m1 * dr_m1);

	var_a1 = (var_a * var_a) / ((1.0 + var_a) * (1.0 + var_a));
	var_a2 = var_a * var_a * (1.0 + var_a) * (1.0 + var_a);
	var_a3 = (var_a + 1.0) * (var_a - 1.0);

	// Radial Loop Calculations

	posr[0] = 0.0;
	expa1 = 1.0;
	expc1 = exp(dx);

	for (i = 1 ; i <= numd-1 ; i++)				// electrode 
	{
		expa1 = expa1 * expc1;
		r1 = (var_a1 * expa1) / (1.0 - (1.0 / expa1));
		r2 = var_a1 * (expa1 * expa1);
		x2k = r2;
		xk = r1 + r2;
		cofx[i] = ((-1.0*lamx*x2k) + (xk*lamx*dx/2.0));
		a2[i] = (1.0 + (x2k*2.0*lamx))/cofx[i];
		a2b[i] = ((dim_K*dt) + 1.0 + (x2k*2.0*lamx))/cofx[i];
		a3[i] = -1.0*((x2k*lamx) + (xk*lamx*dx/2.0))/cofx[i]; 
		posr[i] = ((1.0 + var_a)*(1.0 - (1.0 / expa1))) / var_a;
	}          

	i = numd;									// electrode - glass boundary
	expa1 = expa1 * expc1;
	cofx[i] = ((-1.0*maxlam) + (maxlam*dr_m1/2.0));
	a2[i] = (1.0 + (2.0*maxlam))/cofx[i];
	a2b[i] = ((dim_K*dt) + 1.0 + (2.0*maxlam))/cofx[i];
	a3[i] = -1.0*((maxlam) + (maxlam*dr_m1/2.0))/cofx[i]; 
	posr[numd] = 1.0;

	for (i = numd+1 ; i <= numxtot+1 ; i++)		// glass and bulk boundary
	{
		expa1 = expa1 * expc1;
		r1 = (var_a2 / expa1) / (expa1 + var_a3);
		r2 = var_a2 / (expa1 * expa1);
		x2k = r2;
		xk = r1 - r2;
		cofx[i] = ((-1.0*lamx*x2k) + (xk*lamx*dx/2.0));
		a2[i] = (1.0 + (x2k*2.0*lamx))/cofx[i];
		a2b[i] = ((dim_K*dt) + 1.0 + (x2k*2.0*lamx))/cofx[i];
		a3[i] = -1.0*((x2k*lamx) + (xk*lamx*dx/2.0))/cofx[i]; 
		posr[i] = (expa1 + var_a3) / (var_a * (var_a + 1.0));
	}    
    
	// now calculate inward parameter 
	a_1[numxtot] = a2[numxtot] + a3[numxtot];	// if there is an a3 term here then there is a no flux condition at the radial bulk axis
	a_1b[numxtot] = a2b[numxtot] + a3[numxtot];	// if there is an a3 term here then there is a no flux condition at the radial bulk axis

	for ( i = numxtot - 1 ; i >= 1 ; i--)
	{
		a_1[i] = a2[i] - ( a3[i] / a_1[i+1] );
		a_1b[i] = a2b[i] - ( a3[i] / a_1b[i+1] );
	}

	// Axial Loop Calculations
	posz[0] = 0.0;
	expa1 = exp(numdepth*dy);
	expc1 = exp(-1.0*dy);

	expa2 = exp(numy * dy);

	for (j = 1 ; j <= numdepth-1 ; j++)					// behind electrode assembly
	{
		expa1 = expa1 * expc1;
		z1 = var_a * var_a / (expa1 * expa1);
		y2k = z1;
		yk = z1;
		cofy[j] = ((-1.0*y2k*lamy)+(yk*lamy*dy/2.0));
		c2[j] = (1.0 + (2.0*y2k*lamy)) / cofy[j];
		c2b[j] = ((dim_K*dt) + 1.0 + (2.0*y2k*lamy)) / cofy[j];
		c3[j] = -1.0*((y2k*lamy)+(yk*lamy*dy/2.0)) / cofy[j];
		posz[j] = depth + ((1.0 - expa1) / var_a);     
	}

	j = numdepth;										// grid join flush with face
	expa1 = expa1 * expc1;
	cofy[j] = -1.0 * dt / (dz_b1 * dz_b1);
	c2[j] = (1.0 + (2.0*dt/(dz_b1*dz_b1)))/cofy[j];
	c2b[j] = ((dim_K*dt) + 1.0 + (2.0*dt/(dz_b1*dz_b1)))/cofy[j];
	c3[j] = 1.0;
	posz[j] = depth;

	for (j = (numdepth+1) ; j <= (numdepth+numy1-1) ; j++)	// lower tip - substrate gap
	{
		expa1 = expa1 * expc1;
		z1 = var_a * var_a * (expa1 * expa1);
		y2k = z1;
		yk = -1.0 * z1;
		cofy[j] = ((-1.0*y2k*lamy)+(yk*lamy*dy/2.0));
		c2[j] = (1.0 + (2.0*y2k*lamy)) / cofy[j];
		c2b[j] = ((dim_K*dt) + 1.0 + (2.0*y2k*lamy)) / cofy[j];
		c3[j] = -1.0*((y2k*lamy)+(yk*lamy*dy/2.0)) / cofy[j];
		posz[j] = depth + (((1.0/expa1) - 1.0) / var_a);     
	}

	j = numy1 + numdepth;									// mid gap grid join	
	expa1 = expa1 * expc1;
	zsumdiv2 = (dz_m1 + dz_m2)/2.0;
	cofy[j] = -1.0 * dt / (dz_m1 * zsumdiv2);
	c2[j] = (1.0 + (dt/(dz_m2*zsumdiv2)) + (dt/(dz_m1*zsumdiv2)))/cofy[j];
	c2b[j] = ((dim_K*dt) + 1.0 + (dt/(dz_m2*zsumdiv2)) + (dt/(dz_m1*zsumdiv2)))/cofy[j];
	c3[j] = (-1.0 * dt / (dz_m2 * zsumdiv2))/cofy[j];
	posz[j] = midz + depth;

	expa1 = expa1 * expa2;
	for (j = (numy1+1+numdepth) ; j <= (numy+numdepth) ; j++)	// upper tip - substrate gap
	{
		expa1 = expa1 * expc1;
		z1 = var_b * var_b / (expa1 * expa1);
		y2k = z1;
		yk = z1;
		cofy[j] = ((-1.0*y2k*lamy)+(yk*lamy*dy/2.0));
		c2[j] = (1.0 + (2.0*y2k*lamy)) / cofy[j];
		c2b[j] = ((dim_K*dt) + 1.0 + (2.0*y2k*lamy)) / cofy[j];
		c3[j] = -1.0*((y2k*lamy)+(yk*lamy*dy/2.0))/cofy[j];
		posz[j] = (depth + ((double)ejgsd/(double)eidisc)) + ((1.0 - expa1) / var_b);
	}

	// normal params
	c_1[numytot-1] = c2[numytot-1]+c3[numytot-1];
	c_1b[numytot-1] = c2b[numytot-1]+c3[numytot-1];

	// loop to calculate all params.
	for (j=numytot-2 ; j>=1 ; j--)
	{
		c_1[j] = c2[j]-(c3[j]/c_1[j+1]);
		c_1b[j] = c2b[j]-(c3[j]/c_1b[j+1]);
	}

return;

} // adi_init

// ****************

void adi_calc(double e_epot, double e_eincr, double d_t)
{
	int i,j,int_limit;							// interger counters

	double Ab4[MAX_X+2], A_b[MAX_X+2];	// species A ADI parameters (radial implicit, explicit and implicit terms)
	double Bb4[MAX_X+2], B_b[MAX_X+2];	// species B ADI parameters (radial implicit, explicit and implicit terms)

	double B_d[MAX_Y+1], A_d[MAX_Y+1]; // A&B ADI parameters (axial implicit)
	double Ad4[MAX_Y+1], Bd4[MAX_Y+1];	// A and B ADI parameters (axial implicit, explicit terms)

	double dz_top;								// gap between j = numytot-1 and j = numytot (on surface of substrate)

	// ********************************************
	// Implicit calculation in the RADIAL direction
	// ********************************************

	dz_top = posz[numytot]-posz[numytot-1];

	// Initialise pointers to arrays
	poldA=Amatrix1;						
	poldB=Bmatrix1;
	pnewA=Amatrix2;
	pnewB=Bmatrix2;

	// set bulk boundary values (not really neccessary)
	// perp(numxtot+1,0,numytot); 
	parral(0,numx+1,numxtot);

	// loop upwards
	for (j = 1; j<=(numy+numdepth-1); j++)
	{	
		// outward (towards bulk) sweep to calculate explicit terms 
		if ( j <= numdepth ) { int_limit = numx+2; } else { int_limit = 1; }         
		for (i = int_limit; i<=numxtot; i++)
		{
			Ab4[i] = ((-1.0*poldA[i][j-1]*cofy[j]) + 
				(((-1.0*c2[j]*cofy[j])+2.0)*poldA[i][j]) +
				(-1.0*c3[j]*cofy[j]*poldA[i][j+1]))/cofx[i];
			Bb4[i] = ((-1.0*poldB[i][j-1]*cofy[j]) + 
				(((-1.0*c2[j]*cofy[j])+2.0)*poldB[i][j]) +
				(-1.0*c3[j]*cofy[j]*poldB[i][j+1]))/cofx[i];
		}
		
		// inward (towards symmetry axis) sweep to consider implicit bulk BC
		A_b[numxtot] = Ab4[numxtot]; // if 1 term here, no flux at axial limit
		B_b[numxtot] = Bb4[numxtot]; // if 1 term here, no flux at axial limit
		for ( i = numxtot-1 ; i>=int_limit; i-- )
		{
			A_b[i] = Ab4[i] - (a3[i]*A_b[i+1]/a_1[i+1]);
			B_b[i] = Bb4[i] - (a3[i]*B_b[i+1]/a_1b[i+1]);
		}

		// final outward loop to calculate new concentrations
		pnewA[int_limit-1][j] = A_b[int_limit]/(1.0+a_1[int_limit]);
		pnewB[int_limit-1][j] = B_b[int_limit]/(1.0+a_1b[int_limit]);
		for (i=int_limit ; i<=numxtot ; i++)
		{
			pnewA[i][j] = (A_b[i] - pnewA[i-1][j])/a_1[i];
			pnewB[i][j] = (B_b[i] - pnewB[i-1][j])/a_1b[i];
		}
	}

	// set remaining boundarys
	electrode(numdepth,0,numd,e_epot);	// electrode b.c determined by potential
	j = numdepth;						// no flux glass
	for (i=numd+1; i<=numx ; i++) 
	{
		pnewA[i][j] = pnewA[i][j+1];
		pnewB[i][j] = pnewB[i][j+1];
	}
	j = numy+numdepth;					// no flux substrate
	for (i=0; i<=numxtot ; i++) 
	{
		pnewA[i][j] = pnewA[i][j-1];
		pnewB[i][j] = pnewB[i][j-1];
	}
	i= numxtot+1;
	for (j=0; j<=numytot; j++)			// zero flux at radial limit
	{
		pnewA[i][j] = pnewA[i-1][j];
		pnewB[i][j] = pnewB[i-1][j];
	}

	// *******************************************
	// Implicit calculation in the AXIAL direction
	// *******************************************

	// Initialise pointers to arrays
	poldA=Amatrix2;						
	poldB=Bmatrix2;
	pnewA=Amatrix1;
	pnewB=Bmatrix1;

	e_epot = e_epot * e_eincr;			// change potentials for half time step

	// Now start concentration changing steps 

	// Boundary values - not really necessary
	parral(0,numx+1,numxtot);  

	// Sideways sweep (away from symmetry axis)
	for (i=1; i<=numxtot; i++)
	{
		// Are we above microelectrode assembly or bulk solution?
		if ( i <= numx+1 ) { int_limit = numdepth+1; } else { int_limit = 1; } 
	
		// Calculate upwardly the explicit terms
		for (j=int_limit; j<=numytot-1; j++)
		{
			Ad4[j] = ((-1.0*cofx[i]*poldA[i-1][j]) +
						(((-1.0*a2[i]*cofx[i])+2.0)*poldA[i][j]) +
						(-1.0*a3[i]*cofx[i]*poldA[i+1][j]))/cofy[j];
			Bd4[j] = ((-1.0*cofx[i]*poldB[i-1][j]) +
						(((-1.0*a2[i]*cofx[i])+2.0)*poldB[i][j]) +
						(-1.0*a3[i]*cofx[i]*poldB[i+1][j]))/cofy[j];
		}

		// Calculate downward considering substrate boundary condition
		A_d[numytot-1] = Ad4[numytot-1];
		B_d[numytot-1] = Bd4[numytot-1];
	
		for (j=numytot-2 ; j>=int_limit ; j--) 
		{
			A_d[j] = Ad4[j] - (c3[j]*A_d[j+1]/c_1[j+1]);
			B_d[j] = Bd4[j] - (c3[j]*B_d[j+1]/c_1b[j+1]);
		}

		// over bulk solution already taken care of (by parallel function)

		if ((i > numd)&&(i <= numx+1))		// over glass
		{
			pnewA[i][int_limit-1] = A_d[int_limit]/(1.0 + c_1[int_limit]);
			pnewB[i][int_limit-1] = B_d[int_limit]/(1.0 + c_1b[int_limit]);
		}
				
		else if (i <= numd)					// cycled potential electrode 
		{
			pnewA[i][int_limit-1] = ((delta*B_d[int_limit])+A_d[int_limit])
				/(1.0 + c_1[int_limit] + (delta/e_epot) + (delta*c_1[int_limit]/e_epot)); 
			pnewB[i][int_limit-1] = ((delta*B_d[int_limit])+A_d[int_limit])
				/(e_epot + (c_1b[int_limit]*e_epot) + delta + (delta*c_1b[int_limit])); 
		}
		  
		for (j=int_limit;j<=numytot-1;j++) 
		{
			pnewA[i][j] = (A_d[j] - pnewA[i][j-1])/c_1[j];
			pnewB[i][j] = (B_d[j] - pnewB[i][j-1])/c_1b[j];
		}

		// along substrate - no flux
		j = numytot;
		pnewA[i][j] = pnewA[i][j-1];
		pnewB[i][j] = pnewB[i][j-1];

	} // i loop

	// Symmetry axis
	for (j=numdepth; j<=numytot ; j++) 
	{
		pnewA[0][j] = pnewA[1][j];
		pnewB[0][j] = pnewB[1][j];
	}
	// Side of glass
	for (j = 1; j <= numdepth-1 ; j++) 
	{ 
		pnewA[numx+1][j] = pnewA[numx+2][j];
		pnewB[numx+1][j] = pnewB[numx+2][j];
	}
	// Zero flux at radial limit
	i = numxtot+1;
	for (j=0; j<=numytot; j++)			
	{
		pnewA[i][j] = pnewA[i-1][j];
		pnewB[i][j] = pnewB[i-1][j];
	}

	return;

} // adi_calc

// ****************

inline void perp(int a,int b,int c)
{
	int i,j;		// integer counters

	i = a;
	for (j = b; j<=c; j++)
		{
		pnewA[i][j] = cobA;
		pnewB[i][j] = cobB;
		}
	return;

} // perp

// ****************

inline void parral(int a,int b,int c)
{
	int i,j;		// integer counters

	j = a;
	for (i = b; i<=c; i++)
		{
		pnewA[i][j] = cobA;
		pnewB[i][j] = cobB;
		}
	return;

} // parral

// ****************

// ****************

inline void electrode(int a,int b,int c,double exp_pot)
{
	int i,j;		// integer counters

	j = a;
	for (i = b; i<=c; i++)
		{
		pnewA[i][j] = ((delta*pnewB[i][j+1]) + pnewA[i][j+1])/(1.0 + (delta/exp_pot));
		pnewB[i][j] = ((delta*pnewB[i][j+1]) + pnewA[i][j+1])/(delta + exp_pot); 
		}
	return;

} // electrode_z

// ****************

// ****************

// ****************

void map_the_conc(int bl, int br, int lb, int lt, double timepos, int timecount, bool test)
{

	ofstream outMAP;					// output file stream

	int i,j,conclimitx,conclimity;		// integer counters and size limiters (for printing lareg maps)
	char concfile[] ={"xmap.smap"};		// the filename mask (x is overwritten)

	concfile[0]=(char)(timecount+65);	// number filenames Amap,Bmap,Cmap etc

	// if its the final map use a special filename
	if (test==1) { strcpy(concfile,"FILE.smap"); }

	// open datafile, setup floating point output preferences and stamp it with the time
	outMAP.open(concfile);

	outMAP.setf(ios::fixed, ios::floatfield);			// decimal notation specified
	outMAP.setf(ios::showpoint);						// decimal point always shown even for whole numbers

	outMAP << setprecision(8) << timepos << "\n";		// set number of decimal places + output time
    
	// limit printout size to 100x100
	conclimitx=numxtot/100;
	conclimity=numytot/100;
	if (conclimitx==0) {conclimitx=1;}
	if (conclimity==0) {conclimity=1;}

	j=lb;

	// loop through and print concentrations of A and B
	for (; j<=lt;)
    {
		i = bl;
		for (; i<=(br);)
		{
			outMAP<<posr[i]<<"\t"<<posz[j]<<"\t"<<pnewA[i][j]<<"\t"<<pnewB[i][j]<<endl;
			i+=conclimitx;
		}
    j+=conclimity;
    }

	// close mapping data file
	outMAP.close();

return;
}

inline double calc_tipcurrent(void)
{
	
	double nsum,dlctip;		// summing variable and calculated substrate current
	int i;					// integer counter

	nsum = 0.0;
	// for (i = 1; i<=numd ; i++) { nsum += posr[i]*(posr[i]-posr[i-1])*delta*(pnewB[i][numdepth+1]-pnewB[i][numdepth]); }
	for (i = 1; i<=numd ; i++) { nsum += posr[i]*(posr[i]-posr[i-1])*(pnewA[i][numdepth]-pnewA[i][numdepth+1]); }
	dlctip = M_PI * nsum / (2.0 * dz_b1);

	return dlctip;

} // calc_tipcurrent


