#!/usr/bin/env python3

import os
import shutil
from subprocess import Popen, PIPE

from PyPython import SpectrumUtils


PARAMETERS = [
    "Central_object.mass(msol)",
    "Central_object.radius(cm)",
    "Disk.mdot(msol/yr)",
    "Disk.radmax(cm)",
    "Wind.mdot(msol/yr)",
    "SV.diskmin(units_of_rstar)",                 
    "SV.diskmax(units_of_rstar)",                 
    "SV.thetamin(deg)",
    "SV.thetamax(deg)",
    "SV.mdot_r_exponent",
    "SV.v_infinity(in_units_of_vescape",
    "SV.acceleration_length(cm)",
    "SV.acceleration_exponent",
    "SV.v_zero_mode(fixed,sound_speed)",
    "SV.v_zero(multiple_of_sound_speed)",
    "SV.gamma(streamline_skew;1=usually)",
    "Wind.radmax(cm)",
    "Wind.filling_factor(1=smooth,<1=clumped)"
]


def create_new_spectrum_file(root: str, directory: str, name: str):
    """Create a new spectrum file with the description of the wind parameters."""

    print("Making spectrum for", name)

    with open(directory + "/" + root + ".pf", "r") as f:
        pf = f.readlines()

    n = len(pf)
    parameters = []
    for i in range(n):
        pf[i] = pf[i].replace("\n", "")
        for p in PARAMETERS:
            if pf[i].find(p) != -1:
                parameters.append(pf[i].split())
                break

    preamble = []
    with open(directory + "/" + root + ".spec", "r") as f:
        for i in range(6):
            preamble.append(f.readline().replace("\n", "")) 
    preamble.append("")
    del preamble[3]

    s = SpectrumUtils.read_spec(directory + "/" + root + ".spec")
    s = s.drop(["Created", "Emitted", "CenSrc", "Disk", "Wind", "HitSurf", "Scattered"], axis=1) 

    f = open(name + ".txt", "w")

    f.write("# https://github.com/saultyevil/python/tree/tde\n")
    for l in preamble:
        f.write("{}\n".format(l))

    for l in parameters:
        f.write("# {:45s} {}\n".format(l[0], l[1]))
    f.write("\n")

    for l in s:
        f.write("{:20s} ".format(l),)
    f.write("\n")

    for index, row in s.iterrows():
        f.write("{:20s} {:20s} {:20s} {:20s} {:20s} {:20s} {:20s} {:20s}\n".format(
            str(row["Freq."]), str(row["Lambda"]), str(row["10"]), str(row["30"]), str(row["45"]), str(row["60"]), 
            str(row["75"]), str(row["85"])
        ))

    f.close()

    try:
        os.mkdir("pf")
    except:
        pass

    f = open("pf/" + name + ".pf", "w")

    for p in pf:
        p = p.split()
        if p[0][0] != "@":
            f.write("{:115} {:40s}\n".format(p[0], p[1]))

    f.close()

    return


home = "/home/saultyevil/PySims/tde_uv/models/"

directories = [
    "clump/1e-1/agn/cno",
    "clump/1e-1/agn/solar",
    "clump/1e-1/cv/cno",
    "clump/1e-1/cv/solar",
    "smooth/agn/cno",
    "smooth/agn/solar",
    "smooth/cv/cno",
    "smooth/cv/solar",
]

names = [
    "clump_equatorial_cno",
    "clump_equatorial_solar",
    "clump_wide_angle_cno",
    "clump_wide_angle_solar",
    "smooth_equatorial_cno",
    "smooth_equatorial_solar",
    "smooth_wide_angle_cno",
    "smooth_wide_angle_solar",
]

roots = [
    "tde_agn", "tde_agn", "tde_cv", "tde_cv", "tde_agn", "tde_agn", "tde_cv", "tde_cv"
]


nspec = len(directories)

for i in range(nspec):
    create_new_spectrum_file(roots[i], home + directories[i], names[i])

