%% Draw Cf vs. Rex (Figure 3)
clc; clear all; close all;

addpath '..\Functions\'

%% Input
%Measurement data
x = 8.6; %streamwise location in m

%Porous wall (use this as a reference to plot Cf at constant Unif/nu and constant ks/x
BFR = 8.5;
k = 0.015; %porous height, 15 mm
ks = 7.83E-3; %fitted ks for porous wall 
K = 3.62E-8; 

%Directory: smooth, porous, rough 1, rough 2, porous-rough 1, porous-rough
%2 (HWA data)
dir0 = {'..\Data\S\' ...
    '..\Data\P\' ...
    '..\Data\R1\' ...
    '..\Data\R2\' ...
    '..\Data\PR1\' ...
    '..\Data\PR2\'};

%Smooth wall FE drag balance data
dir_fe = '..\Data\S\';

%Ref porous wall (Esteban, 2022)
dir_ref = '..\Data\Ref\';
name_ref = '45ppi_15mm';
par_ref.x = 3.3; %streamwise location (m)
par_ref.ks = 7.97E-3; %calculated ks

%For plotting Cf at constant Uinf/nu and constant ks/x
Re_const.ks = ks;
Re_const.Re_unit = [0.15/8 0.15/4 0.15 0.6 1.8 7.2 28.8]*1E6;

%Constant L (or ks(x))
L_const.L = [3.3 8.8 35.2 140.8 563.2 2.2528E3]; %length in m

%% Color and marker setup for plotting
%Case number for matching Re_tau
id = {[7] ; ... 
    [3 4 5 6 7]; ... 
    [1 2 3 4 5]; ... 
    [1 2 3 4]; ... 
    [2 3 4 5 6]; ... 
    [2 3 4 5]};

%Color scheme for matching Re_tau
col = [1.000, 0.881, 0.494; ... %11600
       1.000, 0.683, 0.130; ... %14400
       0.976, 0.452, 0.057; ... %18600
       0.788, 0.260, 0.271; ... %23400
       0.374, 0.136, 0.493; ... %27700
       0 0 0]; %33100

id_col = {[1]; ...
    [2 3 4 5 6]; ...
    [2 3 4 5 6]; ...
    [2 3 4 5]; ...
    [2 3 4 5 6]; ...
    [2 3 4 5]};

mark = ['s' 'o' '^' 'd' 'p' 'h'];

%% Load data
%HWA data (all test cases)
N = length(dir0);
for i = 1:N
    data{i} = load([dir0{i},'\hw_data.mat']); 
end

%Smooth wall (FE drag balance data)
load([dir_fe,'\wss_fe_data.mat'])

%Reference data (porous wall, Cf vs. Re_x)
ref = readtable([dir_ref,'\',name_ref,'.xlsx']);

%% Derivation of function C_f vs Re_x for smooth wall (see Monty et al, 2016)

%delta_plus (Re_tau)
lb = 1E2; ub = 1E7;
npoints = 50;
delta_plus = logspace(log10(lb),log10(ub),npoints)';
Rex_lim = [3E5 1E8]; 
Cf_lim = [1.75E-3 1E-2]*1E3; 

%parameters based on measurements
nu = mean(data{1}.hw.nu);
B = mean(data{1}.hw.B);
PI = mean(data{1}.hw.PI);
kappa = mean(data{1}.hw.kappa);

%find C_f vs Re_theta
for j = 1:length(delta_plus)
    [yplus,uplus,S] = wake_jones(kappa, B, PI, delta_plus(j));
    Re_theta(j,1) = trapz(yplus,uplus-uplus.^2/S);
    Cf(j,1) = 2./S.^2;
end

%find C_f vs Re_x
for i = 2:length(delta_plus)
    Rex(i,1) = trapz(Re_theta(1:i),2./Cf(1:i));
end

%% Porous wall: Cf for constant unit Rex (Uinf/nu)
for ii = 1:length(Re_const.Re_unit)
    %find C_f vs Re_theta
    for j = 1:length(delta_plus)

        %solve for Cf
        optfun = @(Cf_guess) solve_Cf_rough(Cf_guess, kappa, delta_plus(j), Re_const.ks, Re_const.Re_unit(ii), BFR, PI);
        Re_const.Cf(j,ii) = fmincon(optfun,0.001,[],[],[],[],0,1,[]);

        %solve for theta
        ks_plus = Re_const.ks*Re_const.Re_unit(ii)*sqrt(Re_const.Cf(j,ii)/2);
        deltau_plus = 1/kappa*log(ks_plus) + B - BFR;
        [yplus,uplus,S] = wake_jones(kappa, B, PI, delta_plus(j));
        uplus  = uplus - deltau_plus;
        S = S - deltau_plus;
        Re_theta(j,1) = trapz(yplus,uplus-uplus.^2/S);
    end
    
    %find C_f vs Re_x
    for i = 2:length(delta_plus)
        Re_const.Rex(i,ii) = trapz(Re_theta(1:i),2./Re_const.Cf(1:i,ii));
    end
end

%% Porous wall: Cf for constant L (ks/x)
for j = 1:length(L_const.L)
    %lookup table
    for ii = 1:length(Re_const.Re_unit)
        L_const.Rex(ii,j) = L_const.L(j)*Re_const.Re_unit(ii);
        L_const.Cf(ii,j) = interp1(Re_const.Rex(:,ii),Re_const.Cf(:,ii),L_const.Rex(ii,j),'linear');
    end
end

%% Figure
[fig1,axx] = setFigure([13.5 10],[0.15 1.35 0.9 0.1],1,[2 3],[1 3],'centimeters');
pos = get(axx{1},'Position');
set(axx{1},'Position',[pos(1) pos(2)-0.5 pos(3)-6 pos(4)+0.5])

for i = 2:4
    pos = get(axx{i},'Position');
    set(axx{i},'Position',[pos(1) pos(2)-0.4 pos(3) pos(4)-0.5])
end

%% Start the plot
ax1 = axx{1}; axes(ax1);
set(ax1,'XScale','log','YScale','log','TickLength',[0.02 0.04],'GridLineStyle','-','MinorGridLineStyle','-',...
    'XTick',[1E4 1E5 1E6 1E7 1E8 1E9],'YTick',[1:10],'FontSize',9)

% Porous wall: draw lines for constant unit Rex (Uinf/nu)
cols = cbrewer('div','RdBu',5,'spline');
for ii = 1:length(Re_const.Re_unit)
    loglog(Re_const.Rex(:,ii),Re_const.Cf(:,ii)*1E3,'-','Color',cols(4,:),'LineWidth',0.5)
end

%% Porous wall: draw lines for constant L (ks/x)
for ii = 1:length(L_const.L)
    loglog(L_const.Rex(:,ii),L_const.Cf(:,ii)*1E3,'-','Color',cols(2,:),'LineWidth',0.5)
end

%% Annotations

for ii = 2:length(Re_const.Re_unit)
    if ii == 3
        [~,id1] = min(abs(Re_const.Cf(:,ii)-6E-3));
        text(Re_const.Rex(id1,ii),6,['$Re_k = ',num2str(Re_const.Re_unit(ii)*k*1E-4,'%.2f'),'\times 10^4$'],...
            'Background','w','FontSize',7,'Color',cols(4,:),'Margin',0.1,...
            'HorizontalAlignment','center','VerticalAlignment','middle','Rotation',-25)
    elseif ii == 2
        [~,id1] = min(abs(Re_const.Cf(:,ii)-5E-3));
        text(Re_const.Rex(id1,ii),5,[num2str(Re_const.Re_unit(ii)*k*1E-4,'%.2f')],...
            'Background','w','FontSize',7,'Color',cols(4,:),'Margin',0.1,...
            'HorizontalAlignment','left','VerticalAlignment','middle','Rotation',-30)
    else
        [~,id1] = min(abs(Re_const.Cf(:,ii)-9E-3));
        text(Re_const.Rex(id1,ii),9,[num2str(Re_const.Re_unit(ii)*k*1E-4,'%.2f')],...
            'Background','w','FontSize',7,'Color',cols(4,:),'Margin',0.1,...
            'HorizontalAlignment','left','VerticalAlignment','middle','Rotation',-35)
    end
end

for ii = 1:length(L_const.L)
    [~,id1] = min(abs(L_const.Rex(:,ii)-1E8));
    if ii == 3
        text(1.25*L_const.Rex(id1,ii),L_const.Cf(id1,ii)*1E3,['$k_s/x = ',num2str(ks./L_const.L(ii)*1E4,'%.2f'),'\times 10^{-4}$'], ...
            'Background','w','FontSize',7,'Color',cols(2,:),'Margin',0.1,...
            'HorizontalAlignment','right','VerticalAlignment','middle')
    else
        text(8E7,L_const.Cf(id1,ii)*1E3,num2str(ks./L_const.L(ii)*1E4,'%.2f'), ...
            'Background','w','FontSize',7,'Color',cols(2,:),'Margin',0.1,...
            'HorizontalAlignment','right','VerticalAlignment','middle')
    end
end

%% Smooth walls
loglog(Rex,Cf*1E3,'-k','LineWidth',2) %theoretical smooth wall ZPG TBL Cf
hold on
loglog(x*data{1}.hw.Rex,data{1}.hw.Cf*1E3,'sk','MarkerFaceColor',[0.5 0.5 0.5],'MarkerSize',5) %from composite fitting
loglog(x*mean(fe.Rex,1),mean(fe.Cf*1E3,1),'sk','MarkerFaceColor','w','MarkerSize',5) %from FE measurements
xlim(Rex_lim); ylim(Cf_lim)
xlabel('$Re_x$'); ylabel('$C_f$ $[\times 10^{-3}]$')

[~,id1] = min(abs(Cf-3.5E-3));
text(Rex(id1),3.5,'Smooth wall',...
    'Background','w','FontSize',7,'Color','k','Margin',0.1,...
    'HorizontalAlignment','center','VerticalAlignment','middle','Rotation',-25)

%% Porous walls
%reference case
loglog(par_ref.x*ref.Re_x,ref.C_f*1E3,'ok','MarkerFaceColor','w','MarkerSize',4) %reference, Esteban et al (2022)

%measurement results
for j = 1:length(id{2})
    loglog(x*data{2}.hw.Rex(id{2}(j)),data{2}.hw.Cf(id{2}(j))*1E3,mark(2), ...
        'Color',col(id_col{2}(j),:),'MarkerEdgeColor',col(id_col{2}(j),:),'MarkerFaceColor','w','MarkerSize',4)
end

%% Rough walls
for j = 1:length(id{3})
    loglog(x*data{3}.hw.Rex(id{3}(j)),data{3}.hw.Cf(id{3}(j))*1E3,mark(3), ...
        'Color',col(id_col{3}(j),:),'MarkerEdgeColor',col(id_col{3}(j),:),'MarkerFaceColor','w','MarkerSize',4)
end

for j = 1:length(id{4})
    loglog(x*data{4}.hw.Rex(id{4}(j)),data{4}.hw.Cf(id{4}(j))*1E3,mark(4), ...
        'Color',col(id_col{4}(j),:),'MarkerEdgeColor',col(id_col{4}(j),:),'MarkerFaceColor','w','MarkerSize',4)
end

%% Porous-rough walls
for j = 1:length(id{5})
    loglog(x*data{5}.hw.Rex(id{5}(j)),data{5}.hw.Cf(id{5}(j))*1E3,mark(5), ...
        'Color',col(id_col{5}(j),:),'MarkerEdgeColor',col(id_col{5}(j),:),'MarkerFaceColor','w','MarkerSize',5)
end

for j = 1:length(id{6})
    loglog(x*data{6}.hw.Rex(id{6}(j)),data{6}.hw.Cf(id{6}(j))*1E3,mark(6), ...
        'Color',col(id_col{6}(j),:),'MarkerEdgeColor',col(id_col{6}(j),:),'MarkerFaceColor','w','MarkerSize',5)
end

box on

%% Inset (zoom on test surfaces)
Cf_lim2 = [5 9];
Rex_lim2 = [7E6 4E7];

f = fill([Rex_lim2(1) Rex_lim2(1) Rex_lim2(2) Rex_lim2(2)],[Cf_lim2(1) Cf_lim2(2) Cf_lim2(2) Cf_lim2(1)],'k');
set(f,'FaceColor','none','EdgeColor','k','LineWidth',0.5)

pos = get(axx{1},'Position');
axi = axes('Position',[pos(1)+pos(3)+1 pos(2)+pos(4)-3.5 4.125 3]);

% Porous wall: draw lines for constant unit Rex (Uinf/nu)
cols = cbrewer('div','RdBu',5,'spline');
for ii = 1:length(Re_const.Re_unit)
    loglog(Re_const.Rex(:,ii),Re_const.Cf(:,ii)*1E3,'-','Color',cols(4,:),'LineWidth',0.5)
    hold on
end

% Porous wall: draw lines for constant L (ks/x)
for ii = 1:length(L_const.L)
    loglog(L_const.Rex(:,ii),L_const.Cf(:,ii)*1E3,'-','Color',cols(2,:),'LineWidth',0.5)
end

% Porous walls
%reference case
loglog(par_ref.x*ref.Re_x,ref.C_f*1E3,'ok','MarkerFaceColor','w','MarkerSize',5)

%measurements/fitting results
for j = 1:length(id{2})
    loglog(x*data{2}.hw.Rex(id{2}(j)),data{2}.hw.Cf(id{2}(j))*1E3,mark(2), ...
        'Color',col(id_col{2}(j),:),'MarkerEdgeColor',col(id_col{2}(j),:),'MarkerFaceColor','w','MarkerSize',5)
end

% Rough walls
for j = 1:length(id{3})
    loglog(x*data{3}.hw.Rex(id{3}(j)),data{3}.hw.Cf(id{3}(j))*1E3,mark(3), ...
        'Color',col(id_col{3}(j),:),'MarkerEdgeColor',col(id_col{3}(j),:),'MarkerFaceColor','w','MarkerSize',5)
end

for j = 1:length(id{4})
    loglog(x*data{4}.hw.Rex(id{4}(j)),data{4}.hw.Cf(id{4}(j))*1E3,mark(4), ...
        'Color',col(id_col{4}(j),:),'MarkerEdgeColor',col(id_col{4}(j),:),'MarkerFaceColor','w','MarkerSize',5)
end

% Porous-rough walls
for j = 1:length(id{5})
    loglog(x*data{5}.hw.Rex(id{5}(j)),data{5}.hw.Cf(id{5}(j))*1E3,mark(5), ...
        'Color',col(id_col{5}(j),:),'MarkerEdgeColor',col(id_col{5}(j),:),'MarkerFaceColor','w','MarkerSize',6)
end

for j = 1:length(id{6})
    loglog(x*data{6}.hw.Rex(id{6}(j)),data{6}.hw.Cf(id{6}(j))*1E3,mark(6), ...
        'Color',col(id_col{6}(j),:),'MarkerEdgeColor',col(id_col{6}(j),:),'MarkerFaceColor','w','MarkerSize',6)
end

% Set
set(axi,'XScale','log','YScale','log','TickLength',[0.03 0.06],'GridLineStyle','-','MinorGridLineStyle','-',...
    'XTick',[1E4 1E5 1E6 1E7 1E8 1E9],'YTick',[1:10],'FontSize',9,'TickLabelInterpreter','LaTex')
axi.XAxis.Exponent = 0;

xlabel('$Re_x$'); ylabel('$C_f$ $[\times 10^{-3}]$')
xlim(Rex_lim2); ylim(Cf_lim2)
box on

%% Delta U+ vs ks+ 

for i = [2 5 6]
    data{i}.hw.Rek = sqrt(K).*data{i}.hw.utau./data{i}.hw.nu;
end

ax1 = axx{3}; axes(ax1)
ax1.XAxisLocation = 'bottom';
ax1.YAxisLocation = 'left';
set(ax1,'XScale','log','YScale','linear','TickLength',[0.03 0.06],'GridLineStyle','-','MinorGridLineStyle','-',...
    'XTick',[1E0 1E1 1E2 1E3 1E4],'YTick',[0:2:20],'FontSize',9)
limx = [80 4E3];

%line for fully rough surface
dulog = 1/kappa*log(limx) + B - BFR;
plot(limx,dulog,'--k','LineWidth',0.5)
hold on

%plot all test surfaces
for i = 2:N
    for j = 1:2:length(id{i})
        if i == N || i == N-1
        plot(data{i}.hw.ks_plus(id{i}(j)),data{i}.hw.deltau_plus(id{i}(j)),mark(i), ...
            'Color',col(id_col{i}(j),:),'MarkerEdgeColor',col(id_col{i}(j),:),'MarkerFaceColor','w','MarkerSize',5,'HandleVisibility','off')
        else
        plot(data{i}.hw.ks_plus(id{i}(j)),data{i}.hw.deltau_plus(id{i}(j)),mark(i), ...
            'Color',col(id_col{i}(j),:),'MarkerEdgeColor',col(id_col{i}(j),:),'MarkerFaceColor','w','MarkerSize',4,'HandleVisibility','off')
        end
    end
end

xlim(limx); ylim([11 17])
xlabel('$k_{s_{(r, \, p, \, pr)}}^+$'); ylabel('$\Delta U_{(r, \, p, \, pr)}^+$')
box on

% leg = legend('$1/\kappa \ln k_s^+ + B - B_{FR}$','Location','SouthEast');
% set(leg,'Interpreter','LaTex','FontSize',8)

%% Delta U^+ vs Re_k
pos = get(ax1,'Position');
ax2 = axes('Position',pos); 

for i = [2 5 6]
    for j = 1:length(id{i})
        if i == N || i == N-1
        plot(data{i}.hw.Rek(id{i}(j)),data{i}.hw.deltau_plus(id{i}(j)),mark(i), ...
            'Color',col(id_col{i}(j),:),'MarkerEdgeColor','k','MarkerFaceColor',col(id_col{i}(j),:),'MarkerSize',5,'HandleVisibility','off','LineWidth',0.5)
        else
        plot(data{i}.hw.Rek(id{i}(j)),data{i}.hw.deltau_plus(id{i}(j)),mark(i), ...
            'Color',col(id_col{i}(j),:),'MarkerEdgeColor','k','MarkerFaceColor',col(id_col{i}(j),:),'MarkerSize',4,'HandleVisibility','off','LineWidth',0.5)
        end
        hold on
    end
end

ax2.XAxisLocation = 'top';
ax2.YAxisLocation = 'right';
ax2.Color = 'none'; ax2.Box = 'off';
set(ax2,'XScale','log','YScale','linear','TickLength',[0.03 0.06],'GridLineStyle','-','MinorGridLineStyle','-',...
    'XTick',[1E0 1E1 1E2],'YTick',[0:2:20],'FontSize',9,'YTickLabel',[],'TickLabelInterpreter','LaTex')
limx = [8 400];

xlabel('$Re_K$')
xlim(limx); ylim([11 17])

%% U+ vs y+
idpl = 4; %plot at Retau = 27700

ax1 = axx{2}; axes(ax1)
set(ax1,'XScale','log','YScale','linear','TickLength',[0.03 0.06],'GridLineStyle','-','MinorGridLineStyle','-',...
    'XTick',[1E0 1E1 1E2 1E3 1E4],'YTick',[0:5:40],'FontSize',9)
limx = [10 5E4];

p1 = plot(limx,1/kappa*log(limx)+B,'--k','LineWidth',0.5);
hold on
[uplus, yplus] = profile(data{2}.hw.y{id{2}(idpl)}(2:end), data{2}.hw.utau(id{2}(idpl)), ...
    data{2}.hw.kappa(id{2}(idpl)), data{2}.hw.s(id{2}(idpl)), data{2}.hw.delta(id{2}(idpl)), ...
    data{2}.hw.PI(id{2}(idpl)), data{2}.hw.nu(id{2}(idpl))); %smooth wall
plot(yplus,uplus,'-k','LineWidth',2)

for i = 2:6
    for j = idpl
        down = 2;

        if i == N || i == N-1
        plot(data{i}.hw.yplus{id{i}(j)}(2:end),data{i}.hw.uplus{id{i}(j)}(2:end),'-', ...
            'Color',col(id_col{i}(j),:),'LineWidth',0.5,'HandleVisibility','off')
        plot(data{i}.hw.yplus{id{i}(j)}(2:down:end),data{i}.hw.uplus{id{i}(j)}(2:down:end),mark(i), ...
            'Color',col(id_col{i}(j),:),'MarkerEdgeColor',col(id_col{i}(j),:),'MarkerFaceColor','w','MarkerSize',4,'HandleVisibility','off')
        else
        plot(data{i}.hw.yplus{id{i}(j)}(2:end),data{i}.hw.uplus{id{i}(j)}(2:end),'-', ...
            'Color',col(id_col{i}(j),:),'LineWidth',0.5,'HandleVisibility','off')
        plot(data{i}.hw.yplus{id{i}(j)}(2:down:end),data{i}.hw.uplus{id{i}(j)}(2:down:end),mark(i), ...
            'Color',col(id_col{i}(j),:),'MarkerEdgeColor',col(id_col{i}(j),:),'MarkerFaceColor','w','MarkerSize',3,'HandleVisibility','off')
        end
    end
end

xlabel('$(y+d)^+$'); ylabel('$U_{(r, \, p, \, pr)}^+$')
xlim(limx); ylim([0 34])
box on

text(1000,22,'Smooth wall',...
    'Background','none','FontSize',8,'Color','k','Margin',0.1,...
    'HorizontalAlignment','center','VerticalAlignment','bottom','Rotation',32.5)

% leg = legend(p1,'$1/\kappa \ln y^+ + B $','Location','SouthEast');
% set(leg,'Interpreter','LaTex','FontSize',8)

%% Defect U+

ax1 = axx{4}; axes(ax1)
set(ax1,'XScale','log','YScale','linear','TickLength',[0.03 0.06],'GridLineStyle','-','MinorGridLineStyle','-',...
    'XTick',[1E-3 1E-2 1E-1 1E0 1E1],'YTick',[0:4:20],'FontSize',9)
limx = [7E-4 3];

for i = 2:6
    for j = idpl
        if i == N || i == N-1
        plot(data{i}.hw.y{id{i}(j)}(2:end)/data{i}.hw.delta99(id{i}(j)), ...
            data{i}.hw.u0(id{i}(j))/data{i}.hw.utau(id{i}(j))-data{i}.hw.uplus{id{i}(j)}(2:end),'-', ...
            'Color',col(id_col{i}(j),:),'LineWidth',0.5)
        plot(data{i}.hw.y{id{i}(j)}(2:down:end)/data{i}.hw.delta99(id{i}(j)), ...
            data{i}.hw.u0(id{i}(j))/data{i}.hw.utau(id{i}(j))-data{i}.hw.uplus{id{i}(j)}(2:down:end),mark(i), ...
            'Color',col(id_col{i}(j),:),'MarkerEdgeColor',col(id_col{i}(j),:),'MarkerFaceColor','w','MarkerSize',4,'HandleVisibility','off')
        else
        plot(data{i}.hw.y{id{i}(j)}(2:end)/data{i}.hw.delta99(id{i}(j)), ...
            data{i}.hw.u0(id{i}(j))/data{i}.hw.utau(id{i}(j))-data{i}.hw.uplus{id{i}(j)}(2:end),'-', ...
            'Color',col(id_col{i}(j),:),'LineWidth',0.5)
        plot(data{i}.hw.y{id{i}(j)}(2:down:end)/data{i}.hw.delta99(id{i}(j)), ...
            data{i}.hw.u0(id{i}(j))/data{i}.hw.utau(id{i}(j))-data{i}.hw.uplus{id{i}(j)}(2:down:end),mark(i), ...
            'Color',col(id_col{i}(j),:),'MarkerEdgeColor',col(id_col{i}(j),:),'MarkerFaceColor','w','MarkerSize',3,'HandleVisibility','off')
        end
    end
end

xline(0.2,'--k','LineWidth',0.5);

xlabel('$(y+d)/\delta$'); ylabel('$U_{\infty}^+ - U_{(r, \, p, \, pr)}^+$')
xlim(limx); ylim([0 16])
box on

%% Subfigure title
[axsub,txtsub] = setSubFigTitle(axx,'topleft');
for j = 1:4
    pos = get(axsub{j},'Position');
    set(axsub{j},'Position',[pos(1)+0.05 pos(2)-0.5 pos(3:4)])
    txt = get(txtsub{j},'String');
    set(txtsub{j},'Background','w')
end
