function [figid,ax] = setFigure(figsize,margin,sep,grid,overlap,defunit)
%SETFIGURE Set a figure with multiple subplots.
%% Inputs:
% figsize   = figure width & height, [Width Height] (1x2 matrix)
% margin    = top, bottom, left, right margin, [top bottom left right]
%           (1x4 matrix)
% sep       = length of separator between rows and columns
% grid      = number of subplots inside figure, [num_row num_col] 
%           (1x2 matrix)
% overlap   = overlapping subplot IDs, see HELP SUBPLOT for usage 
%           https://au.mathworks.com/help/matlab/ref/subplot.html#btw1_6e-1
%           (nx2 matrix)
% defunit   = default figure units (cm, mm, etc.)

% Output:
% figid     = figure ID
% ax        = axes properties (1xN cell) 
%
% **Note for usage:
% To plot in each axis: plot(...,'Parent',ax{id}) or axes(ax{id})
% To delete a subplot: delete(ax{id})

%%
% Inputs
W = figsize(1); 
H = figsize(2);
t = margin(1);
b = margin(2);
l = margin(3);
r = margin(4);
nRow = grid(1);
nCol = grid(2);

% Set figure settings
set(0,'DefaultFigureUnits',defunit,...
    'DefaultFigurePaperUnits',defunit,...
    'DefaultAxesUnits',defunit,...
    'DefaultAxesGridLineStyle','--',...
    'DefaultAxesMinorGridLineStyle','--',...
    'DefaultAxesFontUnits','points',...
    'DefaultAxesFontSize',12,...
    'DefaultAxesFontName', 'Times',...
    'DefaultTextFontUnits','points',...
    'DefaultTextFontSize',12,...
    'DefaultTextFontName','Times',...
    'DefaultTextInterpreter','LaTex',...
    'DefaultLineLineWidth',1,...
    'DefaultAxesLineWidth',0.5)

% define geometry
h = (H-(nRow-1)*sep-t-b)/nRow; %subplot height
w = (W-(nCol-1)*sep-l-r)/nCol; %subplot width

% point of origin of each subplot  
for i = 1:nCol
    xstart(i,1) = l+(i-1)*sep+(i-1)*w;   
end
xstart = repmat(xstart,[nRow,1]);
for i = 1:nRow
    ystart((i-1)*nCol+1:i*nCol,1) = H-(t+(i-1)*sep+i*h); 
end

% vector position of subplots
pos = [xstart ystart repmat(w,[length(xstart),1]) repmat(h,[length(ystart),1])];

% ID/location of each subplot 
id = (1:nCol);
id = repmat(id,[nRow,1]);
for i = 2:nRow
    id(i,:) = id(i-1,:)+nCol;
end

% check for joined/overlapping subplots
if isempty(overlap) == 0
    % check if overlap is correct
    if length(unique(overlap)) < length(overlap(:))
        error('Subplot cannot be overlapped more than once.')
    end
    if max(overlap(:)) > max(id(:))
        error('Cannot overlap non-existent subplots.')
    end
    
    noverlap = size(overlap,1);
    temp = pos;
    kill = []; %suplot ID to be deleted
    for i = 1:noverlap
        
        %sort and find location of overlap
        overlap(i,:) = sort(overlap(i,:),2);
        [r1,c1] = find(id==overlap(i,1));
        [r2,c2] = find(id==overlap(i,2));
        
        if c1 == c2 %overlap spans column
            % subplots to be joined
            joinc = id(r1:r2,c1)'; 
            temp(joinc(end),4) = temp(joinc(1),2)-temp(joinc(end),2)+h;
            kill = [kill joinc(1:end-1)];
        elseif r1 == r2 %overlap spans row
            % subplots to be joined
            joinr = id(r1,c1:c2);
            temp(joinr(1),3) = temp(joinr(end),1)-temp(joinr(1),1)+w;
            kill = [kill joinr(2:end)];
        elseif r1 ~= r2 && c1 ~= c2 %overlap spans row and col
            joinrc = id(r1:r2,c1:c2); joinrc = joinrc(:)';
            [~,cmax] = max(pos(joinrc,1));
            [~,rmin] = min(pos(joinrc,2));
            [~,rmax] = max(pos(joinrc,2));
            
            temp(joinrc(rmin),3) = temp(joinrc(cmax),1)-temp(joinrc(rmin),1)+w;
            temp(joinrc(rmin),4) = temp(joinrc(rmax),2)-temp(joinrc(rmin),2)+h;
            joinrc(rmin) = [];
            kill = [kill joinrc];
        end
    end
    
    %check for double values in kill
    if length(unique(kill)) < length(kill)
        error('Subplot cannot be overlapped more than once.')
    end
    
    %delete IDs of overlapping subplots
    temp(kill,:) = [];
    pos = temp;
end

% initialize figure
figid = figure();
figure(figid);
set(figid,'position',[0 0 W H])
for i = 1:size(pos,1)
    ax{i} = axes;
    set(ax{i},'position',pos(i,:),'TickLabelInterpreter','LaTex','Layer','top') %set ticks above figures
    hold on
end

end

