%%% INTENSITY HEATMAPS AND WAVEGUIDE OUTPUT IMAGES (FIG.7)

% Created by: Callum Stirling, University of Southampton, UK
% Date: 2021-02-09

clear; close all; clc;
%% wavelength sweep

WL_start = 1920; % in nm
WL_step = 0.5; % in nm
WL_stop = 2020; % in nm

% vector for swept wavelengths, in microns
WL = ([WL_start:WL_step:WL_stop])*1e-3;

WLrange = [1.95 2.01]; % wavelength range to show on plot
WLshow = find(WL==1.95); % wavelength at which to show output image

%% load data and laser spectrum

% load laser spectrum, taken from laser documentation 
LasSpec = load('laser_spectrum.txt');

% load images recorded at each wavelength step, for each waveguide
filelist = dir('*.mat');
n = length(filelist);

% create pixel arrays for plotting output 31x31 pixel images
pixel_pitch = 30; % in microns
x = ([-15:1:15])*pixel_pitch; % in x-direction
y = x; % in y-direction

%% create figures
% loop over all files in filelist
for WGn=1:n
    
    % load file and get dimensions of array
    FileName = filelist(WGn).name;
    Temp = load(FileName).frames;
    [h,w,num]=size(Temp);
    
    % normalise to laser spectrum
    A = zeros(h,w,num);
    for ii=1:num
        % create x by y array of ones, and multiple fractional intensity of
        % laser spectrum
        A(:,:,ii) = ones(h,w)*LasSpec(ii);
    end
    normed = Temp./A;
    
    % create mask for Fourier filtering
    m = ones(1,((length(WL)-1)/2)+1);
    m(46:101) = 0;
    m = [m flip(m(1:100))];
    m = logical(m);
    
    % filter higher order components that are present in all the measure
    % spectra, using filtering function (at bottom of script)
    filted = filtering(normed,m);
    
    % select image to show, at chosen wavelength, and normalise to largest
    % pixel value
    frame = filted(:,:,WLshow);
    frame = real(10*log10(frame./(max(max(frame)))));
    
    % display output image
    figure; surf(x,y,frame,'EdgeColor','none'); view(2);
    colormap(gray);
    xlabel(['{\it x} (' 956 'm']'); xlim([min(x) max(x)]);
    ylabel(['{\it y} (' 956 'm']); ylim([min(y) max(y)]);
    set(gca,'FontSize',14);
    set(gcf,'color','w');
    
    % sum each output image in y-direction and normalise to max intensity
    % of all images
    sumi = squeeze(sum(filted));
    p=max(max(sumi));
    sumi = sumi./p;
    sumi = real(10*log10(sumi));
    
    % plot heatmap
    figure; surf(WL(1:length(sumi)),x,sumi,'EdgeColor','none'); view(2);
    colormap default; c = colorbar;
    c.Label.String = 'Normalised intensity (dB)';
    c.Label.FontSize = 14;
    xlim(WLrange); xticks([min(WLrange):0.02:max(WLrange)]);
    ylim([min(x) max(x)]);
    xlabel(['Wavelength (' 956 'm)']);
    ylabel(['{\it x} (' 956 'm']);
    set(gca,'YDir','reverse');
    set(gca,'FontSize',14);
    set(gcf,'color','w');
end

function filt = filtering(signal, mask)
% function for filtering the Fourier spectrum of a 3D signal array (i.e. an
% image evolving over time), based on a given 1D mask

[H,W,~]=size(signal);
    for w = 1:W
        for h = 1:H
            T = squeeze(signal(h,w,:));
            F = fft(T);
            F(~mask) = 0;
            filt(h,w,:) = real(ifft(F));
        end
    end
end