%%% SPECTRUM FITTING USING GENETIC ALGORITHM
% Created by: Callum Stirling, University of Southampton, UK
% Date: 2021-04-22
%% initialise
clear;
close all; clc;
rng default;

% Load MMI data. Comment/uncomment to fit the spectra for SWG-structured
% and conventional MMIs
FileName = 'SWG.mat'; fig = openfig('SWG.fig','invisible');
%FileName = 'conventional.mat'; fig = openfig('Conventional.fig','invisible');

% extract data from saved figure
dataObjs = findobj(fig,'-property','YData');
x1o = dataObjs(1).XData;
y2o = dataObjs(1).YData;
y1o = dataObjs(2).YData;
close;

wavelength = x1o(251:601);
y1 = y1o(251:601);
y2 = y2o(251:601);
WL = wavelength;
WLplot = [2900:2:3800];

Y1 = 10.^(y1./10);
Y2 = 10.^(y2./10);

%% adjust for noise
load('noisefloor.mat','floorfit');
noise = floorfit.p1.*WL.^3 + floorfit.p2.*WL.^2 + floorfit.p3.*WL + floorfit.p4;
N = 10.^(noise./10);
Y1 = Y1 - N;
Y2 = Y2 - N;
y1 = 10*log10(Y1);
y2 = 10*log10(Y2);

%% create weights for sum squared error
offset = 3;

% find peaks
[~,locs1] = findpeaks(-y1,'MinPeakDistance',30);
[~,locs2] = findpeaks(-y2,'MinPeakDistance',30);
locs2(end)=[];
locs = sort([locs1 locs2]);

% increase the weight of fitting around the peaks
W = ones(1,length(WL));
Wn = 2*ones(1,length(locs)); Wn(5:7) = 3; Wn(end) = 5; % wide
for ii = 1:length(locs)
    if locs(ii)>offset && locs(ii)<length(WL)-offset
        W(locs(ii)-offset:locs(ii)+offset) = Wn(ii);
    elseif locs(ii)<offset
        W(locs(1):locs(ii)+offset) = Wn(ii);
    else
        W(locs(ii)-offset:locs(end)) = Wn(ii);
    end
end

%% multiple shapes

% define polynomial order of performance params and number of fitting vars
order = 6;
nvars = (order+1)*3;

% initialise fit
Acon=[]; Bcon=[]; Aeq=[]; beq=[];
nonlcon = [];
lb = -Inf*ones(3,nvars/3); lb(3,:) = -5; lb = reshape(lb,1,nvars);
ub = Inf*ones(3,nvars/3); ub(3,:) = 5; ub = reshape(ub,1,nvars);
fo = optimoptions('ga','FunctionTolerance',1e-10,...
                    'MaxGenerations',50*nvars);
% calculate 'error' between measured spectra and analytical spectra
SSE  = @(F) err(y1,y2,WL,F,W);
% fit based on error between spectra
[f,sse,exitflag,output] = ga(SSE,nvars,Acon,Bcon,Aeq,beq,lb,ub,nonlcon,fo);
% generate MZI outputs
[f1, f2] = MZI(WLplot,f);
f = reshape(f,3,nvars/3);
%% performance

% convert wavelength to unitless variable for polynomial
dWL = max(wavelength)-min(wavelength); WLc = wavelength(round(length(wavelength)/2));
auxlambda = (WLplot-WLc)./(dWL);

% generate polynomials from fitted variables
pa = f(1,1:nvars/3);
pb = f(2,1:nvars/3);
pc = f(3,1:nvars/3);
shapeA = polyval(pa,auxlambda);
shapeB = polyval(pb,auxlambda);
shapeC = polyval(pc,auxlambda);

% convert polynomials to MMI outputs
phi0 = pi/2;
a0 = 1/sqrt(2);
b0 = 1/sqrt(2);
A = 0.1;
B = 0.5;
C = pi/15;
phi = phi0 + C*shapeC;
a = a0 + A*shapeA;
b = b0 + B*shapeB;

%% plots

% calculate performance parameters from MMI outputs
phaseError=phi-phi0;
imbalance=10*log10(abs(a./b).^2);
insertionLoss=-10*log10((abs(a).^2+abs(b).^2));

% plot MZI spectra
figure; hold on
plot(wavelength,y1,'b');
plot(wavelength,y2,'r');
plot(WLplot,f1,'m--');
plot(WLplot,f2,'k--');
xlim([3000 3700]);
ylim([-35 0]);

% plot phase error
figure; hold on
plot(WLplot,phaseError*180/pi,'r','Linewidth',2)
set(gca,'FontSize',16)
xlim([3000 3700]);
xlabel('Wavelength (nm)')
ylabel('MMI phase error (degree)')
grid on;

% plot insertion loss and imbalance
figure; hold on
plot(WLplot,insertionLoss,'r','Linewidth',2);
plot(WLplot,imbalance,'b','Linewidth',2);
legend('MMI insertion loss','MMI imbalance');
set(gca,'FontSize',16)
xlabel('Wavelength (nm)')
ylabel('(dB)')
xlim([3000 3700])
grid on;

%% save for plotting
%save(FileName,'wavelength','y1','y2','f1','f2','WLplot','imbalance','insertionLoss','phaseError');
