#!/usr/bin/env Rscript

colours <- c( "#007293", "#8A0065", "#8A0056" );
model <- function( f, Rs, Rp, Cp, alpha ) {
	Rs + 1 / ( 1 / Rp + ( 2i * pi * f * Cp ) ^ ( 1 - alpha ) );
} 

params <- read.csv( "data/tissue_params.csv", header = TRUE );
comsol <- read.csv( "data/comsol.csv", header =  FALSE );
colnames( comsol ) <- c( "distance", "mh", "hh", "dh", "eh", "sh",
	"freq", "imp" );
comsol <- comsol[comsol$mh == 25e-3 & comsol$hh == 5e-3 &
	comsol$dh == 1.7e-3 & comsol$eh == 100e-6 & comsol$sh == 40e-6,];

cairo_pdf( "graphs/comparison.pdf", onefile = TRUE,
	width = 6, height = 3 );
par( xaxs = "i", bty = "l", cex = 0.5, mgp = c( 2, 1, 0 ) );

layout( matrix( c( 1, 1, 2, 3, 3, 4 ), 3, 2 ), TRUE );

for ( d in c( 0.05, 0.09 ) ) {

	phy <- data.frame( freq = c(), imp = c() );
	n_phy <- 0;

	for ( i in seq( 4 ) ) {

		filename <- paste0(
			"data/tissue/", d * 1000, "mm-50g-w-", i, ".csv" );

		if ( file.exists( filename ) ) {

			physical <- read.csv( filename, header = TRUE );
			colnames( physical ) <- c( "freq", "imp", "phase" );

			if ( length( phy$imp ) < length( physical$imp ) ) {
				phy <- rbind( phy, data.frame( freq = physical$freq,
					imp = physical$imp *
						exp( physical$phase * pi / 180 ) ) );
			} else {
				phy$imp <- phy$imp + physical$imp *
						exp( 1i * physical$phase * pi / 180 );
			}

			n_phy <- n_phy + 1;

		} else {

			cat( paste0( "Missing ", filename, "\n" ) );

		}

	}

	phy$imp <- phy$imp / n_phy;

	c <- comsol[comsol$distance == d,];
	p <- params[params$dampness == "w" & params$mass == 150 &
		params$distance == d * 1000,];
	m <- sapply( c$freq, model,
		mean( p$Rs ), mean( p$Rp ),
		mean( p$Cp ), mean( p$alpha ) );

	par( mar = c( 0.5, 3, 3, 1 ) + 0.3, cex = 0.5 );

	plot( c( c$freq, c$freq ), Mod( c( c$imp, m ) ),
		type = "n", log = "xy", axes = FALSE,
		xlab = "", ylab = "Impedance / Ω",
		main = paste0(
			"Model Comparison, Electrodes ", d * 1000, "mm apart" ) );

	box();
	axis( side = 2 );

	lines( c$freq, Mod( c$imp ), col = colours[1] );
	lines( c$freq, Mod( m ), col = colours[2] );
	lines( phy$freq, Mod( phy$imp ), col = colours[3],
		lty = "dashed", lwd = 0.5 );

	legend( x = "topright", lty = c( "solid", "solid", "dashed" ),
		col = colours, legend = c(
			"FE Simlation", "Equiv. Circuit", "Measured" ) );

	par( mar = c( 3, 3, 0, 1 ) + 0.3 );

	plot( c( c$freq, c$freq ), Arg( c( c$imp, m ) ) * 180 / pi,
		type = "n", log = "x",
		xlab = "Frequency / Hz", ylab = "Phase / °" );

	lines( c$freq, Arg( c$imp ) * 180 / pi, col = colours[1] );
	lines( c$freq, Arg( m ) * 180 / pi, col = colours[2] );
	lines( phy$freq, Arg( phy$imp ) * 180 / pi,
		col = colours[3], lty = "dashed", lwd = 0.5 );

}
