#!/usr/bin/env Rscript

# Show the initial values for the nls fit, as dotted lines on the
# graphs.
show_initial = FALSE;
lwd = 1;

cairo_pdf( "graphs/copper.pdf", onefile = TRUE, width = 3, height = 3 );
par( mar = c( 5, 4, 4, 1 ) + 0.3, xaxs = "i", yaxs = "r", bty = "l", 
	cex = 0.5 );

model <- function( f, Rs, Rp, Cp, alpha ) {

	Rs = Mod( Rs );
	Rp = Mod( Rp );
	Cp = Mod( Cp );
	alpha = tanh( alpha ) / 2 + 0.5; # Hack to keep alpha in [0, 1]

	CL = 2e-17;
	CoL = 4.0e-5;

	Ri = 18e3;
	#Ci = 1.55e-10;
	#Li = 1.22e-7;
	t = 5.5e-9;
	Ci <- sqrt( CL * CoL );
	Li <- sqrt( CL / CoL );

	(
		1 / (
			1 / Rp + 
			( 1i * 2 * pi * f * Cp ) ^ alpha ) + 
		Rs + 
		1 / ( 
			1 / Ri + 
			2 * pi * f * 1i * Ci + 
			1 / ( 2 * pi * f * 1i * Li ) + 
			0 ) ) * 
		exp( -t * 2 * pi * f * 1i * ( f > 1e6 ) );

}

colours <- c( "#007293", "#8A0065", "#00811C" );

params <- data.frame();

for ( electrode in c( "a", "b" ) ) {
		for ( dampness in c( "d", "w" ) ) {
		for ( mass in c( 50, 100, 150 ) ) {

	# Get all the data.
	data <- data.frame();

	for ( rep in seq( 3 ) ) {

		filename <- paste( "data/copper/", electrode, "-", 
			mass, "g-", dampness, "-", rep, ".csv", sep = "" );

		if ( !file.exists( filename ) ) {
			print( paste( "Missing", filename ) );
			next;
		}

		file <- read.csv( filename, header = TRUE );

		rdata <- data.frame( freq = file[,1], rep = rep,
			imp = file[,2] * exp( 1i * file[,3] * pi / 180 ) );


		# Rough calculation of parameters to start the fit from:
		# - Series resistance is the minimum value
		# - Parallel is impedance at the start - series resistance
		# - Capacitance is calculated from parallel resistance and a
		#   rough break frequency
		# - Alpha starts at 0.95, N.B. tanh hack in model function
		sRs = min( Mod( rdata$imp ) );
		sRp = Mod( rdata$imp[20] ) - sRs;
		break_sample = 
			which( Mod( rdata$imp ) < 0.7 * ( sRs + sRp ) )[1];
		sCp = 1 / 
			( 2 * pi * sRp * rdata$freq[break_sample] );
		salpha = atanh( 2 * 0.95 - 1 );

		cont <- nls.control( 
			maxiter = 1000,
			minFactor = 1e-20, 
			warnOnly = TRUE );
		fit <- nls( 
			0 ~ Mod( imp - model( freq, Rs, Rp, Cp, alpha ) ),
			control = cont,
			data = rdata[rdata$freq > 1e3 & rdata$freq < 1e7,],
			start = list( 
				Rs = sRs, Rp = sRp, Cp = sCp, alpha = salpha ) );

		rdata$modelled <- model( rdata$freq,
			coef( fit )["Rs"], coef( fit )["Rp"],
			coef( fit )["Cp"], coef( fit )["alpha"] );

		rdata$medelled <- model( rdata$freq,
			sRs, sRp, sCp, salpha );

		data <- rbind( data, rdata );

		params <- rbind( params, data.frame(
			electrode = electrode, dampness = dampness, mass = mass,
			rep = rep, Rs = coef( fit )["Rs"], Rp = coef( fit )["Rp"],
			Cp = coef( fit )["Cp"], alpha = coef( fit )["alpha"] ) );

	}

	rm <- range( Mod( c( data$imp, data$modelled ) ) );
	rp <- range(
		( Arg( c( data$imp, data$modelled ) ) * 180 / pi + 270 ) %%
			360 - 270 );
	rr <- range( Re( c( data$imp[data$freq < 1e6],
		data$modelled[data$freq < 1e6] ) ), 0 );
	ri <- range( Im( c( data$imp[data$freq < 1e6],
		data$modelled[data$freq < 1e6] ) ), 0 );

	par( mfcol = c( 2, 1 ), mar = c( 0, 3, 3, 1 ) + 0.3, cex = 0.5,
		mgp = c( 2, 1, 0 ) );

	# Plot the magnitude

	plot( range( data$freq ), rm,
		log = "xy", type = "n",
		axes = FALSE,
		main = paste0( "On Copper, Electrode: ",
			if ( "a" == electrode ) "A" else "B", ", Mass: ", mass,
			"g, ", if ( "w" == dampness ) "Moistened" else "Dry" ),
		xlab = "", ylab = "" );

	for ( rep in seq( 3 ) ) {

		rdata <- data[data$rep == rep,];

		lines( rdata$freq, Mod( rdata$imp ), col = colours[rep], 
			lwd = lwd );
		lines( rdata$freq, Mod( rdata$modelled ), col = colours[rep],
			lty = "dashed", lwd = lwd );
		if ( show_initial ) {
			lines( rdata$freq, Mod( rdata$medelled ),
				col = colours[rep], lty = "dotted", lwd = lwd );
		}

	}

	box();
	axis( side = 2,
		at = axTicks( side = 2 )[axTicks( side = 2 ) > rm[1]] );
	mtext( side = 2, line = 2, text = "Impedance / Ω", cex = 0.5,
		at = sqrt( rm[1] * rm[2] ) );

	# Plot the phase

	par( mar = c( 4, 3, 0, 1 ) + 0.3 );

	plot( range( data$freq ), rp,
		log = "x", type = "n",
		axes = FALSE,
		xlab = "Frequency / Hz", ylab = "" );

	for ( rep in seq( 3 ) ) {

		rdata <- data[data$rep == rep,];

		lines( rdata$freq,
			( Arg( rdata$imp ) * 180 / pi + 270 ) %% 360 - 270,
			col = colours[rep], lwd = lwd );
		lines( rdata$freq,
			( Arg( rdata$modelled ) * 180 / pi + 270 ) %% 360 - 270,
			col = colours[rep], lty = "dashed", lwd = lwd );
		if ( show_initial ) {
			lines( rdata$freq,
				( Arg( rdata$medelled ) * 180 / pi + 270 ) %% 360 - 270,
				col = colours[rep], lty = "dotted", lwd = lwd );
		}

	}

	box();
	axis( side = 1 );
	axis( side = 2,
		at = axTicks( side = 2 )[axTicks( side = 2 ) < rp[2]] );
	mtext( side = 2, line = 2, text = "Phase / °", cex = 0.5,
		at = ( rp[1] + rp[2] ) / 2 );

	# Quick Wessel plot, for my own edification

	par( mar = c( 4, 3, 3, 1 ) + 0.3, mfcol = c( 1, 1 ), cex = 0.5 );

	plot( rr, ri, asp = 1, type = "n", xaxs = "r", yaxs = "r", 
		xlab = "Resistance / Ω", ylab = "Reactance / Ω" );

	abline( v = 0, lwd = 0.5 );
	abline( h = 0, lwd = 0.5);

	for ( rep in seq( 3 ) ) {

		rdata <- data[data$rep == rep,];

		lines( Re( rdata$imp ), Im( rdata$imp ), col = colours[rep], 
			lwd = lwd );
		lines( Re( rdata$modelled ), Im( rdata$modelled ), 
			col = colours[rep], lty = "dashed", lwd = lwd );
		if ( show_initial ) {
			lines( Re( rdata$medelled ), Im( rdata$medelled ), 
				col = colours[rep], lty = "dotted", lwd = lwd );
		}

	}

} } }

params$Rs = Mod( params$Rs );
params$Rp = Mod( params$Rp );
params$Cp = Mod( params$Cp );
params$alpha = tanh( params$alpha ) / 2 + 0.5; # Undo hack in method
params$alpha = 1 - params$alpha; # and define alpha correctly.

cairo_pdf( "./graphs/copper-params.pdf", onefile = TRUE,
	width=6, height = 3 );

par( mar = c( 4, 3, 3, 0 ) + 0.3, xaxs = "r", yaxs = "r", cex = 0.5, 
	mgp = c( 2, 1, 0 ), mfrow = c( 1, 4 ), cex.main = 0.8 );

plot( params$mass, params$Rs, 
	type = "p", log = "", axes = FALSE,
	main = "Series Resistance", 
	xlab = "Mass / g", ylab = "Resistance / Ω", 
	xlim = c( 40, 160 ),
	#ylim = c( 10, 5e3 ), 
	pch = sapply( params$electrode, 
			function( e ) {
				if ( "b" == e ) 1 else 3 } ),
	col = sapply( params$dampness, 
			function( d ){ 
				if ( d == "w" ) "#007293" else "#BA000A" } ) );
box();
axis( side = 2 );
axis( side = 1, at = c( 50, 100, 150 ) );
legend( x = "topright", 
	col = c( "#007293", "#BA000A", "black", "black" ), 
	pch = c( 19, 19, 3, 1 ), 
	legend = c( "Wet", "Dry", "A", "B" ), cex = 0.8 );

plot( params$mass, params$Rp, 
	type = "p", log = "", axes = FALSE,
	main = "Parallel Resistance", 
	xlab = "Mass / g", ylab = "Resistance / Ω", 
	xlim = c( 40, 160 ),
	#ylim = c( 1e3, 2e4 ),
	#pch = c( 1, 2, 3 ), 
	pch = sapply( params$electrode, 
			function( e ) {
				if ( "b" == e ) 1 else 3 } ),
	col = sapply( params$dampness,
			function( d ){ 
				if ( d == "w" ) "#007293" else "#BA000A" }) );
box();
axis( side = 2 );
axis( side = 1, at = c( 50, 100, 150 ) );
legend( x = "topright", 
	col = c( "#007293", "#BA000A", "black", "black" ), 
	pch = c( 19, 19, 3, 1 ), 
	legend = c( "Wet", "Dry", "A", "B" ), cex = 0.8 );

plot( params$mass, params$Cp, 
	type = "p", log = "y", axes = FALSE,
	main = "Parallel Capacitance", 
	xlab = "Mass / g", ylab = "Capacitance / F", 
	xlim = c( 40, 160 ),
	pch = sapply( params$electrode, 
			function( e ) {
				if ( "b" == e ) 1 else 3 } ),
	col = sapply( params$dampness, 
			function( d ){ 
				if ( d == "w" ) "#007293" else "#BA000A" }) );
box();
axis( side = 2 );
axis( side = 1, at = c( 50, 100, 150 ) );
legend( x = "bottomright", 
	col = c( "#007293", "#BA000A", "black", "black" ), 
	pch = c( 19, 19, 3, 1 ), 
	legend = c( "Wet", "Dry", "A", "B" ), cex = 0.8 );

plot( params$mass, params$alpha, 
	type = "p", axes = FALSE,
	main = "Alpha", 
	xlab = "Mass / g", ylab = "Alpha", 
	xlim = c( 40, 160 ),
	pch = sapply( params$electrode, 
			function( e ) {
				if ( "b" == e ) 1 else 3 } ),
	col = sapply( params$dampness,
			function( d ){ 
				if ( d == "w" ) "#007293" else "#BA000A" } ) );
box();
axis( side = 2 );
axis( side = 1, at = c( 50, 100, 150 ) );
legend( x = "topright",
	col = c( "#007293", "#BA000A", "black", "black" ), 
	pch = c( 19, 19, 3, 1 ), 
	legend = c( "Wet", "Dry", "A", "B" ), cex = 0.8 );
