#!/usr/bin/env Rscript

cairo_pdf( "graphs/tissue.pdf", onefile = TRUE, width = 3, height = 3 );
par( mar = c( 5, 4, 4, 4 ) + 0.3, xaxs = "i", yaxs = "r", bty = "l",
	cex = 0.5 );

colours <- c( "#007293", "#8A0065", "#00811C", "#c3b200" );

model <- function( freq, Rs, Rp, Cp, alpha ) {

	CL = 2e-17;

	CoL = 4.0e-5;

	Ri = 18e3;
	#Ci = 1.55e-10;
	#Li = 1.22e-7;
	t = 5.5e-9;
	Ci <- sqrt( CL * CoL );
	Li <- sqrt( CL / CoL );

	(
		Mod( Rs ) + 
			1 / ( 
				1 / Mod( Rp ) + 
				( 1i * 2 * pi * freq * Mod( Cp ) ) ^ alpha ) +
		1 / ( 
			1 / Ri + 
			2 * pi * freq * 1i * Ci + 
			1 / ( 2 * pi * freq * 1i * Li ) + 
			0 ) ) * 
		exp( -t * 2 * pi * freq * 1i * ( freq > 1e6 ) );

}

params <- data.frame( 
	dampness = c(), 
	mass = c(), 
	distance = c(), 
	rep = c(), 
	Rs = c(), 
	Rp = c(), 
	Cp = c(), 
	alpha = c() );

for ( damp in c( "d", "w" ) ) {
		for ( mass in c( 50, 100, 150 ) ) {
		for ( distance in c( 30, 50 , 70, 90 ) ) {

	data <- data.frame();

	for ( rep in seq( 1, 4 ) ) {

		filename <- paste( "data/tissue/", distance, "mm-", mass, "g-", 
			damp, "-", rep, ".csv", sep = "" )

		print( filename );

		if ( file.exists( filename ) ) {

			rdata <- cbind( rep = rep, 
					read.csv( filename, header = TRUE ) );

			colnames( rdata ) <- c( "rep", "freq", "mag", "phase" );

			rdata$imp = rdata$mag * exp( 1i * pi * rdata$phase / 180 );

			cont = nls.control(
				maxiter = 10e3, 
				minFactor = 1E-10,
				warnOnly = TRUE );
			fit = nls( 0 ~ Mod( imp - model( freq, Rs, Rp, Cp, alpha ) ), 
				data = rdata, 
				control = cont, 
				start = list( 
					Rs = 80, Rp = 50e3, Cp = 3e-8, alpha = 0.9 ) );

			#print( paste( 
			#	mass, "g, ", distance, "mm, ", damp, " ", rep, sep = "" ) );
			#print( fit );

			model_max <- 0.9 * Mod( 1 / (
				2 * pi * coef( fit )["Rs"] * coef( fit )["Cp"] ) );
			model_min <- Mod( 1 / ( 
				2 * pi * coef( fit )["Rp"] * coef( fit )["Cp"] ) );

			rdata <- rbind( rdata, 
				data.frame(
					rep = -rep, 
					freq = 10^c( seq( 
						log10( model_min ) - 2, 
						log10( model_max ) + 2, 0.05 ) ),
					mag = 0,
					phase = 0, 
					imp = model( 
						10^c( seq( 
							log10( model_min ) - 2, 
							log10( model_max ) + 2, 0.05 ) ), 
						coef( fit )["Rs"], 
						coef( fit )["Rp"], 
						coef( fit )["Cp"], 
						coef( fit )["alpha"] ) ) );

			data <- rbind( data, rdata );

			params <- rbind( params, data.frame( 
				dampness = damp, 
				mass = mass, 
				distance = distance, 
				rep = rep, 
				Rs = Mod( coef( fit )["Rs"] ), 
				Rp = Mod( coef( fit )["Rp"] ), 
				Cp = Mod( coef( fit )["Cp"] ), 
				alpha = Mod( coef( fit )["alpha"] ) ) );

		}

	}

	par( mfcol = c( 2, 1 ), mar = c( 0, 3, 3, 1 ) + 0.3, cex = 0.5,
		mgp = c( 2, 1, 0 ) );

	plot( data$freq, Mod( data$imp ), 
		type = "n", log = "xy", 
		axes = FALSE, 
		main = paste0( "On tissue, Mass: ", mass, "g, Distance: ",
			distance, "mm, ",
			if ( "w" == damp ) "Moistened" else "Dry" ),
		xlab = "", ylab = "Impedance / Ω" );
	box();
	axis( side = 2 );

	for ( rep in seq( 1, 4 ) ) {

		lines( data$freq[data$rep == rep], 
			Mod( data$imp[data$rep == rep] ), 
			lty = "solid", lwd = 1, col = colours[rep] );

		lines( data$freq[data$rep == -rep], 
			Mod( data$imp[data$rep == -rep] ),
			lty = "dashed", lwd = 1, col = colours[rep] );

	}

	par( mar = c( 4, 3, 0, 1 ) + 0.3 );

	plot( data$freq,
		( Arg( data$imp ) * 180 / pi + 270 ) %% 360 - 270,
		type = "n", log = "x", 
		axes = TRUE,
		xlab = "Frequency / Hz", ylab = "Phase / °" );

	for ( rep in seq( 1, 4 ) ) {

		lines( data$freq[data$rep == rep], 
			( Arg( data$imp[data$rep == rep] ) * 180 / pi + 270 ) %%
				360 - 270,
			lty = "solid", lwd = 1, col = colours[rep] );

		lines( data$freq[data$rep ==-rep], 
			( Arg( data$imp[data$rep == -rep] ) * 180 / pi + 270 ) %%
				360 - 270,
			lty = "dashed", lwd = 1, col = colours[rep] );

	}

	box();

	par( mar = c( 5, 4, 4, 2 ) + 0.1, xaxs = "i", yaxs = "i" );

	plot( Re( data$imp ), Im( data$imp ), 
		xlim = c( 
			min( 0, min( Re( data$imp ) ) ), 
			max( 0, max( Re( data$imp ) ) ) ), 
		ylim = c( 
			min( 0, min( Im( data$imp ) ) ), 
			max( 0, max( Im( data$imp ) ) ) ), 
		type = "n", asp = 1, 
		main = paste( mass, "g, ", distance, "mm, ", damp, sep = "" ), 
		xlab = "Re( Impedance ) / Ω", ylab = "Im( Impedance ) / Ω" );

	abline( h = 0, lty = "solid", lwd = 0.5 );
	abline( v = 0, lty = "solid", lwd = 0.5 );

	for ( rep in seq( 1, 4 ) ) {

		lines( Re( data$imp[data$rep == rep] ), 
			Im( data$imp[data$rep == rep] ), 
			lty = "solid", lwd = 1, col = colours[rep] );

		lines( Re( data$imp[data$rep == -rep] ), 
			Im( data$imp[data$rep == -rep] ), 
			lty = "dashed", lwd = 1, col = colours[rep] );

	}

} } }

params$alpha = 1 - params$alpha; #define alpha correctly

cairo_pdf( "graphs/tissue-params.pdf", onefile = TRUE );

write.csv( params, "data/tissue_params.csv", row.names = FALSE );
q();

par( mar = c( 5, 4, 4, 3 ) + 0.1, xaxs = "r", yaxs = "r" );

plot( params$distance + ( params$mass - 100 ) / 20, params$Rs, 
	type = "p", log = "y", 
	ylim = c( 20, 2e3 ), 
	main = "Series Resitance", 
	xlab = "distance / mm + ( Mass / g - 100 ) / 20", 
	ylab = "Resistance / Ω", 
	pch = params$mass / 50,
	col = sapply( params$dampness, function( d ) {
		if ( d == "d" ) colours[1] else colours[4] } ) );
legend( x = "bottomright",
	col = c( 
		"#000000", "#000000", "#000000", colours[1], colours[4] ), 
	pch = c( 1, 2, 3, 19, 19 ), 
	legend = c( "50g", "100g", "150g", "d", "Wet" ) );

plot( params$distance + ( params$mass - 100 ) / 20, params$Rp, 
	type = "p", log = "y", 
	ylim = c( 5e4, 5e7 ), 
	main = "Parallel Resitance", 
	xlab = "distance / mm + ( Mass / g - 100 ) / 20", 
	ylab = "Resistance / Ω", 
	pch = params$mass / 50,
	col = sapply( params$dampness, function( d ) {
		if ( d == "d" ) colours[1] else colours[4] } ) );
legend( x = "topleft",
	col = c( 
		"#000000", "#000000", "#000000", colours[1], colours[4] ), 
	pch = c( 1, 2, 3, 19, 19 ), 
	legend = c( "50g", "100g", "150g", "d", "Wet" ) );

plot( params$distance + ( params$mass - 100 ) / 20, params$Cp, 
	type = "p", log = "y", 
	ylim = c( 1e-12, 5e-8 ), 
	main = "Parallel Capacitance", 
	xlab = "distance / mm + ( Mass / g - 100 ) / 20", 
	ylab = "Capacitance / F", 
	pch = params$mass / 50,
	col = sapply( params$dampness, function( d ) {
		if ( d == "d" ) colours[1] else colours[4] } ) );
legend( x = "bottomright",
	col = c( 
		"#000000", "#000000", "#000000", colours[1], colours[4] ), 
	pch = c( 1, 2, 3, 19, 19 ), 
	legend = c( "50g", "100g", "150g", "d", "Wet" ) );

plot( params$distance + ( params$mass - 100 ) / 20, params$alpha, 
	type = "p", 
	ylim = c( 0.6, 0.93 ), 
	main = "Alpha", 
	xlab = "distance / mm + ( Mass / g - 100 ) / 20", 
	ylab = "Alpha", 
	pch = params$mass / 50,
	col = sapply( params$dampness, function( d ) {
		if ( d == "d" ) colours[1] else colours[4] } ) );
legend( x = "bottomright",
	col = c( 
		"#000000", "#000000", "#000000", colours[1], colours[4] ), 
	pch = c( 1, 2, 3, 19, 19 ), 
	legend = c( "50g", "100g", "150g", "d", "Wet" ) );

plot( params$distance + ( params$mass - 100 ) / 20, 
	1 / ( 2 * pi * params$Rp * params$Cp ), 
	type = "p", log = "xy", 
	ylim = c( 1e0, 1e4 ), 
	main = "Break Frequency", 
	xlab = "distance / mm + ( Mass / g - 100 ) / 20", 
	ylab = "Frequency / Hz", 
	pch = params$mass / 50,
	col = sapply( params$dampness, function( d ) {
		if ( d == "d" ) colours[1] else colours[4] } ) );
legend( x = "bottomright",
	col = c( 
		"#000000", "#000000", "#000000", colours[1], colours[4] ), 
	pch = c( 1, 2, 3, 19, 19 ), 
	legend = c( "50g", "100g", "150g", "d", "Wet" ) );

param_summary = data.frame();

for ( damp in c( "d", "w" ) ) { for ( mass in c( 50, 100, 150 ) ) {
			for ( distance in c( 30, 50, 70, 90 ) ) {

	cparams <- params[params$dampness == damp & params$mass == mass &
		params$distance == distance,];

	param_summary <- rbind( param_summary, data.frame( 
		damp = damp, 
		mass = mass, 
		distance = distance, 
		ERs = mean( cparams$Rs ), 
		sRs = sqrt( mean( cparams$Rs ^ 2 ) - mean( cparams$Rs ) ^ 2 ), 
		ERp = mean( cparams$Rp ), 
		sRp = sqrt( mean( cparams$Rp ^ 2 ) - mean( cparams$Rp ) ^ 2 ), 
		ECp = mean( cparams$Cp ), 
		sCp = sqrt( mean( cparams$Cp ^ 2 ) - mean( cparams$Cp ) ^ 2 ), 
		Ealpha = mean( cparams$alpha ), 
		salpha = sqrt( 
			mean( cparams$alpha ^ 2 ) - mean( cparams$alpha ) ^ 2 ) ) );

} } }

#colnames( params ) <- c( "Dampness", "Mass / g", "distance / mm", 
	#"Repeat", "$R_{serial}$ / Ω", "$R_{parallel}$ / Ω", 
	#"$C_{parallel}$ / F", "$ \\alpha $" );

#write.csv( params, "output/tissue.csv", row.names = FALSE );

colnames( param_summary ) <- c( "Dampness", "Mass / g", "distance / mm", 
	"$ \\overline{R_{s}} $ / Ω", "$ \\sigma(R_{s}) $ / Ω", 
	"$ \\overline{R_{p}} $ / Ω", "$ \\sigma(R_{p}) $ / Ω", 
	"$ \\overline{C_{p}} $ / F", "$ \\sigma(C_{p}) $ / F", 
	"$ \\overline{\\alpha} $", "$ \\sigma(\\alpha) $" );

write.csv( param_summary, "data/tissue.csv", row.names = FALSE );

warnings();
