#!/usr/bin/env Rscript

colours <- c( "#007293", "#8A0065", "#00811C", "#c3b200" );

file <- read.csv( "data/comsol.csv", header = FALSE );
colnames( file ) <- c( "distance", "mh", "hh", "dh", "eh", "sh",
	"freq", "imp" );

cairo_pdf( "graphs/comsol.pdf", width = 3, height = 3 );
par( xaxs = "i", bty = "l", cex = 0.5, mgp = c( 2, 1, 0 ) );

data <- file[file$mh == 25e-3 & file$hh == 5e-3 & file$dh == 1.7e-3 & 
	file$eh == 100e-6 & file$sh == 40e-6,];

layout( matrix( c( 1, 2 ), 2, 1 ), TRUE );

par( mar = c( 0, 3, 3, 1 ) + 0.3, cex = 0.5 );

plot( data$freq, Mod( data$imp ), 
	type = "n", log = "xy", axes = FALSE,
	xlab = "", ylab = "Impedance / Ω", 
	main = "Simulated Impedance" );

box();
axis( side = 2 );

for ( i in seq( 4, 1 ) ) {

	lines( data$freq[data$distance == 0.01 * c( 3, 5, 7, 9)[i]],
		Mod( data$imp[data$distance == 0.01 * c( 3, 5, 7, 9)[i]] ),
		col = colours[i] );

}

legend( x = "topright", lty = "solid", title = "Distance", 
	col = colours,
	legend = paste0( as.character( c( 3, 5, 7, 9 ) ), "0mm" ) );

par( mar = c( 4, 3, 0, 1 ) + 0.3 );

plot( data$freq, Arg( data$imp ) * 180 / pi, 
	type = "n", log = "x", axes = TRUE,
	xlab = "Frequency / Hz", ylab = "Phase / °" );

for ( i in seq( 4, 1 ) ) {

	lines( data$freq[data$distance == 0.01 * c( 3, 5, 7, 9)[i]],
		Arg( data$imp[data$distance == 0.01 * c( 3, 5, 7, 9)[i]] ) 
			* 180 / pi,
		col = colours[i] );

}

cairo_pdf( "graphs/comsol_thickness.pdf", width = 6, height = 4.5 );
data <- file[file$distance == 0.05,];

layout( matrix( c( 1, 2, 0, 0, 1, 2, 7, 8, 3, 4, 7, 8, 
		3, 4, 9, 10, 5, 6, 9, 10, 5, 6, 0, 0 ), 4, 6 ),
	TRUE );

thickness_plot <- function( d1, d2, d3, title, legend_text ) {

	par( mar = c( 0, 3, 2, 0 ) + 0.3, cex = 0.5, mgp = c( 2, 1, 0 ),
		xaxs = "i", bty = "l" )

	plot( c( d1$freq, d2$freq, d3$freq ), 
		Mod( c( d1$imp, d2$imp, d3$imp ) ),
		type = "n", log = "xy", axes = FALSE,
		xlab = "", ylab = "Impedance / Ω",
		main = title );

	lines( d1$freq, Mod( d1$imp ), col = colours[1] );
	lines( d2$freq, Mod( d2$imp ), col = colours[2] );
	lines( d3$freq, Mod( d3$imp ), col = colours[3] );

	box();
	axis( side = 2 );

	legend( x = "topright", lty = "solid", title = "Thickness",
		col = colours, legend = legend_text );

	par( mar = c( 3, 3, 0, 0 ) + 0.3, cex = 0.5 );

	plot( c( d1$freq, d2$freq, d3$freq ),
		Arg( c( d1$imp, d2$imp, d3$imp ) ) * 180 / pi,
		type = "n", log = "x",
		xlab = "Frequency / Hz", ylab = "Phase / °" );

	lines( d1$freq, Arg( d1$imp ) * 180 / pi, col = colours[1] );
	lines( d2$freq, Arg( d2$imp ) * 180 / pi, col = colours[2] );
	lines( d3$freq, Arg( d3$imp ) * 180 / pi, col = colours[3] );

}

base_data <- data[data$mh == 25e-3 & data$hh == 5e-3 &
	data$dh == 1.7e-3 & data$eh == 100e-6 & data$sh == 40e-6,];

thickness_plot(
	data[data$sh == 36e-6,], base_data, data[data$sh == 44e-6,],
	"Stratum Corneum", c( "36µm", "40µm", "44µm" ) );
thickness_plot(
	data[data$eh == 90e-6,], base_data, data[data$eh == 110e-6,],
	"Epidermis", c( "90µm", "100µm", "110µm" ) );
thickness_plot(
	data[data$dh == 1.53e-3,], base_data, data[data$dh == 1.87e-3,],
	"Dermis", c( "1.53mm", "1.7mm", "1.87mm" ) );
thickness_plot(
	data[data$hh == 4.5e-3,], base_data, data[data$hh == 5.5e-3,],
	"Hypodermis", c( "4.5mm", "5mm", "5.5mm" ) );
thickness_plot(
	data[data$mh == 22.5e-3,], base_data, data[data$mh == 27.5e-3,],
	"Muscle", c( "22.5mm", "25mm", "27.5mm" ) );
