# PhytoCap
### Project for open-source automation of low-cost imaging flow cytometry for Phytoplankton imaging, using Jetson, Python and Arduino

_Code for control of **Gilson MicroPuls3 Peristaltic Pump**, **Rheodyne EV750-107 Microfluidic Valve Selector**,  and **FLIR/Point Grey Research USB3 Vision Camera** in an interactive environment, or using a control script_

## Description/Instructions:
This repository is designed to allow autonomous operation of the Plankton Imaging Flow Cytometer (IFC) described in [paper] using simple Python functions on an NVIDIA Jetson platform. Please see the INSTALL.md for installation instructions before beginning.

**Links to Sections:**  
[0. Control Mode](#0-gui-or-direct-control)  
[1. Imports](#1-imports)  
[2. Config](#2-config-settings-for-ultrasound-gpio-pin-connections-analysis-functions)  
[3a. Acoustic Focussing](#3a-ultrasound-focussing-control)  
[3b. Fluidics](#3b-fluidic-control-pump-and-valve-selector)  
[4. Camera ](#4-camera)  
  
### 0. Control Mode (GUI, Interactive Interpreter or Script)
You must decide whether you want to control the IFC using a simplified Graphical User Interface (GUI) (by running `python3 gui.py`) or directly via code (explained below). Before making this decision it is useful to have an understanding of the functions implemented, which are designed to allow very simple control of the system's basic components.

If you decide to use direct control of the code you can either do this from within an interactive Python environment (i.e. by typing `python3` into a terminal and writing commands line by line) or by using an instruction script (example given in `example_instructions.py` and executed by typing `python3 example_instructions.py` into a terminal). The latter allows autonomous operation over potentially long times. 

**The main functionality implemented is described in sections 1 to 4 and GUI use is explained in section 5 below.**


### 1. Imports
The following files need to be imported for full functionality:
```python
import camera
import time
import gpio
import config
```

### 2. Config (Settings for Ultrasound, GPIO pin connections, Analysis functions)
`config.py` stores settings for the following properites:
- GPIO (General Purpose Input Output) pin numbers (Variables starting `GPIO_`). These are the pins on the Jetson which are connection to the acoustic DDS and (if available) peristaltic pump and/or valve selector. Pin numbers are printed on the Jetson.
- Ultrasound Sweep Start Frequency (`DDS_SWEEP_START`)
- Ultrasound Sweep Stop Frequency (`DDS_SWEEP_STOP`)
- Ultrasound Sweep Duration (in seconds) (`DDS_SWEEP_TIME`)
- Variables used in the analysis functions (Expanded on below in section 6.)
Changing these variables is as simple as reassigning them in your script or interactive session:

i.e. to control the frequency and timings used for the ultrasound sweep which focusses plankton, edit the parameters `config.DDS_SWEEP_START`, `config.DDS_SWEEP_STOP` and `config.DDS_SWEEP_TIME`.
e.g.: 
```python 
config.DDS_SWEEP_START = 1500000	# Sets the sweep start frequency to 1.5MHz
config.DDS_SWEEP_START = 1800000	# Sets the sweep stop frequency to 1.7MHz
config.DDS_SWEEP_TIME = 0.5		# Sets the sweep duration to 0.5 seconds
```



### 3. GPIO (Functions for control of ultrasonic wave focussing, peristaltic pump and fluidic valve selector)
`gpio.py` stores functions used to control the following external hardware:

#### 3a. Ultrasound Focussing Control
Generation and amplification of the acoustic particle focussing signal is handled by the DDS and Amplifier board connected to the Jetson GPIO pins. In order to toggle the usw (UltraSonic Wave), use the functions [an object is created to allow a background process to control the USW and be later disabled as required]:
```python
t = gpio.usw_on()
``` 
and 
```python
gpio.usw_off(t)
``` 


#### 3b. Fluidic Control (pump and valve selector)
For use in a lab setting, particularly with live culures, the software can control a peristaltic pump and fluidic valve actuator which are connected, through a transistor array, to the GPIO pins of the Jetson. This allows interactive or preprogrammed selection and pumping of different samples through the flow cell on command.

The software & electronics assume your pump has direction and enable inputs which are activated by connecting them to the ground input on the pump. Similarly your valve selector is assumed to have a binary control input which is set by grounding the binary representation of the valve to select (e.g. to set to valve 3, pins 1 and 2 would be grounded and the other pins left floating)

Peristaltic pumps (e.g. Gilson MiniPULS3) with enable and direction (forward/reverse) pins controlled by the Jetson's GPIO can be set calling the function:
```python
gpio.pumpenable(enable={True/False})
gpio.pumpdirection(direction={True/False})
```

Similarly, mechanical valve selectors with up to 4 binary selection inputs (e.g. Rheodyne EV-750) are set using the function:
```python
gpio.valve(number)
```


### 4. Camera
`camera.py` implements the class `cameraclass` to simplify much of the typical camera operation.

_The code assumes you are using a FLIR/Point Grey USB3.1 camera. Other camera will require modification of camera.py_

To use the camera, your script must first create a camera object as below:
```python
cam = camera.cameraclass()
``` 
After this, you can use the `init()` function to connect the _cam_ object to the camera and load the default settings:
```python
cam.init()
``` 
In an interactive Python environment you should see message confirming that the camera successfully connected after running this line.

_Once the camera has been initialised you can change the following settings:_
- Image Format _{default : .jpg}_: e.g. `cam.imageformat = '.bmp'`
- Save directory _{default: /captures/}_ e.g. `cam.setsavedir('/captures/test/')`
- Red/Blue balance _{default: Red = 1.9, Blue = 1.5}_ e.g. `cam.setredbluebalance(red=2.0, blue=1.4)`
- Camera analogue gain (dB) _{default: 19}_ e.g. `cam.setgain(15)`
- Sensor Pixel Format _{default: RGB8}_ e.g. `cam.setpixelformat('Mono12')`

_Capturing images from the camera can be done in several ways:_
- To display a live video feed without saving the images use `cam.liveview(FPS)`, where FPS is an optional argument for the display frames per second (will use the maximum possible FPS if not specified).
- To return a single image frame use `image = cam.grabimage()`
- To save n images to disk, use `cam.capture_n(n, FPS)`, where FPS is an optional argument for the capture frames per second (will use the maximum possible FPS if not specified).
- To save images to disk at maximum FPS for a specified time _s_, use `cam.capture_secs(s)`. s can be any positive real decimal or integer.
- To capture a background image (saves a single frame to /captures/background/ for later use in image analysis), use `cam.capture_backgroundimage()`

**Important Points:** 
- To stop any running capture process early (either `liveview` or `capture_n/secs`), you can call `cam.abortcapture()`.
- All images are saved with the filename *DD-MM-YYYY-HH_MM_SS*


### 5. Graphical User Interface

TODO:GUI




**_Misc:_**


#### Anthony's todo list: 
	
- [x] ~~Upload arduino code~~
- [x] ~~Implement USW switching via trigger pin (prior to eventual swap to DDS)~~ ~~_11/20 now uses USBTMC instead_~~ _12/20 now uses DDS_
- [x] ~~Add control of USW to python code (command will be 31 for USW on, 30 for off)~~ _11/20 depreciated_
- [x] ~~Separate into main script and fluidic/camera modules~~ _10/20 done_
- [x] ~~Implement looped camera capture~~ _11/20 done_
- [x] ~~Update controlfile parser to ignore #comments, whether whole-line or after a command~~ _12/20 depreciated due to use of python script instead of text file_
- [x] ~~Code for capture an image every x seconds for y time~~ _12/20 camera method now allows defined FPS_
- [x] ~~Pass the last v instruction to camera functions so images are saved in captures/v_n/img.bmp - to separate images into folders by the cultures they are imaged from~~ _12/20 removed - set default folder and this wasn't that useful_
- [x] ~~Implement looped instructions; i.e. l5, v1, p1f, cn10, le to loop between l5 and le 5 times~~ _12/20 depreciated due to use of python script instead of text file_
- [x] ~~Implement saved sweep config to function generator (settable params?)~~ _12/20 depreciated but DDS equivalent now in gpio.py_
- [x] ~~Write code to use DDS controlled by Jetson's GPIO~~ _12/20 done in gpio.py_
- [ ] ~~Add switch/flag for main.py to allow swapping between DDS and function generator~~ _12/20 depreciated function generator as DDS sufficient_
- [x] ~~Add parallelisation to control DDS and other functionality simultaneously~~ _12/20 done_
- [x] ~~Write code to use Jetson GPIO to control pump and valve selector~~
- [ ] Improve commenting throughout all code _12/20 half done_
- [x] **Code GUI**
- [ ] **Fix cam setgain(), sort out the blocking function/s**
- [x] **Code 'USW tester' - a 6kHz- 10kHz sweep to check connection**
- [ ] Clean up logging calls
- [ ] Add to arduino flash control to allow doubleflash toggle
- [ ] Implement arduino pump speed control (analog)

