#__author__ = "Chris Maidens"
#__copyright__ = "Copyright (C) 2020 Chris Maidens"
#__license__ = "No license granted"
#__version__ = "0.1"

#  https://github.com/mitre-attack/attack-datasources-stix-beta
# https://buildmedia.readthedocs.org/media/pdf/stix2/latest/stix2.pdf
# https://docs.oasis-open.org/cti/stix/v2.1/os/stix-v2.1-os.pdf
#       https://stix2.readthedocs.io/en/latest/guide/custom.html (to understand x-mitre-data-component etc)
#

# pip install taxii2-client
from taxii2client.v20 import Collection
# pip install stix2
from stix2 import CompositeDataSource, FileSystemSource, \
    TAXIICollectionSource, Filter, MemorySource,   FileSystemSink

# https://taxii2client.readthedocs.io/en/latest/
from taxii2client.v20 import Server
# https://2.python-requests.org/en/master/user/quickstart/
# pip install requests
import requests

# Used for basic date utilities
import datetime

import pandas as pd

import os

# import packages
# pip install pypdf2
import PyPDF2
import re
#
#import urllib

import random

import sys

import json

import networkx as nx

#import time

# the lib that handles the url stuff

#from os import listdir
#from os.path import isfile, join
# Used to load and process NVD input files
# Have a look here as well https://github.com/CVEProject/cvelist



#################################
# NVD Database class and methods
class MAFpt_ATTACK_DB():
    """ MAFpt_ATTACK_DB: <<Object for ATTACK Data and utility methods>>
        ______________________________________________
        
        MAFpt_ATTACK_DB.__init__( 
        ________________________
                                self,
                                reload [string - Y or not Y],
                                reindex [string - Y or not Y],
                                src_dir [string - URL for filelist],
                                src_file_re [string - Reg exp for filenames in src_dir webpage],
                                src_dload_dir [string - URL for dowload site],
                                loc_dir [string - Directory to store files on local machine],
                                index_fname [string - Name of file to store index])
                                
            if reload = "Y": Reload files named in webpage (URL src_dir) webpage to local
                                    directory (loc_dir)
                                Filenames are found by looking for src_file_re in webpage
                                    and downloaded from src_dload_dir
                                Will not reload if files already present.
                                
            if reindex = "Y": Build an index to CVEs in files loaded to locdir
                            will only index on file names that match src_file_re
                            Index written to loc_dir/NVD_INDEX_FILE.csv 
                                    
                            Used in object methods
                                                                                   
                            pd.DataFrame(columns=['CVE_ID',
                                'CWE_ID',  ( ['problemtype_data'][0]['description'][0]['value'] )
                                        ( ID | NVD-CWE-Other | NVD-CWE-noinfo | CWE_MISSING)
                                'impact', ( ['baseMetricV2']['cvssV2']['baseScore'] | UNKNOWN )
                                'pub_date', ( TBC )
                                'mod_date', ( TBC )
                                'status', (  LIVE | 
                                            REJECT if ['description_data'][0]['value'] starts "** REJECT **")
                                'fname' ( source filname ) ])
                                                                    
            else:
                            read in existing index from loc_dir
                                        
        MAFpt_NVD_DB.MAFpt_NVD_Summary(
        ____________________________________
                                            self,
                                            type [string - F])
                                            
            Return pandas dataframe containing summary counts for the files.
                
                pd.DataFrame(columns=["Item": For each item,
                     "Total":Total for this item,
                     "CWE": Total with CWE (but reference not checked),
                     "CWE_Other": Total with CWE  ""NVD-CWE-Other"
                     "CWE_Missing": Total with no CWE data (and not marked as rejected)
                     "Rejected": Total with status REJECT
                     "NoInfo": Total with CWE ""NVD-CWE-noinfo")
                                     
            if type = "F": Counts are prepared by file
 
        MAFpt_NVD_DB.MAFpt_NVD_list_CVE(
        _______________________________
                                            self,
                                            type, [string - CWE | Impact | pub_date | mod_date | status]
                                            t_val, [string - Value in type to search for]
                                            fname [string - CVE filename])
                                            
            Return pandas dataframe column containing list of CVE meeting criteria.
  
        MAFpt_NVD_DB.MAFpt_NVD_list_CWE(
        _______________________________
                                            self,
                                            type, [string - fname]
                                            t_val, [string - Value in type to search for])
                                             
            Only supports type 'fname'.  
            Does not count CWE_MISSING.
            Returns a dataframe with a count for each CWE type in this file.
            
            pd.DataFrame(columns=['Item' (filename),
                        'CWE',  ( ID | NVD-CWE-Other | NVD-CWE-noinfo)
                        'Count', ( int - count ) ] )
                        
        MAFpt_NVD_DB.MAFpt_NVD_output_list_CWE(
        _______________________________
                                            self,
                                            loc_dir [string - Directory to store files on local machine],
                                            count_file [string - Filename for count file])
                                             
            Writes a dataframe from MAFpt_NVD_list_CWE() for each file 
                in the main index built in __init__ .
            CSV written to loc_dir.
            
        Requires:
            # pip install requests
            # pip install pandas
            # pip install psutil
            # re, os, zipfile, json, datetime from python standard libraries """            
                                             
    # Build an empty dataframe to be populated and written out as the index
    ATTACK_Index = pd.DataFrame(columns=['Group',
                                                            'GID', 
                                                            'Aliases',
                                                            'Group_Create_Date',
                                                            'Group_Modify_Date', 
                                                            'TTP_Name', 
                                                            'TTP_Type', 
                                                            'TTP_ID', 
                                                            'TTP_EXT_ID', 
                                                            'TTP_Phase', 
                                                            'Revoked', 
                                                            'ISet_ID', 
                                                            'REL_Date_Created', 
                                                            'REL_ID', 
                                                            'TTP_Date_Created'
                                                            ])
                                                      
    ATTACK_Sub_Index = pd.DataFrame(columns=['GID',
                                                            'TTP_EXT_ID',
                                                            'TTP_ID_SUB', 
                                                            'TTP_EXT_ID_SUB', 
                                                            'REL_ID', 
                                                            'REL_Date_Created'                                                        
                                                            ])
                                                        
    ATTACK_TOOL_TTP_Index = pd.DataFrame(columns=[
                                                            'TOOL_NAME', 
                                                            'TOOL_EXT_ID',
                                                            'TTP_ID', 
                                                            'TTP_EXT_ID', 
                                                            'REL_ID', 
                                                            'REL_Date_Created'                                                        
                                                            ])
                                                            
                                                            
#   Dataframes to be stored as CSVs
#
#  ATTACK_META_MODEL_Index 
#  ATTACK_GROUP_Index
# ATTACK_GROUP_Aliases_Index
# ATTACK_TTP_Index
# ATTACK_TTP_TO_PLATFORM_Index
# ATTACK_TTP_TO_PERMS_REQ_Index
# ATTACK_TTP_TO_PROCEDURE_Index
# ATTACK_TTP_TO_DATA_SOURCE_Index
# ATTACK_REL_Index
# ATTACK_CVE_REF_Index
# ATTACK_TTP_REF_Index
# ATTACK_REL_REF_Index
# ATTACK_MIT_REF_Index
# ATTACK_TACTIC_Index
# ATTACK_TECH_TO_TACTIC_Index
# ATTACK_META_MODEL_Index
# ATTACK_MITIGATION_Index
# ATTACK_TTP_TO_DATA_COMP_Index
# ATTACK_CAR_COVERAGE_Index
# ATTACK_DATA_SOURCE_Index
# ATTACK_DATA_SOURCE_REF_Index
# ATTACK_DATA_SOURCE_TO_PLATFORM_Index
# ATTACK_DATA_SOURCE_TO_COLL_LAYER_Index
#
#
#  ATTACK meta model
    ATTACK_META_MODEL_Index = pd.DataFrame(columns=['Collection_Name',
                                                            'Collection_URL', 
                                                            'Collection_ID',
                                                            'Source_Type', 
                                                            'Target_Type', 
                                                            'Rel_Type'
                                                            ])

#  List of Groups.
    ATTACK_GROUP_Index = pd.DataFrame(columns=['Group',
                                                            'GID', 
                                                            'Aliases',
                                                            'Description', 
                                                            'Attribution', 
                                                            'TCERT_Attribution', 
                                                            'Group_Create_Date',
                                                            'Group_Modify_Date',  
                                                            'ISet_ID', 
                                                            'Revoked'
                                                            ])

#  List of Groups aliases
    ATTACK_GROUP_Aliases_Index = pd.DataFrame(columns=['Group',
                                                            'GID', 
                                                            'Alias'
                                                            ])


#  List of Techniques, Tools and Malware
#     This also inlcudes the new sub techniques
    ATTACK_TTP_Index = pd.DataFrame(columns=[
                                                            'TTP_Name', 
                                                            'TTP_Type', 
                                                            'TTP_ID', 
                                                            'TTP_EXT_ID',
                                                            'TTP_URL', 
                                                            'TTP_DESC', 
                                                            'TTP_Phase', 
                                                            'TTP_Domain', 
                                                            'TTP_x_mitre_detection',  
                                                            'TTP_Date_Created',
                                                            'TTP_Date_Modified',
                                                            'TTP_x_mitre_deprecated',
                                                            'TTP_x_mitre_is_subtechnique', 
                                                            'Revoked', 
                                                            'TTP_CAR_Available'
                                                            ])
                                                            
#  Linking techniques to platforms
#     
    ATTACK_TTP_TO_PLATFORM_Index = pd.DataFrame(columns=[
                                                                     "TTP_EXT_ID", 
                                                                     "Platform"
                                                                     ])
                                                                     
#  Linking techniques to permissions
#     
    ATTACK_TTP_TO_PERMS_REQ_Index = pd.DataFrame(columns=[
                                                                     "TTP_EXT_ID", 
                                                                     "Permission"
                                                                     ])
                                                                     
#  Linking techniques to examples of APT procedures
#     
    ATTACK_TTP_TO_PROCEDURE_Index = pd.DataFrame(columns=[
                                                                     "TTP_EXT_ID", 
                                                                     "DATA_SOURCE"
                                                                     ])
                                                                     
#  Linking techniques to examples of APT procedures
#     
    ATTACK_TTP_TO_DATA_SOURCE_Index = pd.DataFrame(columns=[
                                                                     "TTP_EXT_ID", 
                                                                     "Procedure Desc", 
                                                                     "USER_EXT_ID", 
                                                                     "USER_TYPE"
                                                                     ])
                                                                     
#  Relationship List
#     
    ATTACK_REL_Index = pd.DataFrame(columns=[
                                                            'REL_SOURCE_ID', 
                                                            'REL_SOURCE_TYPE', 
                                                            'REL_SOURCE_NAME', 
                                                            'REL_SOURCE_EXT_ID', 
                                                            'REL_TARGET_ID', 
                                                            'REL_TARGET_TYPE', 
                                                            'REL_TARGET_NAME', 
                                                            'REL_TARGET_EXT_ID', 
                                                            'REL_TYPE', 
                                                            'REL_ID', 
                                                            'REL_DESC', 
                                                            'REL_Date_Created', 
                                                            'REL_Date_Modified', 
                                                            'Revoked'
                                                            ])

#  References associated with groups
#                                                
    ATTACK_CVE_REF_Index = pd.DataFrame(columns=['GID',
                                                            'Group', 
                                                            'source_ref',
                                                            "description", 
                                                            "source_name", 
                                                            'CVE_List'
                                                            ])
                                                            
#  References associated wiht TTPs
#                                                            
    ATTACK_TTP_REF_Index = pd.DataFrame(columns=['TTP_EXT_ID',
                                                            'source_name', 
                                                            'url',
                                                            'external_id', 
                                                            'description'
                                                            ])
#  References associated with relationships
#                                                            
    ATTACK_REL_REF_Index = pd.DataFrame(columns=['REL_ID',
                                                            'source_name', 
                                                            'url',
                                                            'external_id', 
                                                            'description'
                                                            ])
#  References associated with mitigations
#                                                            
    ATTACK_MIT_REF_Index = pd.DataFrame(columns=['MIT_EXT_ID',
                                                            'source_name', 
                                                            'url',
                                                            'external_id', 
                                                            'description'
                                                            ])
#  Tactic Index
#                                                              
    ATTACK_TACTIC_Index = pd.DataFrame(columns=[
                                                            'TACTIC_EXT_ID', 
                                                            'TACTIC_ID', 
                                                            'TACTIC_NAME', 
                                                            'TACTIC_DESC', 
                                                            'TACTIC_CREATED', 
                                                            'TACTIC_MODIFIED', 
                                                            'TACTIC_MATRIX', 
                                                            'TACTIC_MATRIX_URL', 
                                                            'LM_KILL_CHAIN_PHASE'
                                                            ])
#  Tech To Tactic Index
#                                                                                                                          
    ATTACK_TECH_TO_TACTIC_Index = pd.DataFrame(columns=[
                                                            'TTP_NAME', 
                                                            'TTP_EXT_ID', 
                                                            'KILL_CHAIN_NAME', 
                                                            'TACTIC_NAME', 
                                                            'TACTIC_EXT_ID', 
                                                            'REL_Date_Created'
                                                            ])
#  Tech To Tactic Index
#                                                                                                                          
    ATTACK_META_MODEL_Index = pd.DataFrame(columns=[
                                                            'REL_SOURCE_TYPE', 
                                                            'REL_SOURCE_SUBTYPE', 
                                                            'REL_TARGET_TYPE', 
                                                            'REL_TARGET_SUBTYPE', 
                                                            'RE_TYPE'
                                                            ])
#  Mitigation Index
#                                                                                                                          
    ATTACK_MITIGATION_Index = pd.DataFrame(columns=[
                                                            'TTP_EXT_ID', 
                                                            'COA_EXT_ID', 
                                                            'COA_ID', 
                                                            'COA_NAME', 
                                                            'COA_REL_DESC', 
                                                            'COA_DESC', 
                                                            'REL_ID',
                                                            'created', 
                                                            'modified'                                                            
                                                            ])
#  TTP to Data Component Index
#                                                                                                                          
    ATTACK_TTP_TO_DATA_COMP_Index = pd.DataFrame(columns=[
                                                            'TTP_EXT_ID', 
                                                            'DC_EXT_ID', 
                                                            'DC_ID', 
                                                            'DC_NAME', 
                                                            'DC_REL_DESC', 
                                                            'DC_DESC', 
                                                            'REL_ID',
                                                            'DC_created', 
                                                            'DC_modified', 
                                                            'DS_ID' , 
                                                            'DS_EXT_ID', 
                                                            'DS_NAME', 
                                                            'DS_DESC', 
                                                            'DS_created', 
                                                            'DS_modified'
                                                            ])
# CAR coverage index (downloaded from MITRE)
    ATTACK_CAR_COVERAGE_Index = pd.DataFrame(columns =[
                                                                             'Technique', 
                                                                             'Name', 
                                                                             'Sub_Name', 
                                                                             'CAR_Num', 
                                                                             'Sigma_Num', 
                                                                             'ES_Num', 
                                                                             'Splunk_Num', 
                                                                             'Tot_Num'])
                                                                             
#  Data Source Index
#                                                                                                                          
    ATTACK_DATA_SOURCE_Index = pd.DataFrame(columns=[
                                                            'DS_ID' , 
                                                            'DS_EXT_ID', 
                                                            'DS_NAME', 
                                                            'DS_DESC', 
                                                            'DS_created', 
                                                            'DS_modified'
                                                            ])
                                                            
#  References associated with Data Sources
#                                                
    ATTACK_DATA_SOURCE_REF_Index = pd.DataFrame(columns=['DS_EXT_ID',
                                                            'DS_NAME', 
                                                            'source_ref',
                                                            "description", 
                                                            "source_name"
                                                            ])

#  Data Source To Platform Index
#                                                                                                                          
    ATTACK_DATA_SOURCE_TO_PLATFORM_Index = pd.DataFrame(columns=[
                                                            'DS_ID' , 
                                                            'DS_EXT_ID', 
                                                            'DS_NAME', 
                                                            'DS_PLATFORM'
                                                            ])
                                                                             
#  Data Source To Collection Layer Index
#                                                                                                                          
    ATTACK_DATA_SOURCE_TO_COLL_LAYER_Index = pd.DataFrame(columns=[
                                                            'DS_ID' , 
                                                            'DS_EXT_ID', 
                                                            'DS_NAME', 
                                                            'DS_COLL_LAYER'
                                                            ])
                                                                             
#  ATTACK_TACTIC_BIN_INDEX    -    rows=GroupNames,  columns=Tactics
#                                                         Constructed in CreateBinIndexes()
    ATTACK_TACTIC_BIN_Index = 0
    ATTACK_TACTIC_ENT_BIN_Index = 0
#  ATTACK_TTP_BIN_INDEX          -    rows=GroupNames,  columns=TTP
#                                                         Constructed in CreateBinIndexes()
    ATTACK_TTP_BIN_Index = 0    
#  ATTACK_TTP_REF_BIN_INDEX          -    rows=Reference_URL  columns=Text,[Tactics],[TTP]
#                                                         Constructed in CreateBinIndexes()
    ATTACK_TTP_REF_BIN_Index = 0          

    Verbose=False
    
    TCERT_group_data=""
    TCERT_tool_data=""

    # Constructor
    def __init__ (self, 
                        run_download_attack, 
                        taxii_server, 
                        attack_local_file_root, 
                        Create_ATTACK_Local_Copy, 
                        reindex_attack, 
                        CVE_SEARCH, 
                        ATTACK_MAIN_INDEX, 
                        ATTACK_SUB_INDEX, 
                        ATTACK_CVE_REF_INDEX, 
                        ATTACK_TTP_INDEX, 
                        ATTACK_TACTIC_INDEX, 
                        ATTACK_TECH_TO_TACTIC_INDEX, 
                        ATTACK_REL_INDEX, 
                        ATTACK_TACTIC_BIN_INDEX, 
                        ATTACK_TTP_BIN_INDEX):
                            
        # "Init function declarations end here" to get to end
    
        def ConnectToTAXIIServerCollections(taxii_server):
                                    
          print(str(datetime.datetime.now().time()) + "Instantiating TAXII server " + taxii_server)
          server = Server(taxii_server)
          api_root = server.api_roots[0]
          print(str(datetime.datetime.now().time()) + "Server API roots retrieved")
          # Print name and ID of all ATT&CK technology-domains available as collections
  
          taxii_collection=""
          taxii_root_rows=[]

          # https://stix2.readthedocs.io/en/latest/guide/taxii.html
          # https://docs.oasis-open.org/cti/taxii/v2.1/cs01/taxii-v2.1-cs01.pdf
          print("The available collections are as follows: ")
          
          for collection in api_root.collections:
            #pdf.write(5, collection.title + ": " + collection.id + "\n")
            print(collection.title + ": " + collection.url)
    
            taxii_root_dict={'col_title':"", 
                  'col_id':"", 
                  'col_url': "", 
                  'tc_source':""}

            taxii_collection=Collection(collection.url)
            #tc_source.append(TAXIICollectionSource(taxii_collection))
    
            taxii_root_dict.update({'col_title': collection.title,  
                                            'col_id': collection.id, 
                                            'col_url': collection.url, 
                                            'tc_source': TAXIICollectionSource(taxii_collection)})
                                    
            taxii_root_rows.append(taxii_root_dict)
            print(str(datetime.datetime.now().time()) + "Established connection to Collection " + collection.id)


          # Establish TAXII2 Collection instance for Enterprise ATT&CK collection
          # Enterprise ATT&CK: 95ecc380-afe9-11e4-9b6c-751b66dd541e
          #PRE-ATT&CK: 062767bd-02d2-4b72-84ba-56caef0f8658
          #Mobile ATT&CK: 2f669986-b40b-4423-b720-4396ca6a462b
          #collection = Collection("https://cti-taxii.mitre.org/stix/collections/95ecc380-afe9-11e4-9b6c-751b66dd541e/")
          #collection2 = Collection("https://cti-taxii.mitre.org/stix/collections/062767bd-02d2-4b72-84ba-56caef0f8658/")
          #collection3 = Collection("https://cti-taxii.mitre.org/stix/collections/2f669986-b40b-4423-b720-4396ca6a462b/")
 
          return(taxii_root_rows)
    
        # Assume that ATTACK_Index and (MemorySource) cs are available as self globals
        # Level will be "SIMPLE" or "Full"
        def GetGroupTTP(GName):

            print("Getting TTP for group " + str(GName))
    
            GIDSet = self.ATTACK_Index[self.ATTACK_Index['Group'] == GName]['GID'] 
    
            for ThisItem in GIDSet:
                GID=str(ThisItem)
    
            TTPSet = self.ATTACK_Index[self.ATTACK_Index['GID'] == GID]
        
            print("Returning the TTP Set")
    
            return(TTPSet)
            
         # "STEP" to get elements in here
        def BaseDataToMemory(filter_objs_1, 
                                            filter_objs_2,
                                            filter_objs_3,  
                                            attack_1, 
                                            attack_2, 
                                            attack_3, 
                                            cs1, 
                                            cs2, 
                                            cs3):
                        
            
            print("<< Getting base data from TAXII server >>")
            
            # Retrieve all Enterprise and Mobile ATT&CK content
            print(str(datetime.datetime.now().time()) + "Loading Enterprise ATT&CK into memory")

            # Python will automatically call .__iter__()
            for key in filter_objs_1:
              # List of <class 'stix2.v20.sdo.AttackPattern'>[], <class 'stix2.v20.sdo.CourseOfAction'>[],
              #   <class 'stix2.v20.sdo.IntrusionSet'>[], <class 'stix2.v20.sdo.Malware'>[], <class 'stix2.v20.sdo.Tool'>[]
              # For use with the TAXII server
              #  if ATT_SRC =="TAXII":
                 attack_1[key] = cs1.query(filter_objs_1[key])
              # For use with the flat files (ATT_src == "FF")
              #  else:
            #    attack[key] = fs.query(filter_objs[key])
            
            print(str(datetime.datetime.now().time()) + "Loading Mobile ATT&CK into memory")
            
            for key in filter_objs_2:
                 attack_2[key] = cs2.query(filter_objs_2[key])
        

            # Retrieve all PRE ATT&CK content
            print(str(datetime.datetime.now().time()) + "Loading PRE ATT&CK into memory")

            for key in filter_objs_3:
              # List of <class 'stix2.v20.sdo.AttackPattern'>[], <class 'stix2.v20.sdo.CourseOfAction'>[],
              #   <class 'stix2.v20.sdo.IntrusionSet'>[], <class 'stix2.v20.sdo.Malware'>[], <class 'stix2.v20.sdo.Tool'>[]
             # For use with the TAXII server
              #  if ATT_SRC =="TAXII":
                 attack_3[key] = cs3.query(filter_objs_3[key])
              # For use with the flat files (ATT_src == "FF")
              #  else:
              #    attack_2[key] = fs.query(filter_objs[key])

            print(str(datetime.datetime.now().time()) + "Loaded all data to memory")

        def TechToDomainList(filter_objs_1, 
                                          cs1, 
                                          cs2, 
                                          cs3): 
            TechListOfLists=[]
            ThisList1=[]
            ThisList2=[]
            ThisList3=[]
            
            # Retrieve all  Enterprise Techniques
            TTP_list=cs1.query(filter_objs_1['techniques'])
                                     
            for TTPNext in TTP_list:                 
                ThisList1.append(TTPNext.external_references[0]['external_id']) 
                
            TechListOfLists.append(ThisList1)
      
            
            # Retrieve all  Mobile Techniques
            TTP_list=cs2.query(filter_objs_1['techniques'])
                                     
            for TTPNext in TTP_list:                 
                ThisList2.append(TTPNext.external_references[0]['external_id']) 
                
            TechListOfLists.append(ThisList2)
            
            # Retrieve all  Pre ATT&CK Techniques
            TTP_list=cs3.query(filter_objs_1['techniques'])
                                     
            for TTPNext in TTP_list:                 
                ThisList3.append(TTPNext.external_references[0]['external_id']) 
                
            TechListOfLists.append(ThisList3)
            
            #for TechList in TechListOfLists:
                #print("<< This technique list >> " + str(TechList))
                
            #
            # Check for duplicates
            #
            for ThisTTP1 in ThisList1:
                for ThisTTP2 in ThisList2:
                    if ThisTTP1 == ThisTTP2:
                        print("<<WARNING Duplicate Tech in Enterprise and Mobile " + ThisTTP1)
                        
                for ThisTTP3 in ThisList3:
                    if ThisTTP1 == ThisTTP3:
                        print("<<WARNING Duplicate Tech in Enterprise and Pre-ATT&CK " + ThisTTP1)
                        
            for ThisTTP2 in ThisList2:
                for ThisTTP3 in ThisList3:
                    if ThisTTP2 == ThisTTP3:
                        print("<<WARNING Duplicate Tech in Enterprise and Mobile " + ThisTTP2)
            
            ThisList1=[]
            ThisList2=[]
            ThisList3=[]
            
            return(TechListOfLists)
            
        def BaseDataToTables(cs, 
                                          TechListOfLists, 
                                          ind_fname_GROUPS,
                                          ind_fname_tactics, 
                                          ind_fname_TTP, 
                                          ind_fname_tech_to_tactic, 
                                          ind_fname_REL, 
                                          ind_fname_REL_REF, 
                                          ind_fname_CVE_ref, 
                                          ind_fname_TTP_REF, 
                                          ind_fname_TACTIC_BIN, 
                                          ind_fname_TTP_BIN, 
                                          ind_fname_TTP_REF_BIN, 
                                          ind_fname_META, 
                                          ind_fname_MITIGATION, 
                                          ind_fname_MIT_REF, 
                                          ind_fname_TTP_PLAT, 
                                          ind_fname_TTP_TO_PROCEDURE, 
                                          ind_fname_TTP_TO_DATA_SOURCE, 
                                          ind_fname_TTP_TO_PERMS_REQ, 
                                          ind_fname_ATTACK_ALIASES, 
                                          ind_fname_ATTACK_CAR_COVERAGE, 
                                          ind_fname_ATTACK_TTP_TO_DATA_COMP, 
                                          ind_fname_ATTACK_DATA_SOURCE, 
                                          ind_fname_ATTACK_DATA_SOURCE_REF, 
                                          ind_fname_ATTACK_DATA_SOURCE_TO_PLATFORM, 
                                          ind_fname_ATTACK_DATA_SOURCE_TO_COLL_LAYER, 
                                          ind_fname_ATTACK_TCERT_GROUP, 
                                          ind_fname_ATTACK_TCERT_TOOL) :
                                              
            #
            #  x_mitre_matrix
            #  techniques
            #  malware
            #  tools
            #
            def GetGroupAttribution(GID, 
                                                    GName, 
                                                    Description):
            #
            # Needs more work but try and extract possible state attribution
            #    from Group text
            # Up to, Winnti Group
            #
                # NoInfo
                #
                # APT18, BlackTech, Blue Mockingbird, Bouncing Golf, Carbanak
                # Cobalt Group, Darkhotel, DarkHydrus, DarkVishnya, Dragonfly
                # DragonOK, Dust Storm, Equation,  Evilnum, FIN10, FIN4, FIN6, FIN7
                # FIN8, Frankenstein, Indrik Spider, Gallmaker, 
                # Gamaredon Group, GCMAN, GOLD SOUTHFIELD
                # Honeybee
                # Inception, Leviathan (China), Lotus Blossom, Machete, Molerats, NEODYMIUM
                # Orangeworm, Patchwork, PLATINUM, Poseidon Group, PROMETHIUM, 
                #  Rancor, Rocke, RTM, Sharpshooter, Silence, Sowbug, Strider, TA505, TA551,
                #  Taidoor, The White Company, Threat Group-1314, Thrip, Tropic Trooper (China)
                #  Whitefly, Windigo, 
                #  Windshift, WIRTE
                # 
                #  Manual via references
                #  Leviathan, Tropic Trooper
                #  Ferocious Kitten
                #
                Attribution="Unknown"
                
                # Up to Tropic Trooper
                # 
                CheckList=["admin@338", "Ajax Security Team", "Andariel", "APT-C-36", 
                "APT1", "APT12", "APT16", "APT17",  
                "APT18", 
                "APT19", "APT28", "APT29", "APT3", "APT30", 
                "APT32", "APT33", "APT37", "APT38", 
                "APT39", "APT41", "Axiom", "BlackOasis", "BlackTech", "Chimera", 
                "Blue Mockingbird", "Bouncing Golf", "BRONZE BUTLER", 
                "Carbanak", "Charming Kitten", 
                "Cleaver", "Cobalt Group", "CopyKittens", "Dark Caracal", 
                "Darkhotel", "DarkHydrus", "DarkVishnya", 
                "Deep Panda" ,
                "Dragonfly", "Dragonfly 2.0",  
                "DragonOK", "Dust Storm", "Elderwood" ,"Equation", "Evilnum", 
                "Ferocious Kitten", # See Iran
                "FIN10", "FIN4", "FIN5", 
                "FIN6", "FIN7", 
                "FIN8", "Fox Kitten", "Frankenstein", "GALLIUM", 
                "Gallmaker", "Gamaredon Group", "GCMAN", "GOLD SOUTHFIELD", 
                "Gorgon Group", "Group5", "HAFNIUM", "Higaisa", 
                "Honeybee", 
                "Inception", "IndigoZebra", "Indrik Spider", 
                "Ke3chang", "Kimsuky", "Lazarus Group", 
                "Leafminer", "Leviathan", # See China
                "Lotus Blossom", "Machete", 
                "Magic Hound", "menuPass", 
                "Moafee",  "Mofang", "Molerats",  "MuddyWater", "Mustang Panda", 
                "Naikon", "NEODYMIUM", "Night Dragon", "OilRig", "Operation Wocao", 
                "Orangeworm", "Patchwork",  "PittyTiger", "PLATINUM", 
                "Poseidon Group", "PROMETHIUM", "Putter Panda", 
                "Rancor", "Rocke", "RTM", "Sandworm Team", "Scarlet Mimic", 
                "Sharpshooter", "Sidewinder", "Silence",  "Silent Librarian", 
                "SilverTerrier", "Soft Cell", "Sowbug", 
                "Stealth Falcon", "Stolen Pencil", "Strider", "Suckfly", 
                "TA459", "TA505", "TA551", 
                "Taidoor", "TeamTNT", 
                "TEMP.Veles", "The White Company", "Threat Group-1314", 
                "Threat Group-3390", "Thrip",  "Tropic Trooper",  # See China, 
                "Turla", "Volatile Cedar", 
                "Whitefly",  "Windigo",  "Windshift",  "Winnti Group",  
                "WIRTE",  "Wizard Spider", 
                "ZIRCONIUM"]
                
                if not GName in CheckList:
                    print("<< WARNING >> The group " + GName + " (" + str(GID) + "} not analysed previously")
                
                # China
                #  ,  
                # admin@338, APT16, APT17, APT3, Suckfly, Mustang Panda
                if "is a China-based" in Description:
                    Attribution = "China"
                # APT1, APT19, Putter Panda, Threat Group-3390
                if "is a Chinese" in Description:
                    Attribution = "China"  
                # APT12, 
                if "attributed to China" in Description:
                    Attribution = "China"
                # APT30, Axiom
                if "suspected to be associated with the Chinese" in Description:
                    Attribution = "China"    
                # APT41
                if "assessed as Chinese state-sponsored" in Description:
                    Attribution = "China"   
                # BRONZE BUTLER
                if "group with likely Chinese" in Description:
                    Attribution = "China" 
                # Deep Panda, Elderwood
                if "is a suspected Chinese" in Description:
                    Attribution = "China" 
                # Ke3chang, ZIRCONIUM, HAFNIUM
                if "operating out of China" in Description:
                    Attribution = "China" 
                # menuPass
                if "appears to originate from China" in Description:
                    Attribution = "China" 
                # Moafee
                if "appears to operate from the Guandong Province of China" in Description:
                    Attribution = "China" 
                # Mofang
                if "is a likely China-based" in Description:
                    Attribution = "China" 
                # Naikon
                if "attributed to the Chinese People’s Liberation Army’s (PLA)" in Description:
                    Attribution = "China" 
                # Night Dragon
                if "originating primarily in China" in Description:
                    Attribution = "China" 
                # PittyTiger, TA459
                if "believed to operate out of China" in Description:
                    Attribution = "China" 
                # Scarlet Mimic
                if "motivations appear to overlap with those of the Chinese government" in Description:
                    Attribution = "China" 
                # Soft Cell
                if "reportedly affiliated with China" in Description:
                    Attribution = "China" 
                # Winnti Group
                if "with Chinese origins" in Description:
                    Attribution = "China" 
                # Operation Wocao
                if "a China-based cyber espionage adversary" in Description:
                    Attribution = "China" 
                # Chimera
                if "is a suspected China-based" in Description:
                    Attribution = "China" 
                # GALLIUM
                if "likely a Chinese state-sponsored group" in Description:
                    Attribution = "China" 
                
                # Tropic Trooper
                # https://securityaffairs.co/wordpress/103292/apt/tropic-trooper-air-gapped-networks.html
                # https://apt.thaicert.or.th/cgi-bin/showcard.cgi?g=Tropic%20Trooper%2C%20Pirate%20Panda%2C%20APT%2023%2C%20KeyBoy    
                if GID == "G0081":
                    Attribution = "China" 
                # Leviathan
                # ATT&CK ref https://www.fireeye.com/blog/threat-research/2018/03/suspected-chinese-espionage-group-targeting-maritime-and-engineering-industries.html
                # ATT&CK ref https://www.fireeye.com/blog/threat-research/2019/03/apt40-examining-a-china-nexus-espionage-actor.html
                if GID == "G0065":
                    Attribution = "China" 
                
                
              
                # Russia
                #
                # APT28
                if "has been attributed to Russia" in Description:
                    Attribution = "Russia" 
                # APT29
                if "has been attributed to the Russian" in Description:
                    Attribution = "Russia" 
                # Dragonfly 2.0
                if "is a suspected Russian" in Description:
                    Attribution = "Russia" 
                # FIN5
                if "made up of actors who likely speak Russian" in Description:
                    Attribution = "Russia" 
                # Sandworm Team
                if "is a destructive Russian threat group" in Description:
                    Attribution = "Russia" 
                # TEMP.Veles, Wizard Spider
                if "is a Russia-based" in Description:
                    Attribution = "Russia" 
                # Turla
                if "is a Russian-based threat group" in Description:
                    Attribution = "Russia" 
                
                
                # South America
                #
                # APT-C-36, 
                if "is a suspected South America" in Description:
                    Attribution = "South America"
                    
                # Vietnam
                #
                # APT32
                if "is a suspected Vietnam-based" in Description:
                    Attribution = "Vietnam"
                    
                # Iran
                #
                # APT33, OilRig
                if "is a suspected Iranian" in Description:
                    Attribution = "Iran"
                # APT39, Charming Kitten, CopyKittens, Leafminer, MuddyWater
                if "is an Iranian" in Description:
                    Attribution = "Iran"
                # Cleaver
                if "attributed to Iranian" in Description:
                    Attribution = "Iran"
                # Group5
                if "a suspected Iranian nexus" in Description:
                    Attribution = "Iran"
                # Magic Hound
                if "is an Iranian-sponsored" in Description:
                    Attribution = "Iran"
                # Ajax Security Team
                if "operating out of Iran" in Description:
                    Attribution = "Iran"
                # Silent Librarian
                if "at the behest of the government of Iran" in Description:
                    Attribution = "Iran"
                # Fox Kitten
                if "a suspected nexus to the Iranian government" in Description:
                    Attribution = "Iran"
                    
                # Ferocious Kitten
                # https://cyware.com/news/ferocious-kitten-uses-markirat-to-target-iranian-regime-633b84b4
                if GID == "G0137":
                    Attribution = "Iran" 
                    
                # North Korea
                #
                #  APT37
                if "is a suspected North Korean" in Description:
                    Attribution = "North Korea"
                #  APT37  (ATT&CK text changed to below - previous left for completeness)
                if "is a North Korean" in Description:
                    Attribution = "North Korea"
                # APT38
                if "backed by the North Korea" in Description:
                    Attribution = "North Korea"
                # Kimsuky
                if "is a North Korean-based" in Description:
                    Attribution = "North Korea"
                # Lazarus Group
                if "is a North Korean state-sponsored cyber threat group" in Description:
                    Attribution = "North Korea"
                # Andariel
                if "is a North Korean state-sponsored threat group" in Description:
                    Attribution = "North Korea"
                # DPRK
                # Democratic People's Republic of Korea (North Korea)
                #
                # Stolen Pencil
                if "likely originating from DPRK" in Description:
                    Attribution = "North Korea" 
                
                # South Korea 
                #   
                # Higaisa
                #
                if "suspected to have South Korean origins" in Description:
                    Attribution = "South Korea" 
                
                # Middle East
                #
                #  BlackOasis
                if "is a Middle Eastern" in Description:
                    Attribution = "Middle East"    
        
                # Lebanon
                #
                # Dark Caracal
                if "attributed to the Lebanese" in Description:
                    Attribution = "Lebanon"    
                #
                # Volatile Cedar
                if "is a Lebanese" in Description:
                    Attribution = "Lebanon"     
      
                # Pakistan
                #
                # Gorgon Group
                if "suspected to be Pakistan-based" in Description:
                    Attribution = "Pakistan" 
               
                # Nigeria
                #
                # SilverTerrier
                if "is a Nigerian threat group" in Description:
                    Attribution = "Nigeria" 

                # UAE                    
                #
                # Stealth Falcon
                if "there could be a link between this group and the United Arab Emirates (UAE) government" in Description:
                    Attribution = "UAE" 
                    
                # India
                #
                # Sidewinder
                if "is a suspected Indian threat actor group" in Description:
                    Attribution = "India" 
                    
                # DPRK
                # Democratic People's Republic of Korea (North Korea)
                #
                # Stolen Pencil
                if "likely originating from DPRK" in Description:
                    Attribution = "North Korea" 
                    
                    
                if Attribution == "Unknown":
                    print("<< WARNING >> The group " + GName + " (" + str(GID) + "} has no location intelligence")

                
                
                
                return(Attribution)
                
            def GetTechDomain(Technique, 
                                          TechListOfLists):
                                              
                ThisDomain=""
                i=0
                
                #
                #  Go through the Tech Lists (created in TechToDomainList)
                #  when we find a match return the relevant domain.
                #
                                              
                for TechList in TechListOfLists:
                    for Tech in TechList:
                        if Technique == Tech:
                            if i == 0:
                                ThisDomain="Enterprise"
                                break
                            if i == 1:
                                ThisDomain="Mobile"
                                break
                            if i == 2:
                                ThisDomain="Pre-ATT&CK"
                                break
                                
                    i=i+1
                
                return(ThisDomain)
                
            def CreateTacticIndex(cs, 
                                                 filter_objs_bd):
                
                def GetTacticKCPhase(tactic, 
                                                    kill_chain_type):
                                                        
                    # https://medium.com/@CyberSecurityWarrior/mitre-att-ck-to-kill-chain-models-mapping-9e68bcfee9f 
                                                    
                    LM_switcher={
                                     'TA0043' : ['Reconnaissance'], # Reconnaissance
                                     'TA0042' :['Weaponization'], # Resource Development
                                     'TA0001' :['Delivery',  'Exploitation'], # Initial Access
                                     'TA0002' :['Exploitation'], # Execution
                                     'TA0003' :['Installation'], # Persistence
                                     'TA0004' :['Lateral Movement'], # Privilge Escalation
                                     'TA0005' :['Delivery','Exploitation','Installation','Lateral Movement'],  # Defense Evasion
                                     'TA0006' :['Actions On Objectives'], # Credential Access
                                     'TA0007' :['Reconnaissance','Lateral Movement'], # Discovery
                                     'TA0008' :['Lateral Movement'], # Lateral Movement
                                     'TA0009' :['Actions On Objectives'], # Collection
                                     'TA0010' :['Actions On Objectives'], # Exfiltration
                                     'TA0011' :['Command & Control'], # Comand & Control
                                     'TA0040' :['Actions On Objectives'],  # Impact 
                                     
                                     'TA0027' : ['Delivery',  'Exploitation'], # Initial Access
                                     'TA0041' :['Exploitation'], # Execution
                                     'TA0028' :['Installation'], # Persistence
                                     'TA0029' :['Lateral Movement'], # Privilge Escalation
                                     'TA0030' :['Delivery','Exploitation','Installation','Lateral Movement'],  # Defense Evasion
                                     'TA0031' :['Actions On Objectives'], # Credential Access
                                     'TA0032' :['Reconnaissance','Lateral Movement'], # Discovery
                                     'TA0033' :['Lateral Movement'], # Lateral Movement
                                     'TA0035' :['Actions On Objectives'], # Collection
                                     'TA0036' :['Actions On Objectives'], # Exfiltration
                                     'TA0037' :['Command & Control'], # Comand & Control
                                     'TA0034' :['Actions On Objectives'],  # Impact
                                     'TA0038' :['Actions On Objectives'], # Network Effects
                                     'TA0039' :['Actions On Objectives'] # Remote Service Effects
                                      }
                                      
                    if kill_chain_type == "LM":                  
                        kill_chain_phase=LM_switcher.get(tactic,  ["Phase undefined"])
                    else:
                        print("Unknown kill chain type: " + kill_chain_type)
                                                        
                    return(kill_chain_phase)
                    
                    
                # #####################
                #  Start core function
                # #####################
                
                # Retrieve all Tactics from the prepared memory source
                x_mitre_matrix_list=cs.query(filter_objs_bd['x_mitre_matrix'])
                print("Preparing a matrix index")
          
                rows_list_tactics=[]
          
                # The code below to create a Tech to Tactic index (necessarily) assumes uniqueness of names
                # So have just inserted a warning to spot if this does not hold true here
                tactic_name_list=[]
                for x_mitre_matrix in x_mitre_matrix_list:

                    for tactic in x_mitre_matrix['tactic_refs']:
                        tactic_details=cs.get(tactic)
                        dict_tactics={
                                 "TACTIC_EXT_ID": tactic_details['external_references'][0]['external_id'], 
                                 "TACTIC_ID": tactic_details['id'], 
                                 "TACTIC_NAME": tactic_details['name'], 
                                 "TACTIC_DESC": tactic_details['description'], 
                                 "TACTIC_CREATED": tactic_details['created'], 
                                 "TACTIC_MODIFIED": tactic_details['modified'], 
                                 "TACTIC_MATRIX": x_mitre_matrix['external_references'][0]['external_id'], 
                                 "TACTIC_MATRIX_URL": x_mitre_matrix['external_references'][0]['url'], 
                                 "LM_KILL_CHAIN_PHASE": GetTacticKCPhase(tactic_details['external_references'][0]['external_id'], 
                                                                                              "LM")
                                 }
 
                        if tactic_details['name'] not in tactic_name_list:
                            tactic_name_list.append(tactic_details['name'])
                        else:
                            print("WARNING: " + tactic_details['name'] + " is not unique")
                  
                        rows_list_tactics.append(dict_tactics) # ATTACK_TACTIC_INDEX
          
                # Added/Moved here so we can use in construction of links to techniques below
                self.ATTACK_TACTIC_Index = self.ATTACK_TACTIC_Index.append(pd.DataFrame(rows_list_tactics), sort=True)
                rows_list_tactics=[]

                return(0)
                
            def AddTechniques(cs, 
                                            filter_objs_bd):
                
                # ##########################
                # Add techniques, malware and tools
                # ##########################
                
                print("Preparing a technique index")
                print("Also TTP to Tactic index ('many' for each TTP)")
                print("Adding TTPs (including sub techniques)")
          
                # Retrieve all TTP from the prepared memory source
                TTP_list=cs.query(filter_objs_bd['techniques']
                                         )
          
                rows_list_TTP=[]
                rows_list_tech_to_tactic=[] 
                TTP_Ref_Rows=[]
                rows_list_TTP_PLAT=[]
                rows_list_TTP_TO_DATA_SOURCE=[]
                rows_list_TTP_TO_PERMS_REQ=[]
            
                #i=0
                for TTPNext in TTP_list:
                    #if i==0:
                       #print(str(TTPNext))
                        #i=1
                    #print("Next Tech << " + TTPNext.external_references[0]['external_id'])
                
                    #if TTPNext.external_references[0]['external_id'] == "T1055.014":
                    #print(str(TTPNext))
                    #if "https://github.com/hfiref0x/UACME" in TTPNext:
                        #print(str(TTPNext))
                    #print("The id is " + str(TTPNext.external_references[0]['external_id']))
              
                    # Need to consider how to deal with x_mitre_deprecated at some point.
                
                    TechDomain=GetTechDomain(TTPNext.external_references[0]['external_id'], 
                                              TechListOfLists)
                                          
                    if TTPNext.revoked == True:
                        TTP_descriptiion = "Unknown"
                    else:
                        TTP_descriptiion = TTPNext.description
                    
                    if "x_mitre_detection" in TTPNext:
                        detect_str=TTPNext.x_mitre_detection
                    
                    dict_TTP={
                                 "TTP_Name": TTPNext.name, 
                                 "TTP_Type": TTPNext.type, 
                                 "TTP_ID": TTPNext.id, 
                                 "TTP_EXT_ID": TTPNext.external_references[0]['external_id'], 
                                 "TTP_URL":TTPNext.external_references[0]['url'],
                                 "TTP_DESC":TTP_descriptiion, 
                                 "TTP_Phase": [],
                                 "TTP_Domain" : TechDomain, 
                                 "TTP_x_mitre_detection":detect_str, 
                                 "TTP_Date_Created": TTPNext.created, 
                                 "TTP_Date_Modified": TTPNext.modified, 
                                 "TTP_x_mitre_deprecated":False,
                                 "TTP_x_mitre_is_subtechnique":False,
                                 "Revoked":TTPNext.revoked, 
                                 "TTP_CAR_Available":HasCARAnalytic(TTPNext.external_references[0]['external_id'])
                                 }
                             
                    if "x_mitre_deprecated" in TTPNext:
                        dict_TTP.update({"TTP_x_mitre_deprecated": TTPNext.x_mitre_deprecated})
                    if "x_mitre_is_subtechnique" in TTPNext:
                        dict_TTP.update({"TTP_x_mitre_is_subtechnique": TTPNext.x_mitre_is_subtechnique})

                    if not TTPNext.revoked:
                        if not dict_TTP.get("TTP_x_mitre_deprecated"):
                   
                            phase_list=[]
                            if TTPNext.type == "attack-pattern":
                              for phase in TTPNext.kill_chain_phases:
                                phase_list.append(phase['phase_name'])
                      
                                if phase['kill_chain_name'] == "mitre-attack":
                                    kill_chain_name_conv="enterprise-attack"
                                elif phase['kill_chain_name'] == "mitre-pre-attack":
                                    kill_chain_name_conv="pre-attack"
                                elif phase['kill_chain_name'] == "mitre-mobile-attack":
                                    kill_chain_name_conv="mobile-attack"
                                else:
                                    print("<<WARNING>>: Unknown kill chain name found ")
                                    print("For " + str(TTPNext.id) + " :name is " + phase['kill_chain_name'])
                         
                                Tactic_ID=self.ATTACK_TACTIC_Index[(self.ATTACK_TACTIC_Index['TACTIC_MATRIX'] == kill_chain_name_conv) & 
                                                                                    (self.ATTACK_TACTIC_Index['TACTIC_NAME'].str.lower() == phase['phase_name'].lower().replace('-',  ' '))]['TACTIC_EXT_ID']
                                                                          
                      
                                if len(Tactic_ID) == 0:
                          
                                    print("<<WARNING>>: No match for " + str(kill_chain_name_conv) + " and " + phase['phase_name'].lower().replace('-',  ' '))
                                    print("For " + TTPNext.external_references[0]['external_id'] + " :name is " + phase['kill_chain_name'])
                          
                                    #print(str(TTPNext))
                          
                                    if not TTPNext.x_mitre_deprecated:
                                               
                                        dict_tech_to_tactic={"TTP_NAME": TTPNext.name, 
                                                "TTP_EXT_ID": TTPNext.external_references[0]['external_id'], 
                                                "KILL_CHAIN_NAME": phase['kill_chain_name'], 
                                                "TACTIC_NAME": phase['phase_name'], 
                                                "TACTIC_EXT_ID": "<<NOT FOUND>>", 
                                                "REL_Date_Created": "TBC"}
                                      
                                        rows_list_tech_to_tactic.append(dict_tech_to_tactic) # ATTACK_TECH_TO_TACTIC_Index
                                                                                                   
                                else: # if len(Tactic_ID) == 0
                                    for NextID in Tactic_ID:
                                        ThisID=str(NextID)
                                    dict_tech_to_tactic={"TTP_NAME": TTPNext.name, 
                                            "TTP_EXT_ID": TTPNext.external_references[0]['external_id'], 
                                            "KILL_CHAIN_NAME": phase['kill_chain_name'], 
                                            "TACTIC_NAME": phase['phase_name'], 
                                            "TACTIC_EXT_ID": ThisID, 
                                            "REL_Date_Created": "TBC"}
                                    rows_list_tech_to_tactic.append(dict_tech_to_tactic) # ATTACK_TECH_TO_TACTIC_Index
                                


                              # end for phase in TTPNext.kill_chain_phases:
                              dict_TTP.update({"TTP_Phase": phase_list})
                                                      
                              # Add rows for TTP reference table
                              #  There appears to be an issue that all references cited in the MITTRE webpage are
                              #       not in the TAXII server (or present in the github STIX files)
                              #   An approach to add simple scraping
                              #        html = urlopen("https://attack.mitre.org/techniques/T1484/002").read().decode('utf-8')
                              #       Should be able to find multiple instances of:
                              #
                              # <li>
                              #    <span  id="scite-1" class="scite-citation">
                              #           <span class="scite-citation-text">
                              #                 < a rel="nofollow" class="external text" name="scite-1" href="https://docs.microsoft.com/en-us/azure/active-directory/hybrid/whatis-fed" target="_blank">
                              #                        Microsoft. (2018, November 28). What is federation with Azure AD?. Retrieved December 30, 2020.
                              #                 </a>
                              #          </span>
                              #    </span>
                              # </li>
                              #  and extract the reference links from relevant href
                              #
                          
                              # This list (and check below) is here to allow additional html reference extraction to be added later.
                              ThisUrlList=[]
                          
                              for next_item in TTPNext.external_references:
                                  # 
                                  dict_TTP_Ref_Line={"TTP_EXT_ID": TTPNext.external_references[0]['external_id'], 
                                                         "source_name": "NA", 
                                                         "url": "NA",
                                                         "external_id": "NA" , 
                                                         "description": "NA"
                                                           }
                              
                                  if "external_id" in next_item: 
                                      dict_TTP_Ref_Line.update({"external_id": next_item.external_id})                      
                                                       
                                  if "description" in next_item:    
                                      dict_TTP_Ref_Line.update({"description": next_item.description})    
                                                       
                        
                                  
                                  if "source_name" in next_item:    
                                      dict_TTP_Ref_Line.update({"source_name": next_item.source_name})  
                                  
                                  if "url" in next_item:  
                                      dict_TTP_Ref_Line.update({"url": next_item.url})    
                                      # See note above (by ThisUrlList=[])
                                      if next_item.url not in ThisUrlList:                                
                                          TTP_Ref_Rows.append(dict_TTP_Ref_Line)
                              
                              # Add rows for TTP platforms table
                              for next_plat in TTPNext.x_mitre_platforms:
                                  # 
                                  dict_TTP_PLAT_Line={"TTP_EXT_ID": TTPNext.external_references[0]['external_id'], 
                                                         "Platform": next_plat
                                                           }                              

                                  rows_list_TTP_PLAT.append(dict_TTP_PLAT_Line)
                              
                              # Add rows for TTP perm req table
                              if "x_mitre_permissions_required" in TTPNext:
                                  for next_perm in TTPNext.x_mitre_permissions_required:
                                      # 
                                      dict_TTP_PERM_REQ_Line={"TTP_EXT_ID": TTPNext.external_references[0]['external_id'], 
                                                         "Permission": next_perm
                                                           }                              

                                      rows_list_TTP_TO_PERMS_REQ.append(dict_TTP_PERM_REQ_Line)
                              
                              # Add rows for TTP to DATA_SOURCE
                              if "x_mitre_data_sources" in TTPNext:
                                  for next_source in TTPNext.x_mitre_data_sources:
                                      # 
                                      dict_TTP_TO_DATA_SOURCE={"TTP_EXT_ID": TTPNext.external_references[0]['external_id'], 
                                                             "DATA_SOURCE": next_source
                                                               }                              

                                      rows_list_TTP_TO_DATA_SOURCE.append(dict_TTP_TO_DATA_SOURCE)
                              
                              # end change
                            # End  if TTPNext.type == "attack-pattern":
                        # if not dict_TTP.get("TTP_x_mitre_deprecated"):
                    # End if not TTPNext.revoked:
                    rows_list_TTP.append(dict_TTP) # ATTACK_TTP_Index
                    
                # ##########################
                # Add malware
                # ##########################

                print("Adding Malware")                  
                TTP_list=cs.query(filter_objs_bd['malware']
                                     )
          
                for TTPNext in TTP_list:
                    #print("TTP is " + TTPNext.external_references[0]['external_id'])
                    #print(str(TTPNext))
                    if "https://github.com/hfiref0x/UACME" in TTPNext:
                        print(str(TTPNext))
                    
                    if TTPNext.revoked == True:
                        TTP_descriptiion = "Unknown"
                    else:
                        TTP_descriptiion = TTPNext.description

                    dict_TTP={
                            "TTP_Name": TTPNext.name, 
                            "TTP_Type": TTPNext.type, 
                            "TTP_ID": TTPNext.id, 
                            "TTP_EXT_ID": TTPNext.external_references[0]['external_id'], 
                            "TTP_URL":TTPNext.external_references[0]['url'],
                            "TTP_DESC":TTP_descriptiion, 
                            "TTP_Phase": [], 
                            "TTP_Date_Created": TTPNext.created, 
                            "TTP_Date_Modified": TTPNext.modified, 
                            "TTP_x_mitre_deprecated":False,
                            "TTP_x_mitre_is_subtechnique":False,
                            "Revoked":TTPNext.revoked, 
                            "TTP_CAR_Available":False
                            }
                             
                    if "x_mitre_deprecated" in TTPNext:
                        dict_TTP.update({"TTP_x_mitre_deprecated": TTPNext.x_mitre_deprecated})
                    if "x_mitre_is_subtechnique" in TTPNext:
                        dict_TTP.update({"TTP_x_mitre_is_subtechnique": TTPNext.x_mitre_is_subtechnique})
                    
                    # Add rows for TTP platforms table
                    if "x_mitre_platforms" in TTPNext:
                        for next_plat in TTPNext.x_mitre_platforms:
                           # 
                            dict_TTP_PLAT_Line={"TTP_EXT_ID": TTPNext.external_references[0]['external_id'], 
                                                         "Platform": next_plat
                                                    }                              

                            rows_list_TTP_PLAT.append(dict_TTP_PLAT_Line)
                        
                    # This list (and check below) is here to allow additional html reference extraction to be added later.
                    ThisUrlList=[]
                
                    if dict_TTP['TTP_x_mitre_deprecated'] == False:
                        if dict_TTP['Revoked'] == False:
                            for next_item in TTPNext.external_references:
                             # 
                                dict_TTP_Ref_Line={"TTP_EXT_ID": TTPNext.external_references[0]['external_id'], 
                                                         "source_name": "NA", 
                                                         "url": "NA",
                                                         "external_id": "NA" , 
                                                         "description": "NA"
                                                           }
                              
                                if "external_id" in next_item: 
                                    dict_TTP_Ref_Line.update({"external_id": next_item.external_id})                      
                                                       
                                if "description" in next_item:    
                                    dict_TTP_Ref_Line.update({"description": next_item.description})    
                                                       
                        
                                  
                        if "source_name" in next_item:    
                            dict_TTP_Ref_Line.update({"source_name": next_item.source_name})  
                                  
                        if "url" in next_item:  
                            dict_TTP_Ref_Line.update({"url": next_item.url})    
                            # See note above (by ThisUrlList=[])
                            if next_item.url not in ThisUrlList:                                
                                TTP_Ref_Rows.append(dict_TTP_Ref_Line)

              
                    rows_list_TTP.append(dict_TTP)
                  
                # ##########################
                # Add tools
                # ##########################

                print("Adding Tools")                  
                TTP_list=cs.query(filter_objs_bd['tools']
                                         )
          
                for TTPNext in TTP_list:
                    #print("TTP is " + TTPNext.external_references[0]['external_id'])
                    #print(str(TTPNext))
                    #if "https://github.com/hfiref0x/UACME" in TTPNext:
                        #print(str(TTPNext))
                
                    if TTPNext.revoked == True:
                        TTP_descriptiion = "Unknown"
                    else:
                        TTP_descriptiion = TTPNext.description
                
                    dict_TTP={
                                 "TTP_Name": TTPNext.name, 
                                 "TTP_Type": TTPNext.type, 
                                 "TTP_ID": TTPNext.id, 
                                 "TTP_EXT_ID": TTPNext.external_references[0]['external_id'], 
                                 "TTP_URL":TTPNext.external_references[0]['url'],
                                 "TTP_DESC":TTP_descriptiion, 
                                 "TTP_Phase": [], 
                                 "TTP_Date_Created": TTPNext.created, 
                                 "TTP_Date_Modified": TTPNext.modified, 
                                 "TTP_x_mitre_deprecated":False,
                                 "TTP_x_mitre_is_subtechnique":False,
                                 "Revoked":TTPNext.revoked, 
                                 "TTP_CAR_Available":False
                                 }

                    if "x_mitre_deprecated" in TTPNext:
                        dict_TTP.update({"TTP_x_mitre_deprecated": TTPNext.x_mitre_deprecated})
                    if "x_mitre_is_subtechnique" in TTPNext:
                        dict_TTP.update({"TTP_x_mitre_is_subtechnique": TTPNext.x_mitre_is_subtechnique})
                    
                    # Add rows for TTP platforms table
                    if "x_mitre_platforms" in TTPNext:
                        for next_plat in TTPNext.x_mitre_platforms:
                           # 
                            dict_TTP_PLAT_Line={"TTP_EXT_ID": TTPNext.external_references[0]['external_id'], 
                                                         "Platform": next_plat
                                                    }                              

                            rows_list_TTP_PLAT.append(dict_TTP_PLAT_Line)
                    
                    # This list (and check below) is here to allow additional html reference extraction to be added later.
                    ThisUrlList=[]
                
                    if dict_TTP['TTP_x_mitre_deprecated'] == False:
                        if dict_TTP['Revoked'] == False:
                            for next_item in TTPNext.external_references:
                             # 
                                dict_TTP_Ref_Line={"TTP_EXT_ID": TTPNext.external_references[0]['external_id'], 
                                                         "source_name": "NA", 
                                                         "url": "NA",
                                                         "external_id": "NA" , 
                                                         "description": "NA"
                                                           }
                              
                                if "external_id" in next_item: 
                                    dict_TTP_Ref_Line.update({"external_id": next_item.external_id})                      
                                                       
                                if "description" in next_item:    
                                    dict_TTP_Ref_Line.update({"description": next_item.description})    
                                                       
                        
                                  
                        if "source_name" in next_item:    
                            dict_TTP_Ref_Line.update({"source_name": next_item.source_name})  
                                      
                        if "url" in next_item:  
                            dict_TTP_Ref_Line.update({"url": next_item.url})    
                            # See note above (by ThisUrlList=[])
                            if next_item.url not in ThisUrlList:                                
                                TTP_Ref_Rows.append(dict_TTP_Ref_Line)
 
                    rows_list_TTP.append(dict_TTP)
                    
                # Populate the dataframe with dict list
                self.ATTACK_TTP_Index = self.ATTACK_TTP_Index.append(pd.DataFrame(rows_list_TTP), sort=True)
                rows_list_TTP=[]
                self.ATTACK_TECH_TO_TACTIC_Index = self.ATTACK_TECH_TO_TACTIC_Index.append(pd.DataFrame(rows_list_tech_to_tactic), sort=True)
                rows_list_tech_to_tactic=[]
                self.ATTACK_TTP_REF_Index = self.ATTACK_TTP_REF_Index.append(pd.DataFrame(TTP_Ref_Rows), sort=True)
                TTP_Ref_Rows=[]
                self.ATTACK_TTP_TO_PLATFORM_Index = self.ATTACK_TTP_TO_PLATFORM_Index.append(pd.DataFrame(rows_list_TTP_PLAT), sort=True)
                rows_list_TTP_PLAT=[]
                self.ATTACK_TTP_TO_DATA_SOURCE_Index = self.ATTACK_TTP_TO_DATA_SOURCE_Index.append(pd.DataFrame(rows_list_TTP_TO_DATA_SOURCE), sort=True)
                rows_list_TTP_TO_DATA_SOURCE=[]
                self.ATTACK_TTP_TO_PERMS_REQ_Index = self.ATTACK_TTP_TO_PERMS_REQ_Index.append(pd.DataFrame(rows_list_TTP_TO_PERMS_REQ), sort=True)
                rows_list_TTP_TO_PERMS_REQ=[]

                
            def CreateBinIndexes():
                
                def SortFunc(e):
                    return e.lower()
                    
                print("Preparing binary indexes")
                
                TacticBinList=[]
                TTPBinList=[]
                
                # Get a list of groups
                GroupNameList=self.ATTACK_GROUP_Index['Group'].tolist()
                     
                # Sort the list (primarily for readability)                
                GroupNameList.sort(key=SortFunc)
                
                # Get a total list of tactics
                TacticList=self.ATTACK_TACTIC_Index['TACTIC_EXT_ID'].tolist()
                
                # Get a total list of TTP
                TTPTotalList=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['Revoked'] == False) &
                                                      (self.ATTACK_TTP_Index['TTP_x_mitre_deprecated'] == False )]['TTP_EXT_ID'].tolist()
            
                # Step through the groups creating a binary vector aech
                #     Presence of a tactic will be marked as a 1
                for group_name in GroupNameList:
                    # Prepare a binary representation (init zeros) for this group
                    TacticBin=[0]*len(TacticList)
                    TTPBin=[0]*len(TTPTotalList)
                             
                    # First get a list of TTP for Group
                    #   
                    TTP_list=self.ATTACK_REL_Index[(self.ATTACK_REL_Index['REL_SOURCE_NAME'] == group_name) &
                                    (self.ATTACK_REL_Index['REL_SOURCE_TYPE'] == "intrusion-set") &
                                    ((self.ATTACK_REL_Index['REL_TARGET_TYPE'] == "attack-pattern") |
                                    (self.ATTACK_REL_Index['REL_TARGET_TYPE'] == "malware") |
                                    (self.ATTACK_REL_Index['REL_TARGET_TYPE'] == "tool"))]['REL_TARGET_EXT_ID'].tolist()
                                                     
                    #
                    GroupTTPList=[]
                    #  Found examples of relationships that point to revoked TTP
                    #  e.g. G0095 and T1032
                    #  So remove revoked TTP from final list
                    for TTP in TTP_list:

                        # Check to see if this TTP is revoked
                        # Returns a Panda Series
                        TTP_revoked=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID']  == TTP)]['Revoked']
                        TTP_x_mitre_deprecated=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID']  == TTP)]['TTP_x_mitre_deprecated']
       
                        # Should only be one row but fo safety take top row only 
                        for Revoked in TTP_revoked:
                            FRevoked = Revoked
                            break
                
                        for Deprecate in TTP_x_mitre_deprecated:
                            FDeprecate = Deprecate
                            break
                
                        if ((FRevoked == False) & (FDeprecate == False)):
                            GroupTTPList.append(TTP)
                            
                    #if group_name == "admin@338":
                        #print("The GroupTTPList for " + group_name + " is " + str(GroupTTPList))
                    
                    #
                    # Now get the tactics for each of the TTP   
                    GroupTacticList=[]            
                    for ThisTTP in GroupTTPList:
                        
                        TTPPos=TTPTotalList.index(ThisTTP)
                        TTPBin[TTPPos]=1
                        
                        #if group_name == "admin@338":
                            #print("The TTP for " + group_name + " is " + str(ThisTTP))
                            
                        ThisTacticList=self.ATTACK_TECH_TO_TACTIC_Index[(self.ATTACK_TECH_TO_TACTIC_Index['TTP_EXT_ID'] == ThisTTP)]['TACTIC_EXT_ID'].tolist()
                        
                        #if group_name == "admin@338":
                            #print("The tactic list for " + group_name + " is " + str(ThisTacticList))  
                            
                        for ThisTactic in ThisTacticList:                
                            if ThisTactic not in GroupTacticList:
                                GroupTacticList.append(ThisTactic)
                                
                    #if group_name == "admin@338":
                        #print("The GroupTacticList for " + group_name + " is " + str(GroupTacticList))
                           
                    for GroupTactic in GroupTacticList:
                       #print("Updating for Group <" + str(NextGID) + "> TTP <" + GroupTTP + ">")
                        TacticPos=TacticList.index(GroupTactic)
                        TacticBin[TacticPos]=1
                    
                    # Can be used to remove groups with no tactics if required
                    TacticBinList.append(TacticBin)
                    TTPBinList.append(TTPBin)
                    

                # Convert to a dataframe (so we can write it out as CSV)
                self.ATTACK_TACTIC_BIN_Index = pd.DataFrame(TacticBinList)
                # Add column names (Tactics) and row indexes (groups)
                self.ATTACK_TACTIC_BIN_Index.columns = TacticList
                self.ATTACK_TACTIC_BIN_Index.index = GroupNameList
                
                # Convert to a dataframe (so we can write it out as CSV)
                self.ATTACK_TTP_BIN_Index = pd.DataFrame(TTPBinList)
                # Add column names (Tactics) and row indexes (groups)
                self.ATTACK_TTP_BIN_Index.columns = TTPTotalList
                self.ATTACK_TTP_BIN_Index.index = GroupNameList
                
                
                #   #########################################
                # CREATE THE ATTACK_TTP_REF_BIN_INDEX to be used for NLP matching
                # Step through the groups creating a binary vector aech
                #     Presence of a tactic will be marked as a 1
                
                #
                RefBinList=[]
                
                # Get a unique list of references from the ATTACK_TTP_REF_Index
                unique_ref_list=self.ATTACK_TTP_REF_Index['url'].unique().tolist()
                
                #cnt=0
                for next_ref in unique_ref_list:
                    
                    RefBin=[0]*(len(TacticList) + len(TTPTotalList))
                    
                    #print("The next reference is " + str(next_ref))
                    
                    TechRefList=self.GetTechForRef(next_ref)
                    for NextTech in TechRefList:
                        
                        TTPPos=TTPTotalList.index(NextTech)
                        RefBin[(len(TacticList)) + TTPPos]=1
                        
                    #for NextTech in TechRefList:
                        #print("The next technique is " + str(NextTech))
                        
                    TacticRefList=self.GetTacticForRef(next_ref)
                    for NextTactic in TacticRefList:
                        TacticPos=TacticList.index(NextTactic)
                        RefBin[TacticPos]=1
                    
                    
                    #for NextTac in TacticRefList:
                        #print("The next tactic is " + str(NextTac))
                    #cnt=cnt+1
                    #if cnt>10:
                    #    break
                        
                    TotRefBin=[0]
                    TotRefBin.extend(RefBin)
                    
                    RefBinList.append(TotRefBin)
                    
                # Convert to a dataframe (so we can write it out as CSV)
                self.ATTACK_TTP_REF_BIN_Index = pd.DataFrame(RefBinList)
                
                ColNames=['Text']
                ColNames.extend(TacticList)
                ColNames.extend(TTPTotalList)  
                
                self.ATTACK_TTP_REF_BIN_Index.columns = ColNames
                self.ATTACK_TTP_REF_BIN_Index.index = unique_ref_list
                        
                return(0)
                
            def GetCARCoverage(url, # https://car.mitre.org/coverage/
                                              ind_fname_ATTACK_CAR_COVERAGE): 
                print("Downloading CAR coverage csv file")
                print("Accessing at " + url)
                
                # <li>A <a href="/coverage/analytic_coverage_02_04_2022.csv">CSV file</a>.</li>
                
                src_file_re='https://car.mitre.org/analytic_coverage*\.csv'
                src_file_re='/coverage/analytic_coverage.*\.csv'
                
                r = requests.get(url)
                #print(r.text)
                # Need to adjust for "modified" and "recent" as well.
                f_count=0
                #f_num=len(re.findall(src_file_re, r.text))
                
                filename = re.findall(src_file_re, r.text)
                #print("The find result is " + str(filename))

                for filename in re.findall(src_file_re, r.text):
                  base_filename=filename.rsplit('/', 1)[1]
                  base_filename=os.path.basename(filename)
                  #print("The base filename is " + str(base_filename))
                  #print("The target filename is " + url + str(base_filename))
                  
                  f_count = f_count+1
                  #print(str(datetime.datetime.now().time()) +
                      #" " + filename + " " + str(f_count) +
                      #" of " + str(f_num))
                  # e.g. "https://cwe.mitre.org/data/csv/"
                  r_file = requests.get(url + base_filename, stream=True)            
                  with open(ind_fname_ATTACK_CAR_COVERAGE, 'wb') as f:
                    for chunk in r_file:
                          f.write(chunk)
                          
                if f_count == 0:
                    print("<<WARNING>> No CAR file downloaded")
                    
                else:
                    # Load the ATTACK_TTP_TO_PLATFORM_Index index
                    #   
                    #              
                    print("Loading index" + ind_fname_ATTACK_CAR_COVERAGE)                    
                    exists = os.path.exists(ind_fname_ATTACK_CAR_COVERAGE)
                    if (exists):
                        
                        colnames=['Technique',  'Name',  'Sub_Name', 
                                               'CAR_Num',  'Sigma_Num',  'ES_Num',  'Splunk_Num',  
                                               'Tot_Num']
                                               
                        self.ATTACK_CAR_COVERAGE_Index=pd.read_csv(ind_fname_ATTACK_CAR_COVERAGE, names=colnames,  header=0)
                        print("Index loaded " + str(self.ATTACK_CAR_COVERAGE_Index.shape[0]))
                    else:
                        print("No index found, rerun with reindex parameter set")
                    self.ATTACK_CAR_COVERAGE_Index.info()
               
                # 
            def HasCARAnalytic(TTP_EXT_ID):
            #                                                                                                               
                CARDictList=self.ATTACK_CAR_COVERAGE_Index[(self.ATTACK_CAR_COVERAGE_Index['Technique'] == TTP_EXT_ID) ].to_dict('records')
                                                              
                # Should only be one returned but ...
                HasAnalytic=False
        
                for NextCARDict in CARDictList:
                    ThisCARDict=NextCARDict
                    if ThisCARDict['Tot_Num'] > 0:
                        HasAnalytic=True
                    break
                    
                return(HasAnalytic)
                
            def GetThaiCERTEnrichmentData(url, # https://apt.etda.or.th/
                                              ind_fname_ATTACK_TCERT_GROUP, 
                                              ind_fname_ATTACK_TCERT_TOOL): 
                                                  
                print("Downloading ThaiCERT Enrichment Data")
                
                # Look for references to JSON creation cgi li nes
                
                src_file_re='/.*\.cgi\?o=.?'
                
                r = requests.get(url + "cgi-bin/aptgroups.cgi")
                #print(r.text)
                # Need to adjust for "modified" and "recent" as well.
                f_count=0
                #f_num=len(re.findall(src_file_re, r.text))
                
                filename = re.findall(src_file_re, r.text)
                
                for filename in re.findall(src_file_re, r.text):
                  get_file=url+filename
                  #print("The base filename is " + str(get_file))
                                   
                  f_count = f_count+1
                  #print(str(datetime.datetime.now().time()) +
                      #" " + filename + " " + str(f_count) +
                      #" of " + str(f_num))
                  # e.g. "https://cwe.mitre.org/data/csv/"
                  r_file = requests.get(get_file, stream=True)  
                  
                  if filename[-1] == "g":
                      #print("The group filename is " + filename)
                      write_file=ind_fname_ATTACK_TCERT_GROUP 
                      #print("About to write group file")
                      #self.TCERT_group_data = json.load(r_file)
                  else:
                      #print("The tool filename is " + filename)
                      write_file=ind_fname_ATTACK_TCERT_TOOL 
                      #print("About to write tool file")
                      #self.TCERT_tool_data = json.load(r_file)
                      
                  with open(write_file, 'wb') as f:
                    for chunk in r_file:
                          f.write(chunk)
                  #print("File written")
                  f.close()
                  
                  #print("About to reopen file")        
                  with open(write_file, 'r',  encoding='utf-8') as f:
                       if filename[-1] == "g":
                          #print("About to load JSON")        
                          #self.TCERT_group_data = json.load(f)
                          self.TCERT_group_data =json.load(f)
                       else:
                          self.TCERT_tool_data = json.load(f)
                  #print("Loaded JSON")     
                          
                if f_count == 0:
                    print("<<WARNING>> No Thai TCERT file downloaded")
                    
                # Dummy check group data
                #print("Checking dummy value (Big Panda)" + self.TCERT_group_data['values'][0]["value"])
                ResList = GetThaiCERTDataItems("APT32",  "Country")
                print("The response value is " + str(ResList))
                    
               
                # 
            def GetThaiCERTDataItems(Group, # https://apt.etda.or.th/
                                              DataType): 
                                                  
                #print("Getting ThaiCERT Enrichment Data Item")
                
                ThisDict={}
                
                ResList=[]
                
                GroupClean=Group
                
                if Group. startswith("APT"):
                    GroupClean="APT "+ Group[3:]
                
                for NextDict in self.TCERT_group_data['values']:
                    
                    ValueList=NextDict['value'].split(",")
                    
                    for ThisValue in ValueList:
                        if ThisValue == GroupClean:
                            ThisDict=NextDict
                            break
                            
                    if not ThisDict == {}:
                        break
                            
                    if "meta" in NextDict:
                        if "synonyms" in NextDict['meta']:
                            for ThisSyn in NextDict['meta']['synonyms']:
                                if ThisSyn == GroupClean:
                                    ThisDict=NextDict
                                    break
                                    
                    if not ThisDict == {}:
                        break
                                    
                if DataType == "Country":
                    if "meta" in ThisDict:
                        if "country" in ThisDict['meta']:
                            ResList.append(ThisDict['meta']['country'])
                            
                if len(ResList) == 0:
                    ResList.append("EMPTY")                            
                
                #res = next((sub for sub in self.TCERT_group_data['values'] if (sub['value'] == Group)), None)
                
                return(ResList)
                
            # #############################
            #  BaseDataToTables  ----  Core function begins
            #      STEP 0 - Get CAR Coverage
            #      STEP 0.1 - Get ThaiCERT enrichment data
            #      STEP 1 - Prepare an index of tactics
            #      STEP 2 - Prepare an index of techniques, malware and tools
            #      STEP 3 - Prepare an index of Groups
            #                      Also create index of references for Groups (1 to many GID/Group)
            #      STEP 4 - Prepare an index of Relationships
            #      STEP 5 - Prepare binary indexes Group x Tactics / Group x TTP
            #      STEP END - Write out the Base Data Tables
            #     
            # #####################            
            print("<< Preparing base data indexes >>")
            
            filter_objs_bd = {"techniques": Filter("type", "=", "attack-pattern"),
                        "mitigations": Filter("type", "=", "course-of-action"), # Not for pre-attack
                        "groups": Filter("type", "=", "intrusion-set"),
                        "malware": Filter("type", "=", "malware"), # Not for pre-attack
                        "tools": Filter("type", "=", "tool"),
                        "relationships": Filter("type", "=", "relationship"), 
                        "x_mitre_tactic": Filter("type", "=", "x-mitre-tactic"), 
                        "x_mitre_matrix": Filter("type", "=", "x-mitre-matrix"), 
                        "x_mitre_data_component": Filter("type", "=", "x-mitre-data-component"), # Only Enterprise
                        "x_mitre_data_source": Filter("type", "=", "x-mitre-data-source") # Only Enterprise
                        }
                        
            # STEP 0 ##########################
            
            GetCARCoverage("https://car.mitre.org/coverage/", 
                                    ind_fname_ATTACK_CAR_COVERAGE)
                                    
            # STEP 0.1 ##########################
            
            GetThaiCERTEnrichmentData("https://apt.etda.or.th/", 
                                    ind_fname_ATTACK_TCERT_GROUP, 
                                    ind_fname_ATTACK_TCERT_TOOL)
            
            # STEP 1 ##########################
            #
            # Prepare an index of tactics
            # ##########################
            
            CreateTacticIndex(cs, 
                                        filter_objs_bd)

            # STEP 2##########################
            # Prepare an index of techniques, malware and tools
            # ##########################
                      
            AddTechniques(cs, 
                                    filter_objs_bd)
             
            # STEP 3 ############################
            # Prepare an index of Groups
            #     Also create index of references for Groups (1 to many GID/Group)
            #
            # ############################
            # 
            print("Preparing a Group index")
            print("Also separate Group to References index ('many' for each group)")
            
            Group_list=cs.query(filter_objs_bd['groups']
                                     )
            #print("<XXX> <XXX> List of group EXT IDs")
            #for GroupNext in Group_list:
                #print("Next is " + GroupNext.external_references[0]['external_id'])
            #print("<XXX> <XXX> List done")
          
            rows_list_Groups=[]
            rows_list_Group_Aliases=[]
            rows_list_group_refs=[]
           # i=0
            for GroupNext in Group_list:
                #if i==0:
                    #print(str(GroupNext))
                    #i=1
                  
                #print("TTP is " + TTPNext.external_references[0]['external_id'])
                #print(str(TTPNext))
                #if "https://github.com/hfiref0x/UACME" in GroupNext:
                    #print(str(TTPNext))
        
                
                if not GroupNext.revoked:
                   # if i==0:
                        #print("<<GROUPS>> in here")
                        
                    dict_Group={
                             "Group": GroupNext.name, 
                             "Aliases": [], 
                             "GID": GroupNext.external_references[0]['external_id'], 
                             "GROUP_EXT_ID": GroupNext.external_references[0]['external_id'], 
                             "Description": GroupNext.description, 
                             "Attribution": GetGroupAttribution(GroupNext.external_references[0]['external_id'],
                                                                                    GroupNext.name, 
                                                                                    GroupNext.description), 
                             "TCERT_Attribution":GetThaiCERTDataItems(GroupNext.name, 'Country')[0], 
                             "ISet_ID": GroupNext.id, 
                              "Group_Create_Date": str(GroupNext.created), 
                              "Group_Modify_Date": str(GroupNext.modified), 
                             "Revoked":False
                             }
                       
                    #alias_list=[]
                    # Build a list of aliases and update the dict
                    #for alias in GroupNext.aliases:
                        #alias_list.append(alias)
                    #dict_Group.update({"Aliases": alias_list})
                    for alias in GroupNext.aliases:
                        dict_Group_Alias={
                             "Group": GroupNext.name, 
                             "GID": GroupNext.external_references[0]['external_id'],
                             "Alias": alias
                             }
                        rows_list_Group_Aliases.append(dict_Group_Alias)
                
                    rows_list_Groups.append(dict_Group)

                if not GroupNext.revoked:                
                    #
                    #   Update references index
                    #
                
                    # Make a directory to store the pdf refs for this group
                    # Replace spaces in group names with _
                    g_doc_dir=str(os.path.dirname(ind_fname_CVE_ref) + "/" + GroupNext.name.replace(' ',  '_'))
                    g_doc_dir=str(os.path.dirname(ind_fname_CVE_ref) + "/" + GroupNext.name)
                    exists = os.path.exists(g_doc_dir)
                    if (not exists):
                        #print("Doc directory exists for " + GroupNext.name)
                        #print("x", end ="")
                    #else:
                        os.mkdir(g_doc_dir)
                    #print(" ")
                        
                    # Put the citations in a list
                    #source_list=[]
                    for next_item in GroupNext.external_references:
                        
                            
                        # Dummy lines here for the moment
                        # Add an option in param file to derive CVE
                        # Add extract line for param in main code
                        # Add parameter to init func()
                        # Add parameter to this function
                        # Add func to 'mine' text and return CVEs (from current code)
                        GetRefs="N"                        
                        if GetRefs=="Y":
                            print("GetRefs function to be added")
                            
                        CVE_List=[]
                        
                        # Dummy lines here for the moment
                        # Add an option in param file to derive CVE
                        # Add extract line for param in main code
                        # Add parameter to init func()
                        # Add parameter to this function
                        # Add func to 'mine' text and return CVEs (from current code)
                        
                        GetCVE="N"                        
                        if GetCVE=="Y":
                            print("GetCVE function to be added")
                        
                        source="NO_URL"    
                        if 'url' in next_item:
                            source=next_item.url
                        desc="NO_DESC"    
                        if 'description' in next_item:
                            desc=next_item.description
       
                        dict_Group_Ref_Line={"GID": GroupNext.external_references[0]['external_id'], 
                                 "Group": GroupNext.name, 
                                 "source_ref": source,
                                 "description": desc,
                                 "source_name": next_item.source_name,
                                 "CVE_List": CVE_List 
                                 }
                             
                        rows_list_group_refs.append(dict_Group_Ref_Line)
                 
            # Populate the Group dataframe with dict list
            self.ATTACK_GROUP_Index = self.ATTACK_GROUP_Index.append(pd.DataFrame(rows_list_Groups), sort=True)
            rows_list_Groups=[]
            # Populate the Group_aliases dataframe with dict list
            self.ATTACK_GROUP_Aliases_Index = self.ATTACK_GROUP_Aliases_Index.append(pd.DataFrame(rows_list_Group_Aliases), sort=True)
            rows_list_Group_Aliases=[]
            # Populate the CVE ref dataframe with dict list
            self.ATTACK_CVE_REF_Index = self.ATTACK_CVE_REF_Index.append(pd.DataFrame(rows_list_group_refs), sort=True)
            rows_list_group_refs=[]
            
            # STEP 4 ############################
            # Prepare an index of Relationships (and META model)
            #
            # ############################
            # 
            print("Preparing a Relationship index")
            
            Rel_list=cs.query(filter_objs_bd['relationships']
                                     )

            rows_list_Rels=[]
            rows_list_Meta=[]
            rows_list_MITIGATION=[]
            rows_list_rel_ref=[]
            rows_list_mit_ref=[]
            rows_list_TTP_TO_PROCEDURE=[]    
            rows_list_TTP_TO_DATA_COMP=[]                    
            
            #i=0
            det_count = 0
            for RelNext in Rel_list:
                #print("TTP is " + TTPNext.external_references[0]['external_id'])
                #print(str(TTPNext))
                #print("NEXT REL" + str(RelNext.id))
                #print("NEXT_REL content " + str(RelNext))
                if "https://github.com/hfiref0x/UACME" in RelNext:
                    print(str(RelNext))
                if not (RelNext.relationship_type == "revoked-by"):
                   #i=1
                   # if i==0:
                        #print("<<GROUPS>> in here")
                    source_obj=cs.get(RelNext.source_ref)
                    # Check that the source_obj has not been revoked etc
                    #  E.g. Found relationships pointing to G0118 and G0086 both revoked
                    #     intrusion-set--dc5e2999-ca1a-47d4-8d12-a6984b138a1b (G0118)
                    #          relationship--045d3fdf-933f-4c8c-ba69-f2e6f2126bd0                       
                    #     intrusion-set--7a0d4c09-dfe7-4fa2-965a-1a0e42fedd70 (G0086)
                    #          relationship--bce63312-8382-4bd2-8ce4-9405fa0d25ff
                    if 'revoked' in source_obj:
                        if source_obj['revoked'] == True:
                            continue                    
                    
                    # From 10.1 dealing with dicts for x-miter objects
                    if (isinstance(source_obj, dict)):
                        #print("Found a dict")
                        if (source_obj.get('external_references')):
                            #print("Found external references")
                            source_ext_ref=['external_references'][0]['external_id']
                        else:
                            source_ext_ref="NOT AVAILABLE"
                    else:
                        source_ext_ref=source_obj['external_references'][0]['external_id']
                            
                    #print("CHECK >> " + str(RelNext.source_ref))
                    #print("TYPE " + str(type(source_obj)))
                    #print("TYPE KEYS " + str(source_obj.keys()))
                    #print(str(source_obj))
                    #print("the id is " + str(source_obj.id))
                    #XMDC_list=cs.query(filter_objs_bd['x_mitre_data_component']
                    #                 )
                    #for XMDCNext in XMDC_list:
                    #    print("CHECK >> " + XMDCNext.id)

                    target_obj=cs.get(RelNext.target_ref)
                    # See above for target_obj
                    if 'revoked' in target_obj:
                        if target_obj['revoked'] == True:
                            continue              
                    #print("TYPE TARG " + str(type(target_obj)))
                    #print("TYPE TARG KEYS " + str(target_obj.keys()))
                    #print(str(target_obj))
                    
                    #if (i<2):
                       #print("<< REL TYPE IS " + source_obj.type)
                       #print(str(RelNext))
                       #i=i+1
                    #else:
                       #return(0)
                    
                    #ckstring=str(target_obj.external_references[0]['external_id'])
                    
                    if "description" in RelNext:
                        ThisRelDesc=RelNext.description
                    else:
                        ThisRelDesc=" "
                    
                    #if source_obj.external_references[0]['external_id'] == "G0095":
                        #if target_obj.external_references[0]['external_id'] == "T1043":
                            #print("<<GROUPS>>: " + str(RelNext))
                    dict_Rel={
                             "REL_SOURCE_ID": RelNext.source_ref, 
                             # "REL_SOURCE_TYPE": source_obj.type, -- Changed to cope with x-mitre dicts
                             "REL_SOURCE_TYPE": source_obj['type'], 
                             #"REL_SOURCE_NAME": source_obj.name, -- Changed to cope with x-mitre dicts
                             "REL_SOURCE_NAME": source_obj['name'], 
                             #"REL_SOURCE_EXT_ID": source_obj.external_references[0]['external_id'], 
                             #"REL_SOURCE_EXT_ID": source_obj['external_references'][0]['external_id'], 
                             "REL_SOURCE_EXT_ID": source_ext_ref, 
                             "REL_TARGET_ID": RelNext.target_ref, 
                             "REL_TARGET_TYPE": target_obj.type, 
                             "REL_TARGET_NAME": target_obj.name, 
                             "REL_TARGET_EXT_ID": target_obj.external_references[0]['external_id'], 
                             "REL_TYPE": RelNext.relationship_type, 
                             "REL_ID": RelNext.id, 
                             "REL_DESC": ThisRelDesc, 
                             "REL_Date_Created": RelNext.created, 
                             "REL_Date_Modified": RelNext.modified, 
                             "Revoked": RelNext.revoked
                             }
                             
                    if RelNext.relationship_type == "detects":
                        if det_count < 0:
                            det_count+=1
                            print("The detects relationship is " + str(RelNext))
                             
                    # Add rows for TTP reference table if there are external references
                    if 'external_references' in RelNext:
                        for next_item in RelNext.external_references:
                        # 
                            dict_REL_REF_Line={"REL_ID": RelNext.id, 
                                                         "source_name": "NA", 
                                                         "url": "NA",
                                                         "description": "NA"
                                                           }
                              
                                                          
                            if "description" in next_item:    
                                dict_REL_REF_Line.update({"description": next_item.description})    
                                                       
                            if "url" in next_item:    
                                dict_REL_REF_Line.update({"url": next_item.url})    
                                  
                            if "source_name" in next_item:    
                                dict_REL_REF_Line.update({"source_name": next_item.source_name})    
                             
                            rows_list_rel_ref.append(dict_REL_REF_Line)
                            
                    if target_obj.type == "attack-pattern":
                        
                        # Skiip subtechnique rels as there is (clearly) no procedure info                        
                        if not RelNext.relationship_type == "subtechnique-of":
                            if not RelNext.relationship_type == "detects":
                        
                                StripDesc=ThisRelDesc
                                                
                                dict_TTP_TO_PROCEDURE={
                                     "TTP_EXT_ID": target_obj.external_references[0]['external_id'], 
                                     "Procedure_Desc": StripDesc, 
                                     "USER_EXT_ID": source_obj.external_references[0]['external_id'], 
                                     "USER_TYPE": source_obj.type
                                     }
                             
                                rows_list_TTP_TO_PROCEDURE.append(dict_TTP_TO_PROCEDURE)
                             
                                 
                    if source_obj['type'] == "course-of-action":
                        if "description" in RelNext:
                            COA_REL_DESC_str=RelNext.description
                        else:
                            COA_REL_DESC_str=""
                        
                        dict_MITIGATION={
                             "TTP_EXT_ID": target_obj.external_references[0]['external_id'], 
                             "COA_EXT_ID": source_obj.external_references[0]['external_id'], 
                             "COA_ID": source_obj.id, 
                             "COA_NAME": source_obj.name, 
                             'COA_REL_DESC':COA_REL_DESC_str, 
                             "COA_DESC": source_obj.description, 
                             'REL_ID':RelNext.id, 
                             "created": source_obj.created, 
                             "modified":source_obj.modified
                             }
                             
                        # Add rows for TTP reference table
                        for next_item in source_obj.external_references:
                            # 
                            dict_MIT_Ref_Line={"MIT_EXT_ID": source_obj.external_references[0]['external_id'], 
                                                     "source_name": "NA", 
                                                     "url": "NA",
                                                     "external_id": "NA" , 
                                                     "description": "NA"
                                                       }
                              
                            if "external_id" in next_item: 
                                dict_MIT_Ref_Line.update({"external_id": next_item.external_id})                      
                                                       
                            if "description" in next_item:    
                                dict_MIT_Ref_Line.update({"description": next_item.description})    
                                                       
                            if "url" in next_item:    
                                dict_MIT_Ref_Line.update({"url": next_item.url})    
                                  
                            if "source_name" in next_item:    
                                dict_MIT_Ref_Line.update({"source_name": next_item.source_name})    
                             
                            rows_list_mit_ref.append(dict_MIT_Ref_Line)
                             
                        if  target_obj.external_references[0]['external_id'] == "XXXX":
                            print("The course of acton object is ")
                            print(str(source_obj))
                            print("The target object is ")
                            print(str(target_obj))
                            print("The relationship object is ")
                            print(str(RelNext))
                             
                        rows_list_MITIGATION.append(dict_MITIGATION)
                        
                    if source_obj['type'] == "x-mitre-data-component":
                        if "description" in RelNext:
                            DS_REL_DESC_str=RelNext.description
                        else:
                            DS_REL_DESC_str="NO_DESC"
                            
                        DS_obj=cs.get(source_obj['x_mitre_data_source_ref'])
                        
                        dict_TTP_TO_DATA_COMP={
                             "TTP_EXT_ID": target_obj.external_references[0]['external_id'], 
                             "DC_EXT_ID": source_ext_ref, 
                             "DC_ID": source_obj['id'], 
                             "DC_NAME": source_obj['name'], 
                             'DC_REL_DESC':DS_REL_DESC_str, 
                             "DC_DESC": source_obj['description'], 
                             'REL_ID':RelNext.id, 
                             "DC_created": source_obj['created'], 
                             "DC_modified":source_obj['modified'], 
                             "DS_ID":source_obj['x_mitre_data_source_ref'], 
                             "DS_EXT_ID":DS_obj['external_references'][0]['external_id'], 
                             "DS_NAME":DS_obj['name'], 
                             "DS_DESC":DS_obj['description'], 
                             "DS_created":DS_obj['created'], 
                             "DS_modified":DS_obj['modified']
                             }
                             
                        # Add rows for TTP reference table
                        #for next_item in source_obj.external_references:
                            # 
                            #dict_MIT_Ref_Line={"MIT_EXT_ID": source_obj.external_references[0]['external_id'], 
                            #                         "source_name": "NA", 
                            #                         "url": "NA",
                            #                         "external_id": "NA" , 
                            #                         "description": "NA"
                            #                          }
                              
                            #if "external_id" in next_item: 
                            #    dict_MIT_Ref_Line.update({"external_id": next_item.external_id})                      
                                                       
                            #if "description" in next_item:    
                            #    dict_MIT_Ref_Line.update({"description": next_item.description})    
                                                       
                            #if "url" in next_item:    
                            #    dict_MIT_Ref_Line.update({"url": next_item.url})    
                                  
                            #if "source_name" in next_item:    
                             #   dict_MIT_Ref_Line.update({"source_name": next_item.source_name})    
                             
                            #rows_list_mit_ref.append(dict_MIT_Ref_Line)
                             
                        if  target_obj.external_references[0]['external_id'] == "XXXX":
                            print("The course of action object is ")
                            print(str(source_obj))
                            print("The target object is ")
                            print(str(target_obj))
                            print("The relationship object is ")
                            print(str(RelNext))
                             
                        rows_list_TTP_TO_DATA_COMP.append(dict_TTP_TO_DATA_COMP)
                             
                    if source_obj['type'] == "attack-pattern":
                        source_sub_type ="technique"
                        #
                        #  Is this a su btechnique
                        #
                        #if source_obj.external_references[0]['external_id'].contains("."):
                            #source_sub_type = "sub-technique"
                        if "." in source_obj.external_references[0]['external_id']:
                            source_sub_type = "sub-technique"
                    else:
                        source_sub_type=source_obj['type']
                        
                    if target_obj.type == "attack-pattern":
                        target_sub_type ="technique"
                        #
                        #  Is this a su btechnique
                        #
                        #print("The target name is " + str(target_obj.name))
                        #if ckstring.contains("."):
                            #target_sub_type = "sub-technique"
                        if "." in target_obj.external_references[0]['external_id']:
                            target_sub_type = "sub-technique"
                    else:
                        target_sub_type=target_obj.type
                             
                    dict_Meta={
                                     'REL_SOURCE_TYPE':source_obj['type'], 
                                     'REL_SOURCE_SUBTYPE':source_sub_type, 
                                     'REL_TARGET_TYPE':target_obj.type, 
                                     'REL_TARGET_SUBTYPE':target_sub_type, 
                                     'REL_TYPE':RelNext.relationship_type
                    }
                       
                    #alias_list=[]
                    # Build a list of aliases and update the dict
                    #for alias in RelNext.aliases:
                        #alias_list.append(alias)
                    #dict_Rel.update({"Aliases": alias_list})
                
                    rows_list_Rels.append(dict_Rel)
                    
                    if dict_Meta not in rows_list_Meta:
                        rows_list_Meta.append(dict_Meta)
                # if not (RelNext.relationship_type == "revoked-by"):
                
            # Populate the dataframe with dict list
            self.ATTACK_REL_Index = self.ATTACK_REL_Index.append(pd.DataFrame(rows_list_Rels), sort=True)
            rows_list_Rels=[]
            # Populate the dataframe with dict list
            self.ATTACK_META_MODEL_Index = self.ATTACK_META_MODEL_Index.append(pd.DataFrame(rows_list_Meta), sort=True)
            rows_list_Meta=[]
            # Populate the dataframe with dict list
            self.ATTACK_MITIGATION_Index = self.ATTACK_MITIGATION_Index.append(pd.DataFrame(rows_list_MITIGATION), sort=True)
            rows_list_MITIGATION=[]
            # Populate the dataframe with dict list
            self.ATTACK_REL_REF_Index = self.ATTACK_REL_REF_Index.append(pd.DataFrame(rows_list_rel_ref), sort=True)
            rows_list_rel_ref=[]
            # Populate the dataframe with dict list
            self.ATTACK_MIT_REF_Index = self.ATTACK_MIT_REF_Index.append(pd.DataFrame(rows_list_mit_ref), sort=True)
            rows_list_mit_ref=[]
            # Populate the dataframe with dict list
            self.ATTACK_TTP_TO_DATA_COMP_Index = self.ATTACK_TTP_TO_DATA_COMP_Index.append(pd.DataFrame(rows_list_TTP_TO_DATA_COMP), sort=True)
            rows_list_TTP_TO_DATA_COMP=[]
            # Populate the dataframe with dict list
            self.ATTACK_TTP_TO_PROCEDURE_Index = self.ATTACK_TTP_TO_PROCEDURE_Index.append(pd.DataFrame(rows_list_TTP_TO_PROCEDURE), sort=True)
            rows_list_TTP_TO_PROCEDURE=[]     
     
            # STEP 5 ############################
            # Prepare an index of Data Sources
            #     Also create index of references for References (1 to many GID/Group)
            #
            # ############################
            # 
            print("Preparing a Data Source index")
            print("Also separate Data Source to References index ('many' for each group)")
            
            data_source_list=cs.query(filter_objs_bd['x_mitre_data_source']
                                     )
            #print("<XXX> <XXX> List of group EXT IDs")
            #for GroupNext in Group_list:
                #print("Next is " + GroupNext.external_references[0]['external_id'])
            #print("<XXX> <XXX> List done")
          
            rows_list_sources=[]
            rows_list_DS_Platforms=[]
            rows_list_DS_CollLayer=[]
            rows_list_source_refs=[]
           # i=0
            for SourceNext in data_source_list:
                #if i==0:
                    #print(str(GroupNext))
                    #i=1
                  
                #print("TTP is " + TTPNext.external_references[0]['external_id'])
                #print(str(TTPNext))
                 
                SourceNextRevoked=False
                if 'revoked' in SourceNext:
                    SourceNextRevoked = SourceNext['revoked']
                
                if not SourceNextRevoked:
                        # if i==0:
                        #print("<<GROUPS>> in here")
                        
                    dict_data_source={
                             "DS_ID": SourceNext['id'], 
                             "DS_EXT_ID": SourceNext['external_references'][0]['external_id'], 
                             "DS_NAME": SourceNext['name'], 
                             "DS_DESC": SourceNext['description'] , 
                             "DS_created": str(SourceNext['created']), 
                             "DS_modified": str(SourceNext['modified']), 
                             "Revoked":False
                             }
                             
                    rows_list_sources.append(dict_data_source)
                       
                if not SourceNextRevoked:               
                    #
                    #   Update references index
                    #
                
                    for next_item in SourceNext['external_references']:
                                                                    
                        source="NO_URL"    
                        if 'url' in next_item:
                            source=next_item['url']
                        desc="NO_DESC"    
                        if 'description' in next_item:
                            desc=next_item['description']
       
                        dict_Source_Ref_Line={"DS_EXT_ID": SourceNext['external_references'][0]['external_id'], 
                                 "DS_NAME": SourceNext['name'], 
                                 "source_ref": source,
                                 "description": desc,
                                 "source_name": next_item['source_name']
                                 }
                             
                        rows_list_source_refs.append(dict_Source_Ref_Line)
                        
                    for nextPlat in SourceNext['x_mitre_platforms']:
                        dict_DS_Platform={
                            "DS_ID": SourceNext['id'], 
                            "DS_EXT_ID": SourceNext['external_references'][0]['external_id'],
                            "DS_NAME": SourceNext['name'], 
                            "DS_PLATFORM": nextPlat
                            }
                        rows_list_DS_Platforms.append(dict_DS_Platform)
                        
                    for nextCollLayer in SourceNext['x_mitre_collection_layers']:
                        dict_DS_CollLayer={
                            "DS_ID": SourceNext['id'], 
                            "DS_EXT_ID": SourceNext['external_references'][0]['external_id'],
                            "DS_NAME": SourceNext['name'], 
                            "DS_COLL_LAYER": nextCollLayer
                            }
                        rows_list_DS_CollLayer.append(dict_DS_CollLayer)
                 
            # Populate the Group dataframe with dict list
            self.ATTACK_DATA_SOURCE_Index = self.ATTACK_DATA_SOURCE_Index.append(pd.DataFrame(rows_list_sources), sort=True)
            rows_list_sources=[]
            # Populate the source ref dataframe with dict list
            self.ATTACK_DATA_SOURCE_REF_Index = self.ATTACK_DATA_SOURCE_REF_Index.append(pd.DataFrame(rows_list_source_refs), sort=True)
            rows_list_source_refs=[]                 
            # Populate the platforms dataframe with dict list
            self.ATTACK_DATA_SOURCE_TO_PLATFORM_Index = self.ATTACK_DATA_SOURCE_TO_PLATFORM_Index.append(pd.DataFrame(rows_list_DS_Platforms), sort=True)
            rows_list_DS_Platforms=[]
            # Populate the collecton layer dataframe with dict list
            self.ATTACK_DATA_SOURCE_TO_COLL_LAYER_Index = self.ATTACK_DATA_SOURCE_TO_COLL_LAYER_Index.append(pd.DataFrame(rows_list_DS_CollLayer), sort=True)
            rows_list_DS_CollLayer=[]

            # STEP 6 ############################
            # Create binary indexes.
            #
            # ############################
            # 
            
            CreateBinIndexes()
    
            # STEP END ############################
            # Write out the Base Data Tables
            #
            print("Writing indexes")
                        
            # ## Write ATTACK_TACTIC_Index
            #
            # Delete the current index if it exists          
            exists = os.path.exists(ind_fname_tactics)
            if (exists):
                os.remove(ind_fname_tactics)
            # Write out the index dataframe to csv
            self.ATTACK_TACTIC_Index.to_csv(path_or_buf=ind_fname_tactics, index=False, 
                                             columns=['TACTIC_EXT_ID','TACTIC_ID',  'TACTIC_NAME', 
                                             'TACTIC_DESC',  'TACTIC_CREATED', 'TACTIC_MODIFIED', 
                                             'TACTIC_MATRIX',  'TACTIC_MATRIX_URL',  'LM_KILL_CHAIN_PHASE'])
                                                         
            print("Tactic index written")
            #
            # ## Write ATTACK_TTP_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_TTP)
            if (exists):
              os.remove(ind_fname_TTP)
              # Write out the index dataframe to csv
            self.ATTACK_TTP_Index.to_csv(path_or_buf=ind_fname_TTP, index=False, 
                                               columns=['TTP_Name',  'TTP_Type', 'TTP_ID',  'TTP_EXT_ID', 
                                                                   'TTP_URL', 'TTP_DESC', 'TTP_Phase', 'TTP_Domain', 'TTP_x_mitre_detection', 
                                                                   'TTP_Date_Created', 'TTP_Date_Modified','TTP_x_mitre_deprecated', 'TTP_x_mitre_is_subtechnique',
                                                                   'Revoked',  'TTP_CAR_Available'])
                                                         
            print("TTP index written") 
            #
            # ## Write ATTACK_TECH_TO_TACTIC_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_tech_to_tactic)
            if (exists):
                 os.remove(ind_fname_tech_to_tactic)
            # Write out the index dataframe to csv
            self.ATTACK_TECH_TO_TACTIC_Index.to_csv(path_or_buf=ind_fname_tech_to_tactic, index=False, 
                                             columns=['TTP_NAME', 'TTP_EXT_ID', 'KILL_CHAIN_NAME', 'TACTIC_NAME', 'TACTIC_EXT_ID',  'REL_Date_Created'])
                                                         
            print("Tech To Tactic index written") 
            #
            # ## Write ATTACK_GROUP_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_GROUPS)
            if (exists):
                 os.remove(ind_fname_GROUPS)
            # Write out the index dataframe to csv
            self.ATTACK_GROUP_Index.to_csv(path_or_buf=ind_fname_GROUPS, index=False, 
                                             columns=['Group', 'GID', 'Aliases', 'Description', 'Attribution', 'TCERT_Attribution', 
                                                                    'Group_Create_Date', 'Group_Modify_Date','ISet_ID',  'Revoked'])
                                                         
            print("Group index written") 
            
            # ## Write ATTACK_GROUP_Alias_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_ATTACK_ALIASES)
            if (exists):
                 os.remove(ind_fname_ATTACK_ALIASES)
            # Write out the index dataframe to csv
            self.ATTACK_GROUP_Aliases_Index.to_csv(path_or_buf=ind_fname_ATTACK_ALIASES, index=False, 
                                             columns=['Group', 'GID', 'Alias'])
                                                         
            print("Group alias index written") 
            #
            # ## Write ATTACK_REL_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_REL)
            if (exists):
                 os.remove(ind_fname_REL)
            # Write out the index dataframe to csv
            self.ATTACK_REL_Index.to_csv(path_or_buf=ind_fname_REL, index=False, 
                                             columns=['REL_SOURCE_ID', 'REL_SOURCE_TYPE', 'REL_SOURCE_NAME', 'REL_SOURCE_EXT_ID',
                                                           'REL_TARGET_ID', 'REL_TARGET_TYPE', 'REL_TARGET_NAME', 'REL_TARGET_EXT_ID',  "REL_TYPE", 
                                                           'REL_ID', 'REL_DESC', 'REL_Date_Created',  'REL_Date_Modified', 'Revoked'])
                                                         
            print("Relationship index written") 

            
            #
            # ## Write ATTACK_CVE_ref_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_CVE_ref)
            if (exists):
                 os.remove(ind_fname_CVE_ref)
            # Write out the index dataframe to csv
            self.ATTACK_CVE_REF_Index.to_csv(path_or_buf=ind_fname_CVE_ref, index=False, 
                                             columns=['GID', 'Group', 'source_ref',  'description', 'source_name', 'CVE_List'])
                                                         
            print("CVE reference index written") 
            
            #
            # ## Write ATTACK_TTP_REF_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_TTP_REF)
            if (exists):
                 os.remove(ind_fname_TTP_REF)
            # Write out the index dataframe to csv
            self.ATTACK_TTP_REF_Index.to_csv(path_or_buf=ind_fname_TTP_REF, index=False, 
                                             columns=['TTP_EXT_ID', 'source_name', 
                                                                    'url', 'external_id',  'description'])
                                                         
            print("TTP reference index written") 
            
            
            #
            # ## Write ATTACK_REL_REF_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_REL_REF)
            if (exists):
                 os.remove(ind_fname_REL_REF)
            # Write out the index dataframe to csv
            self.ATTACK_REL_REF_Index.to_csv(path_or_buf=ind_fname_REL_REF, index=False, 
                                             columns=['REL_ID', 'source_name', 
                                                                    'url', 'description'])
                                                         
            print("REL reference index written") 
            
            
            #
            # ## Write ATTACK_MIT_REF_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_MIT_REF)
            if (exists):
                 os.remove(ind_fname_MIT_REF)
            # Write out the index dataframe to csv
            self.ATTACK_MIT_REF_Index.to_csv(path_or_buf=ind_fname_MIT_REF, index=False, 
                                             columns=['MIT_EXT_ID', 'source_name', 
                                                                    'url', 'description'])
                                                         
            print("MIT REF reference index written") 
            
            #
            # ## Write ATTACK_TACTIC_BIN_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_TACTIC_BIN)
            if (exists):
                 os.remove(ind_fname_TACTIC_BIN)
            # Write out the index dataframe to csv
            self.ATTACK_TACTIC_BIN_Index.to_csv(path_or_buf=ind_fname_TACTIC_BIN, index=True)
                                                         
            print("ATTACK_TACTIC_BIN_Index index written") 

            #
            # ## Write ATTACK_TTP_BIN_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_TTP_BIN)
            if (exists):
                 os.remove(ind_fname_TTP_BIN)
            # Write out the index dataframe to csv
            self.ATTACK_TTP_BIN_Index.to_csv(path_or_buf=ind_fname_TTP_BIN, index=True)
                                                         
            print("ATTACK_TTP_BIN_Index index written") 
            
            #
            # ## Write ATTACK_TTP_REF_BIN_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_TTP_REF_BIN)
            if (exists):
                 os.remove(ind_fname_TTP_REF_BIN)
            # Write out the index dataframe to csv
            self.ATTACK_TTP_REF_BIN_Index.to_csv(path_or_buf=ind_fname_TTP_REF_BIN, index=True)
                                                         
            print("ATTACK_TTP_BIN_REF_Index index written") 
            
            #
            # ## Write ATTACK_META_MODEL_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_META)
            if (exists):
                 os.remove(ind_fname_META)
            # Write out the index dataframe to csv
            self.ATTACK_META_MODEL_Index.to_csv(path_or_buf=ind_fname_META, index=False, 
            columns=['REL_SOURCE_TYPE', 'REL_SOURCE_SUBTYPE', 
                            'REL_TARGET_TYPE', 'REL_TARGET_SUBTYPE', 
                            'REL_TYPE'])
                                                         
            print("ATTACK_META_MODEL_Index index written") 
             
            #
            # ## Write ATTACK_MITIGATION_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_MITIGATION  )
            if (exists):
                 os.remove(ind_fname_MITIGATION  )
            # Write out the index dataframe to csv
            self.ATTACK_MITIGATION_Index.to_csv(path_or_buf=ind_fname_MITIGATION  , index=False, 
            columns=['TTP_EXT_ID', 'COA_EXT_ID', 'COA_ID', 
                            'COA_NAME', 'COA_REL_DESC', 'COA_DESC', 'REL_ID', 
                             'created',  'modified'])
                                                         
            print("ATTACK_MITIGATION_Index index written") 
            
            #
            # ## Write ATTACK_TTP_TO_DATA_COMP_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_ATTACK_TTP_TO_DATA_COMP )
            if (exists):
                 os.remove(ind_fname_ATTACK_TTP_TO_DATA_COMP  )
            # Write out the index dataframe to csv
            self.ATTACK_TTP_TO_DATA_COMP_Index.to_csv(path_or_buf=ind_fname_ATTACK_TTP_TO_DATA_COMP  , index=False, 
            columns=['TTP_EXT_ID', 'DC_EXT_ID', 'DC_ID', 
                            'DC_NAME', 'DC_REL_DESC', 'DC_DESC', 
                            'REL_ID', 
                             'DC_created',  'DC_modified',                              
                             'DS_ID',  'DS_EXT_ID', 
                             'DS_NAME',  'DS_DESC',  
                             'DS_created',  'DS_modified'])
                                                         
            print("ATTACK_TTP_TO_DATA_COMP_Index index written") 
            
             
            #
            # ## Write ATTACK_TTP_TO_PLATFORM_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_TTP_PLAT  )
            if (exists):
                 os.remove(ind_fname_TTP_PLAT  )
            # Write out the index dataframe to csv
            self.ATTACK_TTP_TO_PLATFORM_Index .to_csv(path_or_buf=ind_fname_TTP_PLAT  , index=False, 
            columns=['TTP_EXT_ID', 'Platform'])
                                                         
            print("ATTACK_TTP_TO_PLATFORM_Index index written") 
            
            #
            # ## Write ATTACK_TTP_TO_PROCEDURE_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_TTP_TO_PROCEDURE  )
            if (exists):
                 os.remove(ind_fname_TTP_TO_PROCEDURE  )
            # Write out the index dataframe to csv
            self.ATTACK_TTP_TO_PROCEDURE_Index .to_csv(path_or_buf=ind_fname_TTP_TO_PROCEDURE  , index=False, 
            columns=['TTP_EXT_ID', 'Procedure_Desc',  'USER_EXT_ID',  'USER_TYPE'])
                                                         
            print("ATTACK_TTP_TO_PROCEDURE_Index index written") 

            
            #
            # ## Write ATTACK_TTP_TO_DATA_SOURCE_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_TTP_TO_DATA_SOURCE  )
            if (exists):
                 os.remove(ind_fname_TTP_TO_DATA_SOURCE  )
            # Write out the index dataframe to csv
            self.ATTACK_TTP_TO_DATA_SOURCE_Index .to_csv(path_or_buf=ind_fname_TTP_TO_DATA_SOURCE  , index=False, 
            columns=['TTP_EXT_ID', 'DATA_SOURCE'])
                                                         
            print("ATTACK_TTP_TO_DATA_SOURCE_Index index written") 
                      
            
            #
            # ## Write ATTACK_TTP_TO_PERMS_REQ_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_TTP_TO_PERMS_REQ  )
            if (exists):
                 os.remove(ind_fname_TTP_TO_PERMS_REQ  )
            # Write out the index dataframe to csv
            self.ATTACK_TTP_TO_PERMS_REQ_Index .to_csv(path_or_buf=ind_fname_TTP_TO_PERMS_REQ  , index=False, 
            columns=['TTP_EXT_ID', 'Permission'])
                                                         
            print("ATTACK_TTP_TO_PERMS_REQ_Index index written") 
            
            #
            # ## Write ATTACK_DATA_SOURCE_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_ATTACK_DATA_SOURCE  )
            if (exists):
                 os.remove(ind_fname_ATTACK_DATA_SOURCE  )
            # Write out the index dataframe to csv
            self.ATTACK_DATA_SOURCE_Index .to_csv(path_or_buf=ind_fname_ATTACK_DATA_SOURCE  , index=False, 
            columns=['DS_ID',  'DS_EXT_ID', 
                             'DS_NAME',  'DS_DESC',  
                             'DS_created',  'DS_modified'])
                                                         
            print("ATTACK_DATA_SOURCE_Index index written") 
            
            #
            # ## Write ATTACK_DATA_SOURCE_REF_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_ATTACK_DATA_SOURCE_REF  )
            if (exists):
                 os.remove(ind_fname_ATTACK_DATA_SOURCE_REF  )
            # Write out the index dataframe to csv
            self.ATTACK_DATA_SOURCE_REF_Index .to_csv(path_or_buf=ind_fname_ATTACK_DATA_SOURCE_REF  , index=False, 
            columns=['DS_EXT_ID', 
                             'DS_NAME',  'source_ref',  'description', 
                              'source_name'])
                                                         
            print("ATTACK_DATA_SOURCE_REF_Index index written") 
            
            #
            # ## Write ATTACK_DATA_SOURCE_TO_PLATFORM_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_ATTACK_DATA_SOURCE_TO_PLATFORM  )
            if (exists):
                 os.remove(ind_fname_ATTACK_DATA_SOURCE_TO_PLATFORM  )
            # Write out the index dataframe to csv
            self.ATTACK_DATA_SOURCE_TO_PLATFORM_Index .to_csv(path_or_buf=ind_fname_ATTACK_DATA_SOURCE_TO_PLATFORM  , index=False, 
            columns=['DS_ID', 'DS_EXT_ID', 
                             'DS_NAME',  'DS_PLATFORM'])
                                                         
            print("ATTACK_DATA_SOURCE_TO_PLATFORM_Index index written") 
            
            
            #
            # ## Write ATTACK_DATA_SOURCE_TO_COLL_LAYER_Index
            #  
            #
            # Delete the current index if it exists
            exists = os.path.exists(ind_fname_ATTACK_DATA_SOURCE_TO_COLL_LAYER  )
            if (exists):
                 os.remove(ind_fname_ATTACK_DATA_SOURCE_TO_COLL_LAYER  )
            # Write out the index dataframe to csv
            self.ATTACK_DATA_SOURCE_TO_COLL_LAYER_Index .to_csv(path_or_buf=ind_fname_ATTACK_DATA_SOURCE_TO_COLL_LAYER  , index=False, 
            columns=['DS_ID', 'DS_EXT_ID', 
                             'DS_NAME',  'DS_COLL_LAYER'])
                                                         
            print("ATTACK_DATA_SOURCE_TO_COLL_LAYER_Index index written") 
                        
            print("<< Base data indexes created >>")
            
 
            return(0)
           
            # ############################
            # END
            #    prepare base information to be linked to group descriptions later
            #      TACTIC Table
            #      TECHNIQUE TABLE 
            #       MALWARE Table
            #       TOOLS Table
            #             All linked back to TACTICs
            #              MALWARE & TOOLS linked to techniques

 
       
        ######################################
        # Init function declarations end here


        #
        # Construct the required index file names  here.
        #
        ind_fname_GROUPS=str(attack_local_file_root + ATTACK_MAIN_INDEX)
        #ind_fname_3=str(attack_local_file_root + ATTACK_SUB_INDEX)
        ind_fname_CVE_ref=str(attack_local_file_root + ATTACK_CVE_REF_INDEX)
        ind_fname_TTP=str(attack_local_file_root + ATTACK_TTP_INDEX)
        #ind_fname_6=str(attack_local_file_root + ATTACK_SUB_INDEX + "_TOOL.csv")
        ind_fname_tactics=str(attack_local_file_root + ATTACK_TACTIC_INDEX)
        ind_fname_tech_to_tactic=str(attack_local_file_root + ATTACK_TECH_TO_TACTIC_INDEX)
        ind_fname_rel=str(attack_local_file_root + ATTACK_REL_INDEX)
        ind_fname_TTP_REF=str(attack_local_file_root + "MAFpt_ATTACK_TTP_REF_Index.csv")
        ind_fname_REL_REF=str(attack_local_file_root + "MAFpt_ATTACK_REL_REF_Index.csv")
        #ATTACK_TACTIC_BIN_INDEX, 
                        #ATTACK_TTP_BIN_INDEX
        ind_fname_TACTIC_BIN=str(attack_local_file_root + ATTACK_TACTIC_BIN_INDEX)
        ind_fname_TTP_BIN=str(attack_local_file_root + ATTACK_TTP_BIN_INDEX)
        ind_fname_TTP_REF_BIN=str(attack_local_file_root + "MAFpt_ATTACK_TTP_REF_BIN_INDEX.csv")
        ind_fname_META=str(attack_local_file_root + "MAFpt_ATTACK_META_MODEL_INDEX.csv")   
        ind_fname_MITIGATION=str(attack_local_file_root + "MAFpt_ATTACK_MITIGATION_INDEX.csv")  
        ind_fname_MIT_REF=str(attack_local_file_root + "MAFpt_ATTACK_MIT_REF_Index.csv")  
        ind_fname_TTP_PLAT=str(attack_local_file_root + "MAFpt_ATTACK_TTP_TO_PLATFORM_Index.csv")   
        ind_fname_TTP_TO_PROCEDURE=str(attack_local_file_root + "MAFpt_ATTACK_TTP_TO_PROCEDURE_Index.csv") 
        ind_fname_TTP_TO_DATA_SOURCE=str(attack_local_file_root + "MAFpt_ATTACK_TTP_TO_DATA_SOURCE_Index.csv")  
        ind_fname_TTP_TO_PERMS_REQ=str(attack_local_file_root + "MAFpt_ATTACK_TTP_TO_PERMS_REQ_Index.csv")   
        ind_fname_ATTACK_ALIASES=str(attack_local_file_root + "MAFpt_ATTACK_ALIASES_Index.csv")     
        ind_fname_ATTACK_CAR_COVERAGE=str(attack_local_file_root + "MAFpt_ATTACK_CAR_COVERAGE.csv")     
        ind_fname_ATTACK_TTP_TO_DATA_COMP=str(attack_local_file_root + "MAFpt_ATTACK_TTP_TO_DATA_COMP_Index.csv")  
        ind_fname_ATTACK_DATA_SOURCE=str(attack_local_file_root + "MAFpt_ATTACK_DATA_SOURCE_Index.csv")  
        ind_fname_ATTACK_DATA_SOURCE_REF=str(attack_local_file_root + "MAFpt_ATTACK_DATA_SOURCE_REF_Index.csv")  
        ind_fname_ATTACK_DATA_SOURCE_TO_PLATFORM=str(attack_local_file_root + "MAFpt_ATTACK_DATA_SOURCE_TO_PLATFORM_Index.csv")           
        ind_fname_ATTACK_DATA_SOURCE_TO_COLL_LAYER=str(attack_local_file_root + "MAFpt_ATTACK_DATA_SOURCE_TO_COLL_LAYER_Index.csv")   
        ind_fname_ATTACK_TCERT_GROUP=str(attack_local_file_root + "MAFpt_ATTACK_TCERT_GROUP.json")    
        ind_fname_ATTACK_TCERT_TOOL=str(attack_local_file_root + "MAFpt_ATTACK_TCERT_TOOL.json")            
 
        ####################
       
        #############################################
        # Start by connecting to the TAXII server data sources.


        #ATT_SRC ="TAXII"
        #####################
        # OPTION 1: Flat files
        # create FileSystemSource
        # Based on flat files in directory structure (FileSystemSOurce sorts this out
        # e.g. ATT&CK ...../cti-master/capec|enterprise-attack|mobile-attack|pre-attack
        #             /attack-pattern|course-of-action|identity|intrusion-set|malware
        #               marking-definition|relationship|tool|x-mitre-matrix|x-mitre-tactic

        #fs = FileSystemSource("C:/Users/cjm1e17/OneDrive - University of Southampton/ATTACKTest/master/cti-master/enterprise-attack")
        #fs2 = FileSystemSource("C:/Users/cjm1e17/OneDrive - University of Southampton/ATTACKTest/master/cti-master/mobile-attack")
        #fs3 = FileSystemSource("C:/Users/cjm1e17/OneDrive - University of Southampton/ATTACKTest/master/cti-master/pre-attack")

        # Initialize dictionary to hold ATT&CK content
        attack_1 = {} # Enterprise and Mobile
        attack_2 = {} # Pre Attack
        attack_3 = {} # Pre Attack

        #####################
        # OPTION 2: Through a TAXII server
        # Instantiate server and get API Root

        # Connect to taxii server (returns list of dicts
        #   taxii_root_dict.update({'col_title': collection.title,  
        #                                    'col_id': collection.id, 
        #                                    'col_url': collection.url, 
        #                                    'tc_source': TAXIICollectionSource(taxii_collection)})
        
        taxii_root_rows=[]

        if run_download_attack == "Y":
  
          taxii_root_rows=ConnectToTAXIIServerCollections(taxii_server)
  
          Create_ATTACK_Local_Copy="N"

        else:
  
          fs = FileSystemSource(attack_local_file_root)

        # If we have been asked to download then create a composite data source
        #   from availbale collections.
        #
        # add into the CompositeDataSource
        cs1 = CompositeDataSource()
        cs2 = CompositeDataSource()
        cs3 = CompositeDataSource()
        
        # NOTE:
        # Create a list of CompositeDataSources
        # Move round with taxi root rows to improve genericness of code        

        if run_download_attack == "Y":
            #cs.add_data_sources([tc_source_enterprise,  tc_source_mobile])
            
            for row in taxii_root_rows:
                if row['col_title'].startswith("Enterprise"):
                  print("Adding " + row['col_title']  + " as data source (T1)")
                  cs1.add_data_sources([row['tc_source']])

                if row['col_title'].startswith("Mobile"):
                  print("Adding " + row['col_title']  + " as data source (T1)")
                  cs2.add_data_sources([row['tc_source']])            
          
                if row['col_title'].startswith("PRE"):
                  print("Adding " + row['col_title'] + " as data source (T2)")
                  cs3.add_data_sources([row['tc_source']])

            #cs2.add_data_sources([tc_source_preattack])
            #cs2.add_data_sources([taxii_root_rows[2]['tc_source']])
        
        # This arm not tested currently as problems with data and security software 
        #    and data on disc
        else:
            cs1.add_data_sources([fs])
            cs2.add_data_sources([fs])
            cs3.add_data_sources([fs])
  
        #############################################
        # Next create filters to help extract data by type.
        # The Enterprise/Mobile and Pre Attack have slightly different sets


        #####################
        # Enterprise and Mobile ATT&CK Filters
        #
        # # Create filters to retrieve content from Enterprise and Mobile ATT&CK based on type
        # Create dictionary to hold filters to be executed below
        filter_objs_1 = {"techniques": Filter("type", "=", "attack-pattern"),
                        "mitigations": Filter("type", "=", "course-of-action"), # Not for pre-attack
                        "groups": Filter("type", "=", "intrusion-set"),
                        "malware": Filter("type", "=", "malware"), # Not for pre-attack
                        "tools": Filter("type", "=", "tool"),
                        "relationships": Filter("type", "=", "relationship"), 
                        "x_mitre_tactic": Filter("type", "=", "x-mitre-tactic"), 
                        "x_mitre_matrix": Filter("type", "=", "x-mitre-matrix"), 
                        "x_mitre_data_component": Filter("type", "=", "x-mitre-data-component"), 
                        "x_mitre_data_source": Filter("type", "=", "x-mitre-data-source")
                     }
                     
        filter_objs_2 = {"techniques": Filter("type", "=", "attack-pattern"),
                        "mitigations": Filter("type", "=", "course-of-action"), # Not for pre-attack
                        "groups": Filter("type", "=", "intrusion-set"),
                        "malware": Filter("type", "=", "malware"), # Not for pre-attack
                        "tools": Filter("type", "=", "tool"),
                        "relationships": Filter("type", "=", "relationship"), 
                        "x_mitre_tactic": Filter("type", "=", "x-mitre-tactic"), 
                        "x_mitre_matrix": Filter("type", "=", "x-mitre-matrix") 
           #             "x_mitre_data_component": Filter("type", "=", "x-mitre-data-component"), 
           #             "x_mitre_data_source": Filter("type", "=", "x-mitre-data-source")
                     }
        
        #####################
        # Pre ATT&CK ATT&CK Filters
        #
        # Create filters to retrieve content from Pre ATT&CK ATT&CK based on type
        # Create dictionary to hold filters to be executed below
    
        filter_objs_3 = {"techniques": Filter("type", "=", "attack-pattern"),
          #                "mitigations": Filter("type", "=", "course-of-action"), # Not for pre-attack
                        "groups": Filter("type", "=", "intrusion-set"),
          #                "malware": Filter("type", "=", "malware"), # Not for pre-attack
          #              "tools": Filter("type", "=", "tool"), # Not for pre-attack
                        "relationships": Filter("type", "=", "relationship"), 
                        "x_mitre_tactic": Filter("type", "=", "x-mitre-tactic"), 
                        "x_mitre_matrix": Filter("type", "=", "x-mitre-matrix") 
           #             "x_mitre_data_component": Filter("type", "=", "x-mitre-data-component"), 
           #             "x_mitre_data_source": Filter("type", "=", "x-mitre-data-source")
                     }

        # ####################################        
        # Put base data from TAXII server into local data arrays
        #    attack, attack 2 (keys are the filter names)
        #
        BaseDataToMemory(filter_objs_1, 
                                    filter_objs_2, 
                                    filter_objs_3, 
                                    attack_1, 
                                    attack_2,
                                    attack_3, 
                                    cs1, 
                                    cs2, 
                                    cs3)
        
        # Now place the data which is in STIX format into a memory souirce
        # This is so we can use the STIX library to navigate
        print(str(datetime.datetime.now().time()) + "Building a MemorySource")
        print("Checking types ....")
        for NextOne in attack_1['techniques']:
            print("Technique type is " + str(type(NextOne)))
            break
        for NextOne in attack_1['mitigations']:
            print("Mitigation type is " + str(type(NextOne)))
            break
        for NextOne in attack_1['groups']:
            print("Group type is " + str(type(NextOne)))
            break
        for NextOne in attack_1['malware']:
            print("Malware type is " + str(type(NextOne)))
            break
        for NextOne in attack_1['tools']:
            print("Tool type is " + str(type(NextOne)))
            break
        for NextOne in attack_1['relationships']:
            print("Relationship type is " + str(type(NextOne)))
            break
        for NextOne in attack_1['x_mitre_tactic']:
            print("x_mitre_tactic type is " + str(type(NextOne)))
            break
        for NextOne in attack_1['x_mitre_matrix']:
            print("x_mitre_matrix type is " + str(type(NextOne)))
            break
        for NextOne in attack_1['x_mitre_data_component']:
            print("x_mitre_data_component is " + str(type(NextOne)))
            break
        for NextOne in attack_1['x_mitre_data_source']:
            print("x_mitre_data_source is " + str(type(NextOne)))
            break
        
        #               "x_mitre_data_component": Filter("type", "=", "x-mitre-data-component"), 
        #                "x_mitre_data_source": Filter("type", "=", "x-mitre-data-source")

        cs=MemorySource([attack_1['techniques'],
                             attack_1['mitigations'],
                             attack_1['groups'],
                             attack_1['malware'],
                             attack_1['tools'],
                             attack_1['x_mitre_tactic'], 
                             attack_1['x_mitre_matrix'], 
                             attack_1['x_mitre_data_component'], 
                             attack_1['x_mitre_data_source'], 
                             attack_1['relationships'], 
                             attack_2['techniques'],
                             attack_2['mitigations'],
                             attack_2['groups'],
                             attack_2['malware'],
                             attack_2['tools'],
                             attack_2['x_mitre_tactic'], 
                             attack_2['x_mitre_matrix'], 
                             attack_2['relationships'],                           
                             attack_3['techniques'], 
                             attack_3['groups'], 
                             #attack_2['tools'], 
                             attack_3['relationships'], 
                             attack_3['x_mitre_tactic'], 
                             attack_3['x_mitre_matrix'] 
                              ])
                              
        cs1=MemorySource([attack_1['techniques'],
                             attack_1['mitigations'],
                             attack_1['groups'],
                             attack_1['malware'],
                             attack_1['tools'],
                             attack_1['x_mitre_tactic'], 
                             attack_1['x_mitre_matrix'], 
                             attack_1['relationships'], 
                             attack_1['x_mitre_data_component'], 
                             attack_1['x_mitre_data_source']
                             ]) 

        cs2=MemorySource([attack_2['techniques'],
                             attack_2['mitigations'],
                             attack_2['groups'],
                             attack_2['malware'],
                             attack_2['tools'],
                             attack_2['x_mitre_tactic'], 
                             attack_2['x_mitre_matrix'], 
                             attack_2['relationships']
                             ])
                             
        cs3=MemorySource([                    
                             attack_3['techniques'], 
                             attack_3['groups'], 
                             #attack_2['tools'], 
                             attack_3['relationships'], 
                             attack_3['x_mitre_tactic'], 
                             attack_3['x_mitre_matrix']
                              ])
  
        #env=Environment(source=cs, 
        #                        sink=FileSystemSink("C:/Users/cjm1e17/OneDrive - University of Southampton/ATTACKTestTAXIIDataDump"))

        if Create_ATTACK_Local_Copy == "Y":
            print("Creating file sink")
            fs_sink=FileSystemSink(attack_local_file_root)

            print("<< NOT YET SUPPORTED >> Write composite data source to sink" + str(fs_sink))
            #fs_sink.add(cs.query(filter_objs['techniques']))
            #fs_sink.add(cs.query(filter_objs['mitigations']))
            #fs_sink.add(cs.query(filter_objs['groups']))
            #fs_sink.add(cs.query(filter_objs['malware']))
            #fs_sink.add(cs.query(filter_objs['tools']))
            #fs_sink.add(cs.query(filter_objs['relationships']))
            #fs_sink.add(cs.query(filter_objs['x_mitre_tactic']))
            #fs_sink.add(cs.query(filter_objs['x_mitre_matrix']))
            
        ################################################
        # The STIX2 Memorysource is now prepared
        #
        # If we need to reindex, interrogate the data to create the required indexes.

        if reindex_attack=="Y":  
            
          print(str(datetime.datetime.now().time()) + "Getting domains for techniques")
            
          TechListOfLists=TechToDomainList(filter_objs_1, 
                                                            cs1, 
                                                            cs2, 
                                                            cs3)
                                                            
    
          print(str(datetime.datetime.now().time()) + "Creating base data tables")
          
          BaseDataToTables(cs, 
                                    TechListOfLists, 
                                    ind_fname_GROUPS, 
                                    ind_fname_tactics, 
                                    ind_fname_TTP, 
                                    ind_fname_tech_to_tactic, 
                                    ind_fname_rel, 
                                    ind_fname_REL_REF, 
                                    ind_fname_CVE_ref , 
                                    ind_fname_TTP_REF, 
                                    ind_fname_TACTIC_BIN, 
                                    ind_fname_TTP_BIN, 
                                    ind_fname_TTP_REF_BIN, 
                                    ind_fname_META, 
                                    ind_fname_MITIGATION, 
                                    ind_fname_MIT_REF, 
                                    ind_fname_TTP_PLAT, 
                                    ind_fname_TTP_TO_PROCEDURE, 
                                    ind_fname_TTP_TO_DATA_SOURCE, 
                                    ind_fname_TTP_TO_PERMS_REQ, 
                                    ind_fname_ATTACK_ALIASES, 
                                    ind_fname_ATTACK_CAR_COVERAGE, 
                                    ind_fname_ATTACK_TTP_TO_DATA_COMP, 
                                    ind_fname_ATTACK_DATA_SOURCE, 
                                    ind_fname_ATTACK_DATA_SOURCE_REF, 
                                    ind_fname_ATTACK_DATA_SOURCE_TO_PLATFORM, 
                                    ind_fname_ATTACK_DATA_SOURCE_TO_COLL_LAYER, 
                                    ind_fname_ATTACK_TCERT_GROUP, 
                                    ind_fname_ATTACK_TCERT_TOOL) 
          
          return(None)
          
        else:
            #  ATTACK_META_MODEL_Index 
            #  +ATTACK_GROUP_Index
            # +ATTACK_GROUP_Aliases_Index
            # +ATTACK_TTP_Index
            # +ATTACK_TTP_TO_PLATFORM_Index
            # +ATTACK_TTP_TO_PERMS_REQ_Index
            # +ATTACK_TTP_TO_PROCEDURE_Index
            # +ATTACK_TTP_TO_DATA_SOURCE_Index
            # +ATTACK_REL_Index
            # +ATTACK_CVE_REF_Index
            # +ATTACK_TTP_REF_Index
            # +ATTACK_REL_REF_Index
            # +ATTACK_MIT_REF_Index
            # +ATTACK_TACTIC_Index
            # +ATTACK_TECH_TO_TACTIC_Index
            # ATTACK_META_MODEL_Index
            # +ATTACK_MITIGATION_Index
            # +ATTACK_TTP_TO_DATA_COMP
            # ATTACK_CAR_COVERAGE_Index  
            # +ATTACK_DATA_SOURCE_Index  
            # +ATTACK_DATA_SOURCE_REF_Index
            # +ATTACK_DATA_SOURCE_TO_PLATFORM__Index
            # +ATTACK_DATA_SOURCE_TO_COLL_LAYER_Index
            #
            # +ATTACK_TACTIC_BIN_Index
            # +ATTACK_TTP_BIN_Index
            # +ATTACK_TTP_REF_BIN_Index
            #
          print("Using previously created indexes")          
          # Load the main/group index
          #    'Revoked' is boolean, 'Aliases'  is list  
          #
          print("Loading index" + ind_fname_GROUPS)
          exists = os.path.exists(ind_fname_GROUPS)
          if (exists):
              self.ATTACK_GROUP_Index=pd.read_csv(ind_fname_GROUPS, 
                                                           header=0)
              print("Index loaded" + str(self.ATTACK_GROUP_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_GROUP_Index.info()
              
          # Load the TTP index
          #    boolean - TTP_x_mitre_deprecated ,  TTP_x_mitre_is_subtechnique. Revoked     
          #   
          print("Loading index" + ind_fname_TTP)
          exists = os.path.exists(ind_fname_TTP)
          if (exists):
              self.ATTACK_TTP_Index=pd.read_csv(ind_fname_TTP, header=0, 
                                                                  converters={"TTP_Phase": lambda x:x.strip("[]").split(", ")})
              print("Index loaded " + str(self.ATTACK_TTP_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_TTP_Index.info()

          # Load the REL index
          #    boolean -  Revoked     
          #   
          print("Loading index" + ind_fname_rel)
          exists = os.path.exists(ind_fname_rel)
          if (exists):
              self.ATTACK_REL_Index=pd.read_csv(ind_fname_rel, 
                                                                 header=0)
              print("Index loaded " + str(self.ATTACK_REL_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_REL_Index.info()


          # Load the CVE_REF index
          #    List -  CVE_List  
          #                  
          print("Loading index" + ind_fname_CVE_ref)
          exists = os.path.exists(ind_fname_CVE_ref)
          if (exists):
              self.ATTACK_CVE_REF_Index=pd.read_csv(ind_fname_CVE_ref, header=0)
              print("Index loaded " + str(self.ATTACK_CVE_REF_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_CVE_REF_Index.info()

          # Load the TTP_REF index
          #   
          #                  
          print("Loading index" + ind_fname_TTP_REF)
          exists = os.path.exists(ind_fname_TTP_REF)
          if (exists):
              self.ATTACK_TTP_REF_Index=pd.read_csv(ind_fname_TTP_REF, header=0)
              print("Index loaded " + str(self.ATTACK_TTP_REF_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_TTP_REF_Index.info()     


          # Load the REL_REF index
          #   
          #                  
          print("Loading index" + ind_fname_REL_REF)
          exists = os.path.exists(ind_fname_REL_REF)
          if (exists):
              self.ATTACK_REL_REF_Index=pd.read_csv(ind_fname_REL_REF, header=0)
              print("Index loaded " + str(self.ATTACK_REL_REF_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_REL_REF_Index.info()  
       

          # Load the MIT_REF index
          #   
          #                  
          print("Loading index" + ind_fname_MIT_REF)
          exists = os.path.exists(ind_fname_MIT_REF)
          if (exists):
              self.ATTACK_MIT_REF_Index=pd.read_csv(ind_fname_MIT_REF, header=0)
              print("Index loaded " + str(self.ATTACK_MIT_REF_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_MIT_REF_Index.info()     
          
          # Load the TACTIC index
          #   
          #                       
          print("Loading index" + ind_fname_tactics)                    
          exists = os.path.exists(ind_fname_tactics)
          if (exists):
              self.ATTACK_TACTIC_Index=pd.read_csv(ind_fname_tactics, 
                                                                     header=0, 
                                                                     converters={"LM_KILL_CHAIN_PHASE": lambda x:x.strip("[]").split(", ")})
              print("Index loaded " + str(self.ATTACK_TACTIC_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_TACTIC_Index.info()

          # Load the TECH TO TACTIC index
          #   
          #              
          print("Loading index" + ind_fname_tech_to_tactic)                    
          exists = os.path.exists(ind_fname_tech_to_tactic)
          if (exists):
              self.ATTACK_TECH_TO_TACTIC_Index=pd.read_csv(ind_fname_tech_to_tactic, header=0)
              print("Index loaded " + str(self.ATTACK_TECH_TO_TACTIC_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_TECH_TO_TACTIC_Index.info()
          
          # Load the ATTACK_TACTIC_BIN_Index index
          #   
          #              
          print("Loading index" + ind_fname_TACTIC_BIN)                    
          exists = os.path.exists(ind_fname_TACTIC_BIN)
          if (exists):
              self.ATTACK_TACTIC_BIN_Index=pd.read_csv(ind_fname_TACTIC_BIN, index_col=0,  header=0)
              print("Index loaded " + str(self.ATTACK_TACTIC_BIN_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_TACTIC_BIN_Index.info()
          
          # Load the ATTACK_TTP_BIN_Index index
          #   
          #              
          print("Loading index" + ind_fname_TTP_BIN)                    
          exists = os.path.exists(ind_fname_TTP_BIN)
          if (exists):
              self.ATTACK_TTP_BIN_Index=pd.read_csv(ind_fname_TTP_BIN, index_col=0,  header=0)
              print("Index loaded " + str(self.ATTACK_TTP_BIN_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_TTP_BIN_Index.info()
          
          # Load the ATTACK_TTP_REF_BIN_Index index
          #   
          #              
          print("Loading index" + ind_fname_TTP_REF_BIN)                    
          exists = os.path.exists(ind_fname_TTP_REF_BIN)
          if (exists):
              self.ATTACK_TTP_REF_BIN_Index=pd.read_csv(ind_fname_TTP_REF_BIN, index_col=0,  header=0)
              print("Index loaded " + str(self.ATTACK_TTP_REF_BIN_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_TTP_REF_BIN_Index.info()
          
          # Load the ATTACK_MITIGATION_Index index
          #   
          #              
          print("Loading index" + ind_fname_MITIGATION)                    
          exists = os.path.exists(ind_fname_MITIGATION)
          if (exists):
              self.ATTACK_MITIGATION_Index=pd.read_csv(ind_fname_MITIGATION, header=0)
              print("Index loaded " + str(self.ATTACK_MITIGATION_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_MITIGATION_Index.info()
          
          # Load the ATTACK_TTP_TO_DATA_COMP_Index index
          #   
          #              
          print("Loading index" + ind_fname_ATTACK_TTP_TO_DATA_COMP)                    
          exists = os.path.exists(ind_fname_ATTACK_TTP_TO_DATA_COMP)
          if (exists):
              self.ATTACK_TTP_TO_DATA_COMP_Index=pd.read_csv(ind_fname_ATTACK_TTP_TO_DATA_COMP, header=0)
              print("Index loaded " + str(self.ATTACK_TTP_TO_DATA_COMP_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_TTP_TO_DATA_COMP_Index.info()
          
          
          # Load the ATTACK_TTP_TO_PLATFORM_Index index
          #   
          #              
          print("Loading index" + ind_fname_TTP_PLAT)                    
          exists = os.path.exists(ind_fname_TTP_PLAT)
          if (exists):
              self.ATTACK_TTP_TO_PLATFORM_Index=pd.read_csv(ind_fname_TTP_PLAT, header=0)
              print("Index loaded " + str(self.ATTACK_TTP_TO_PLATFORM_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_TTP_TO_PLATFORM_Index.info()
         
         
          # Load the ATTACK_TTP_TO_PROCEDURE_Index index
          #   
          #              
          print("Loading index" + ind_fname_TTP_TO_PROCEDURE)                    
          exists = os.path.exists(ind_fname_TTP_TO_PROCEDURE)
          if (exists):
              self.ATTACK_TTP_TO_PROCEDURE_Index=pd.read_csv(ind_fname_TTP_TO_PROCEDURE, header=0)
              print("Index loaded " + str(self.ATTACK_TTP_TO_PROCEDURE_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_TTP_TO_PROCEDURE_Index.info()
          
    
          # Load the ATTACK_TTP_TO_DATA_SOURCE_Index index
          #   
          #              
          print("Loading index" + ind_fname_TTP_TO_DATA_SOURCE)                    
          exists = os.path.exists(ind_fname_TTP_TO_DATA_SOURCE)
          if (exists):
              self.ATTACK_TTP_TO_DATA_SOURCE_Index=pd.read_csv(ind_fname_TTP_TO_DATA_SOURCE, header=0)
              print("Index loaded " + str(self.ATTACK_TTP_TO_DATA_SOURCE_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_TTP_TO_DATA_SOURCE_Index.info()

        
         #ind_fname_ATTACK_CAR_COVERAGE=str(attack_local_file_root + "MAFpt_ATTACK_CAR_COVERAGE.csv")       
        
          # Load the ATTACK_TTP_TO_PERMS_REQ_Index
          #
          #
          print("Loading index" + ind_fname_ATTACK_ALIASES)                    
          exists = os.path.exists(ind_fname_ATTACK_ALIASES)
          if (exists):
              self.ATTACK_TTP_TO_PERMS_REQ_Index=pd.read_csv(ind_fname_ATTACK_ALIASES, header=0)
              print("Index loaded " + str(self.ATTACK_TTP_TO_PERMS_REQ_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_TTP_TO_PERMS_REQ_Index.info()
          
          # Load the ATTACK_GROUP_Aliases_Index
          #
          #
          print("Loading index" + ind_fname_TTP_TO_PERMS_REQ)                    
          exists = os.path.exists(ind_fname_TTP_TO_PERMS_REQ)
          if (exists):
              self.ATTACK_TTP_TO_PERMS_REQ_Index=pd.read_csv(ind_fname_TTP_TO_PERMS_REQ, header=0)
              print("Index loaded " + str(self.ATTACK_TTP_TO_PERMS_REQ_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_TTP_TO_PERMS_REQ_Index.info()
          
          # Load the ATTACK_CAR_COVERAGE_Index  
          #
          #
          print("Loading index" + ind_fname_ATTACK_CAR_COVERAGE)                    
          exists = os.path.exists(ind_fname_ATTACK_CAR_COVERAGE)
          if (exists):
              self.ATTACK_CAR_COVERAGE_Index=pd.read_csv(ind_fname_ATTACK_CAR_COVERAGE, header=0)
              print("Index loaded " + str(self.ATTACK_CAR_COVERAGE_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_CAR_COVERAGE_Index.info()        
        
        
          # Load the ATTACK_DATA_SOURCE_Index  
          #
          #
          print("Loading index" + ind_fname_ATTACK_DATA_SOURCE)                    
          exists = os.path.exists(ind_fname_ATTACK_DATA_SOURCE)
          if (exists):
              self.ATTACK_DATA_SOURCE_Index=pd.read_csv(ind_fname_ATTACK_DATA_SOURCE, header=0)
              print("Index loaded " + str(self.ATTACK_DATA_SOURCE_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_DATA_SOURCE_Index.info()   
        
          # Load the ATTACK_DATA_SOURCE_REF_Index  
          #
          #
          print("Loading index" + ind_fname_ATTACK_DATA_SOURCE_REF)                    
          exists = os.path.exists(ind_fname_ATTACK_DATA_SOURCE_REF)
          if (exists):
              self.ATTACK_DATA_SOURCE_REF_Index=pd.read_csv(ind_fname_ATTACK_DATA_SOURCE_REF, header=0)
              print("Index loaded " + str(self.ATTACK_DATA_SOURCE_REF_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_DATA_SOURCE_REF_Index.info()  
       
        
          # Load the ATTACK_DATA_SOURCE_TO_PLATFORM_Index  
          #
          #
          print("Loading index" + ind_fname_ATTACK_DATA_SOURCE_TO_PLATFORM)                    
          exists = os.path.exists(ind_fname_ATTACK_DATA_SOURCE_TO_PLATFORM)
          if (exists):
              self.ATTACK_DATA_TO_PLATFORM_Index=pd.read_csv(ind_fname_ATTACK_DATA_SOURCE_TO_PLATFORM, header=0)
              print("Index loaded " + str(self.ATTACK_DATA_SOURCE_TO_PLATFORM_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_DATA_SOURCE_TO_PLATFORM_Index.info()    
          
        
          # Load the ATTACK_DATA_SOURCE_TO_COLL_LAYER_Index  
          #
          #
          print("Loading index" + ind_fname_ATTACK_DATA_SOURCE_TO_COLL_LAYER)                    
          exists = os.path.exists(ind_fname_ATTACK_DATA_SOURCE_TO_COLL_LAYER)
          if (exists):
              self.ATTACK_DATA_SOURCE_TO_COLL_LAYER_Index=pd.read_csv(ind_fname_ATTACK_DATA_SOURCE_TO_COLL_LAYER, header=0)
              print("Index loaded " + str(self.ATTACK_DATA_SOURCE_TO_COLL_LAYER_Index.shape[0]))
          else:
              print("No index found, rerun with reindex parameter set")
          self.ATTACK_DATA_SOURCE_TO_COLL_LAYER_Index.info()    
          
          return(None)
              
        #####################################################
        # End of main index initialise

        ###########################################
        # Using the indexes to get information

        if CVE_SEARCH == "Y":
          group_list=cs.query(filter_objs['groups'])
          #gnames=["APT37"]

          i=0
          for Agrp in group_list:
          #for gname in gnames:
            i=i+1
            gname=Agrp.name
  

            print("<< " +str(i) + " >> Search for CVE for group " + gname + 
                        " ID " + Agrp.external_references[0]['external_id'])

            # Focus on an individual group                
            #if Agrp.external_references[0]['external_id'] != "G0004":
                #continue
                
            # Make a directory to store the pdf refs for this group
            g_doc_dir=str(attack_local_file_root + gname)
            exists = os.path.exists(g_doc_dir)
            if (exists):
                print("Doc directory exists for " + gname)
            else:
                os.mkdir(g_doc_dir)
      
            # Find the GID associated with this group (use in output index below)
            this_gid=Agrp.external_references[0]['external_id']

            # Put the citations in a list
            source_list=[]
            for next_item in Agrp.external_references:
              if 'url' in next_item:
                source_list.append(next_item.url)
    
            #print(str(source_list))

            #TTPSet=GetGroupTTP(gname)

            #print(str(TTPSet[['TTP_EXT_ID',  'REL_Date_Created']]))
            #print(str(datetime.datetime.now().time()) + "Found relationships")

            rows_list_4=[]
            for source in source_list:
      
              UAS = ("Mozilla/5.0 (Windows NT 6.1; WOW64; rv:40.0) Gecko/20100101 Firefox/40.1", 
               "Mozilla/5.0 (Windows NT 6.3; rv:36.0) Gecko/20100101 Firefox/36.0",
               "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_10; rv:33.0) Gecko/20100101 Firefox/33.0",
               "Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/41.0.2228.0 Safari/537.36",
               "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_10_1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/41.0.2227.1 Safari/537.36",
               "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/41.0.2227.0 Safari/537.36",
                 )

              ua = UAS[random.randrange(len(UAS))]

              headers = {'user-agent': ua}
      
              CVE_List=[]
      
              dict4={"GID": this_gid, 
                       "Group": Agrp.name, 
                       "source_ref": source,
                       "CVE_List": [] 
                       }

              # This cannot deal with all sites
              # data = urllib.request.urlopen(str(source_list[1]))
    
              # This while loop is just here to break out of and execute rows_list 4 append block for all source rows
              # continue and break equivalent here as only one loop is actioned
              dowhile=True
              while dowhile:
        
                dowhile=False
      
                if source.endswith(".pdf"):
                  # Just waiting a little while to show not malicious

                  ind_fname=str(g_doc_dir + "/" + os.path.basename(source)) 
                  # Get a streamed connection
                  try:
                      # Headers used from above
                      this_pdf = requests.get(source,  stream=True,  headers=headers)
                      #print("Status code " + str(this_pdf.status_code))
                      #print("Headers " + str(this_pdf.headers))
                      #print("Content " + str(this_pdf.content))
                      #urllib.request.urlretrieve(source, ind_fname)
                  except Exception as excp:
                      print("requests.get: Failed access to " + source)
                      print("Messages " + str(excp))
                      CVE_List.append("requests.get: Failed access to " + source + "Messages " + str(excp))
                      continue
          
                  # End the try except block for requests.get
      
                  # Read the stream connection (chunk by chunk) and write to disc (used later).
                  try:
                      with open(ind_fname, 'wb') as f:
                          # Have not used this_pdf.text as potential binary response
                          # Maybe big pdfs so have chunked stream (chunk slighly arbitrarily 2000b)
                          for this_chunk in this_pdf.iter_content(2000):
                              f.write(this_chunk)
                      #urllib.request.urlretrieve(source, ind_fname)
                  except IOError as excp:
                      print("I/O error({0}): {1}".format(excp.errno, excp.strerror))
                      #print("urllib.request.urlretrieve: Failed access to " + source)
                      #print("Messages " + str(excp))
                      CVE_List.append("open/write: Failed access to " + ind_fname + "I/O error({0}): {1}".format(excp.errno, excp.strerror))
                      continue
                  except: # Handle other exceptions such as attribute errors
                      print("Unexpected error (open/write): " + ind_fname + " : ", sys.exc_info()[0])
                      CVE_List.append("open/write: Failed access to " + ind_fname + " : " + sys.exc_info()[0])

                  # End the try except block for open/write pdf
      
                  # Search the downloaded pdf
                  try:
                    object = PyPDF2.PdfFileReader(ind_fname)
                  except:
                    print("PyPDF2.PdfFileReader: Failed access to " + ind_fname)
                    CVE_List.append("PyPDF2.PdfFileReader: Failed access to " + ind_fname)
                    continue
        
                  # End the try except block for PyPDF2.PdfFileReader

                  # get number of pages
                  NumPages = object.getNumPages()

                  # extract text and do the search
                  for p_num in range(0, NumPages):
                    try:
                      PageObj = object.getPage(p_num)
                    except:
                      print("object.getPage: Failed access to " + ind_fname + " Page " + str(p_num))
                      CVE_List.append("object.getPage: Failed access to " + ind_fname + " Page " + str(p_num))
                      continue

                    # End the try except block for object.getPage(p_num)
            
                    try:
                      Text = PageObj.extractText() 
                    except:
                      print("PageObj.extractText: Failed access to " + ind_fname + " Page " + str(p_num))
                      CVE_List.append("PageObj.extractText: Failed access to " + ind_fname + " Page " + str(p_num))
                      continue
          
                    # End the try except block for PageObj.extractText()
        
                    # Look for the CVEs        
                    ResSearch=re.findall('CVE-[0-9][0-9][0-9][0-9]-[0-9][0-9][0-9][0-9]', Text)
                    for res in ResSearch:
                        CVE_List.append(res)
            
                  # End for p_num in range(0, NumPages):

                # End if source.endswith(".pdf"):
    
                else:
                    try:
                        r=requests.get(source)
                    except:
                        print("requests.get: Failed to access " + ind_fname)
                        CVE_List.append("requests.get: Failed to access " + ind_fname)
                        continue
                    if r.status_code == 200:
                        ResSearch=re.findall('CVE-[0-9][0-9][0-9][0-9]-[0-9][0-9][0-9][0-9]', r.text)
                        for res in ResSearch:
                          CVE_List.append(res)
              
                # End else (from if source.endswith(".pdf"):)

              # End while(dowhile): 
              # (only one loop actioned to provide break out mech to execute below for all source rows)
    
              # Create a lit of unique items in CVE list
              CVE_Set=set(CVE_List)
              CVE_List_Unique=list(CVE_Set)
              # Sort the list and update the dict
              CVE_List_Unique.sort()
              dict4.update({'CVE_List': CVE_List_Unique})
              # Append the dict to the list of dicts
              rows_list_4.append(dict4)
    
            # End for source in source_list:
  
            self.ATTACK_CVE_REF_Index = self.ATTACK_CVE_REF_Index.append(pd.DataFrame(rows_list_4), sort=True)
            rows_list_4=[]
  
          # End for Agrp in group_list:  

          print("Writing index to " + ind_fname_4)

          # Delete the current index if it exists
          exists = os.path.exists(ind_fname_4)
          if (exists):
            os.remove(ind_fname_4)
  
          # Write out the index dataframe to csv
          self.ATTACK_CVE_REF_Index.to_csv(path_or_buf=ind_fname_4, index=False, 
                                          columns=['GID',  'Group', 'source_ref',  'CVE_List'])
                                        
          print("CVE ref index written")  

        print("Windows Defender sees this as PHP Backdoor risk")
        attack_pattern=cs.get('attack-pattern--c16e5409-ee53-4d79-afdc-4099dc9292df')
        print(attack_pattern)  
     
    # End def __init__()
       
    def GetTechForTTP(self, 
                                    TTP_EXT_ID):
        
        # Get TTP details from the main index
        
        TTP_Tech=[]
    
        if (self.ATTACK_TTP_Index['TTP_EXT_ID'] == TTP_EXT_ID).any():
            TTP_Tech=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID'] == TTP_EXT_ID)]['TTP_Phase'].tolist()[0]
        else:
            if self.Verbose:
                print("Missing TTP ID: " + TTP_EXT_ID)
        
        # Load converter reestablishes list but does not strip out additional "'", so strip here.
        # Will eventually replace this list implmn with a new table - primary key (TTP_EXT_ID, Technique)
        # See above.
        # self.ATTACK_TTP_Index=pd.read_csv(ind_fname_5, header=0, 
        #            converters={"TTP_Phase": lambda x:x.strip("[]").split(", ")})
        
        TTP_Tech_strip=[]        
        for NextTech in TTP_Tech:
            TTP_Tech_strip.append(NextTech.strip("'"))
            
        return(TTP_Tech_strip)
                        
# ###########
#  Rewriiten for new tables
#
#    def GetListOfGroups(self) - return(Group_list)
#    def GetGroupName(self, 
#                               GID) - return(str(GroupNameStr))
#    def GetGroupAttribution(self, 
#                               GroutAttribution) - return(str(GroutAttributionStr))
#     def GetListOfTactics(self) - return(Tactic_list)
#     def GetListOfTacticsInDomain(self, 
#                       [enterprise-attack, mobile-attack, pre-attack] DomainName) - return(Tactic_list)
#    def GetListOfKCPhasesForTactic(self, 
#                                                    tactic_ext_id): - return(KC_Phase_list)
#    def GetListOfTacticsWithMatrix(self) - return(ListOfDicts)
#                                                               dict_row={TACTIC_EXT_ID"], 
#                                                                              "TACTIC_MATRIX"]}
#    def GetListOfTTP(self) - return(TTP_list)
#    def GetListOfTTPInDomain(                self, 
#         [Enterprise|Mobile|Pre-ATT&CK ]    domain) - return(TTP_list)
#    def GetListOfTTPOfType(               self, 
#         [attack-pattern|malware|tool]     type) - return(TTP_list)
#    def GetTTPForGROUP(self, 
#                                    group_name) - return(TTPCurrentList)
#    def GetTTPForGROUPByLevel(self, 
#                                    group_name,
#        [ 'Top'|'Sub']     Level,
#         ["Y"|"N"]               GetTopFromSub) - return(TTPCurrentList)
#    def GetTTPDomain(self, 
#                                ThisTTP) - return(Domain)
#    def GetTTPType(self, 
#                                ThisTTP) - return(Type)  [attack-pattern|malware|tool] 
#    def GetTTPLevel(self, 
#                                ThisTTP) - return(Level) [ 'Top' or 'Sub'] 
#    def GetTacticsForGROUP(self, 
#                                        group_name) - return(GroupTacticList)
#    
#    def GetTacticsForTTP(self, 
#                               ThisTTP) - return(TacticList)        
#    def GetTacticName(self, 
#                                 Tactic_Ext_Id) - return(TacticName)  
#    def GetTacticDomain(self, 
#                                   Tactic_Ext_Id) - return(TacticDomain) 
#                                            return(enterprise-attack|mobile-attack|pre-attack)
#    def GetSchema(self) - prints out dataframe structures
#
#    def GetTOOL_TTPDetails(self, 
#                                            TOOL_id)
#        return - TTP_Details=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID'] == TTP_id)].to_dict(orient='records')[0])
#    def GetTTPDetails(self, 
#                                    TTP_id)
#        return - TTP_Details=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID'] == TTP_id)].to_dict(orient='records')[0]
#    def GetGROUPDetails(self, 
#                                    GROUP_id):
#        return - GROUP_Details=self.ATTACK_Index[(self.ATTACK_Index['Group'] == GROUP_id)].to_dict(orient='records')[0]    
#    def GetGroupTTPRelDate(self, 
#                                         GROUP_NAME, 
#                                         TTP) - return(rel_date)
#    def GetToolTTPRelDate(self, 
#                                        Tool_TTP, 
#                                       TTP) - return(rel_date)
#    def SelectTOOL_TTP(self, 
#                                    TTPList) - return(TOOL_TTPList)
#    def GetTTPForTOOL(self, 
#                                   tool_name):- return(TTP_List)
#    def GetGROUPForTTP(self,  TTP_id):
#                             return(Group_List_Unique)
#    def GetATTACK_TTP_BIN_TOTALS(self):
#                             return(SumSeries)
#    def GetATTACK_TTP_BIN_TOTALS_MATRIX(self, 
#                                                                           Matrix):
#                              return(SumSeries)
#    def GetATTACK_TTP_BIN_TOTALS_MATRIX_TACTIC(self, 
#                                                                           Matrix, 
#                                                                           Tactic):
#                               return(SumSeries)
#    def GetReportsForTech(self, 
#                                          TTP): - return(TTP_List)
#    def GetMitigationsForTech(self, 
#                                          TTP): - return(TTP_List)
#    def GetTechForRef(self, 
#                                          REF): - return(TTPList)
#    def GetTacticForRef(self, 
#                                          REF): - return(TacticList)
#    def GetTTPRefCounts(self, 
#                                          ): - return(Series)
#    def GetListOfTTPReferences(self
#                                            ):- return(TTP_URL_list)
#     def GetTacticForRef(self, 
#                                          REF):  -  return(Tactic_list)
#     def GetProcTextForTTP(self, 
#                                          TTP_EXT_ID): - return(ProcTextList
#     def GetTechForTactic(self, 
#                                          Tactic):
#                                         return(TTPList)
#     def GetMainTechForTactic(self, 
#                                          Tactic):
#                                         return(TTPList)
#      def GetRelLineDetails(self, 
#                                          REL_SOURCE_EXT_ID, 
#                                          REL_TARGET_EXT_ID, 
#                                          REL_TYPE): 
#                                                      return(ThisRELDict)
#    def GetTacticDetails(self, 
#                                          TACTIC_EXT_ID):
#                                                       return(ThisTacticDict)
#  def GetGroupForTTP(self, 
#                                          TTP_EXT_ID):
#                                                          return(GroupList)
#   def GetGroupTechChain(self, 
#                                          GID):    
#                                                          return(List of dicts{TacticId, TechList})
#  def GetTechForSub(self, 
#                                          TTP_EXT_ID):
#                                                          return(ParentTech)  
#  def GetRefsForGroupTech(self, 
#                                          GID, 
#                                          TTP_EXT_ID): 
#                                                       return(ParentTech)
#  def GetTechForTactic(self, 
#                                          Tactic)
#
    def GetListOfGroups(self):
        
        # Return a list of all groups
        
        Group_list=self.ATTACK_GROUP_Index['GID'].tolist()
        
        return(Group_list)
        
    def GetGroupName(self, 
                                   GID):
        
        # Return name of group from GID
        # GID not validated
        
        #print("In GetGroupName " + str(GID))
        
        # Returns a Panda Series
        GroupName=self.ATTACK_GROUP_Index[(self.ATTACK_GROUP_Index['GID']  == GID)]['Group']
       
        # Should only be one row but fo safety take top row only 
        for Group in GroupName:
            GroupNameStr=str(Group)
            break
            
        return(str(GroupNameStr))
        
    def GetGroupAttribution(self, 
                                   GName):
        
        # Return name of group from GID
        # GID not validated
        
        #print("In GetGroupName " + str(GID))
        
        # Returns a Panda Series
        GroupAttribution=self.ATTACK_GROUP_Index[(self.ATTACK_GROUP_Index['Group']  == GName)]['Attribution']
       
        # Should only be one row but fo safety take top row only 
        for GroupAttr in GroupAttribution:
            GroupAttributionStr=str(GroupAttr)
            break
            
        return(str(GroupAttributionStr))
        
    def GetListOfTactics(self):
        
        # Return a list of all tactics
        
        Tactic_list=self.ATTACK_TACTIC_Index['TACTIC_EXT_ID'].tolist()
        
        return(Tactic_list)
        
    def GetListOfTacticsInDomain(self, 
                                                     DomainName):
        
        # Return a list of all tactics
        
        Tactic_list=self.ATTACK_TACTIC_Index[self.ATTACK_TACTIC_Index['TACTIC_MATRIX'] == DomainName]['TACTIC_EXT_ID'].tolist()
        
        return(Tactic_list)
        
    def GetListOfKCPhasesForTactic(self, 
                                                    tactic_ext_id):
        
        # Return a list of all groups
        
        KC_Phase_list=self.ATTACK_TACTIC_Index[self.ATTACK_TACTIC_Index['TACTIC_EXT_ID'] == tactic_ext_id ]['LM_KILL_CHAIN_PHASE']
        
        #print("<<GETLIST: The list is " + str(KC_Phase_list))
        
        # strip() required from converters for lists from csv  read
        for nextList in KC_Phase_list:
            #print("<<GETLIST: The list is " + str(nextList))
            ret_list=[]
            for next_item in nextList:
                ret_list.append(next_item.strip("'"))
            
            break
        
        return(ret_list)
        
    def GetListOfTacticsWithMatrix(self):
        
        # Return a list of all groups
        
        ListOfDicts=[]
        
        TacticSubDF=self.ATTACK_TACTIC_Index[['TACTIC_EXT_ID',  'TACTIC_MATRIX']]
        
        for Index,  ThisRow in TacticSubDF.iterrows():
            dict_row={
                "TACTIC_EXT_ID": ThisRow['TACTIC_EXT_ID'], 
                "TACTIC_MATRIX": ThisRow['TACTIC_MATRIX']
                    } 
                  
            ListOfDicts.append(dict_row)
        
        return(ListOfDicts)
        
    def GetListOfTTP(self):
        
        # Return a list of all TTP
        
        TTP_list=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['Revoked'] == False) &
                                                      (self.ATTACK_TTP_Index['TTP_x_mitre_deprecated'] == False )]['TTP_EXT_ID'].tolist()
        
        return(TTP_list)
        
    def GetListOfSelectedTTP(self, 
                                        domain, 
                                        level, 
                                        type):
        
        # Return a list of all TTP based on selection
        #     domain "Any"|"Enterprise"|"Mobile"|"Pre-ATT&CK"
        #     level "Any"|"Top"|"Sub"
        #     type "attack-pattern"|"malware"|"tool" 
        #          only
        #     Consider self.ATTACK_TTP_Index['TTP_Type'].str.match(type) to use wildcards
        
        # Select by domain
        if domain == "Any":
            TTPDomDF=self.ATTACK_TTP_Index
        else:
            TTPDomDF=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_Domain'] == domain)]
        
        # Further select by  level
        if level == "Any":
            TTPLevDF=TTPDomDF
        else:
            if level == "Sub":
                LevSub=True
            else:
                LevSub=False
            TTPLevDF=TTPDomDF[(TTPDomDF['TTP_x_mitre_is_subtechnique'] == LevSub)]
         
        # Further select by  type
        if type == "Any":
            TTPTypeDF=TTPLevDF[(TTPLevDF['TTP_x_mitre_deprecated'] == False ) &
                                                      (TTPLevDF['Revoked'] == False )]
        else:
            TTPTypeDF=TTPLevDF[(TTPLevDF['TTP_Type'] == type) &
                                                      (TTPLevDF['TTP_x_mitre_deprecated'] == False ) &
                                                      (TTPLevDF['Revoked'] == False )]
            
        #TTP_list=TTPTypeDF[(TTPTypeDF['TTP_Type'] == type ) &
        #                                              (TTPTypeDF['TTP_x_mitre_deprecated'] == False ) &
        #                                              (TTPTypeDF['Revoked'] == False )]['TTP_EXT_ID'].tolist()
            
        TTP_list=TTPTypeDF['TTP_EXT_ID'].tolist()
                                                    
        return(TTP_list)
        
    def GetListOfTTPInDomain(self, 
                                            domain):
        
        # Return a list of TTP in domain
        # One of:
        #      Enterprise
        #      Mobile
        #       Pre-ATT&CK
        
        TTP_list=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_Domain'] == domain) &
                                                       (self.ATTACK_TTP_Index['Revoked'] == False) &
                                                       (self.ATTACK_TTP_Index['TTP_x_mitre_deprecated'] == False )]['TTP_EXT_ID'].tolist()
        
        return(TTP_list)
        
    def GetListOfTTPOfType(self, 
                                        type):
        
        # Return a list of all groups
        # Type is not checked so do not use with invalid types at the moment
        #      expects "attack-pattern" or "malware" or "tool" only
        
        TTP_list=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_Type'] == type ) &
                                                      (self.ATTACK_TTP_Index['TTP_x_mitre_deprecated'] == False ) &
                                                      (self.ATTACK_TTP_Index['Revoked'] == False )]['TTP_EXT_ID'].tolist()
        
        return(TTP_list)
        
    def GetTTPForGROUP(self, 
                                    group_name):
      
        TTP_list=self.ATTACK_REL_Index[(self.ATTACK_REL_Index['REL_SOURCE_NAME'] == group_name) &
                                                     (self.ATTACK_REL_Index['REL_SOURCE_TYPE'] == "intrusion-set") &
                                                     ((self.ATTACK_REL_Index['REL_TARGET_TYPE'] == "attack-pattern") |
                                                     (self.ATTACK_REL_Index['REL_TARGET_TYPE'] == "malware") |
                                                     (self.ATTACK_REL_Index['REL_TARGET_TYPE'] == "tool"))]['REL_TARGET_EXT_ID'].tolist()
                                                     
        #
        TTPCurrentList=[]
        #  Found examples of relationships that point to revoked TTP
        #  e.g. G0095 and T1032
        #  So remove revoked TTP from final list
        for TTP in TTP_list:
            #print("Getting match for TTP " + TTP)

            # Check to see if this TTP is revoked
            # Returns a Panda Series
            TTP_revoked=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID']  == TTP)]['Revoked']
            TTP_x_mitre_deprecated=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID']  == TTP)]['TTP_x_mitre_deprecated']
       
            # Should only be one row but fo safety take top row only 
            for Revoked in TTP_revoked:
                FRevoked = Revoked
                break
                
            for Deprecate in TTP_x_mitre_deprecated:
                FDeprecate = Deprecate
                break
                
            #print("Rev/Dep " + str(FRevoked) + "/" + str(FDeprecate))
                
            if ((FRevoked == False) & (FDeprecate == False)):
 
                 TTPCurrentList.append(TTP)
                
  
        # Create a list (pandas Series) of TTP_IDs
        #TTP_ID_List=ATTACK_Index_Segment['TTP_EXT_ID']
        
    
        return(TTPCurrentList)
        
        
    def GetTTPForGROUPByLevel(self, 
                                    group_name, 
                                    Level, 
                                    GetTopFromSub):
                                        
        # Level can be "Top" or "Sub"
      
        TTP_list=self.ATTACK_REL_Index[(self.ATTACK_REL_Index['REL_SOURCE_NAME'] == group_name) &
                                                     (self.ATTACK_REL_Index['REL_SOURCE_TYPE'] == "intrusion-set") &
                                                     ((self.ATTACK_REL_Index['REL_TARGET_TYPE'] == "attack-pattern") |
                                                     (self.ATTACK_REL_Index['REL_TARGET_TYPE'] == "malware") |
                                                     (self.ATTACK_REL_Index['REL_TARGET_TYPE'] == "tool"))]['REL_TARGET_EXT_ID'].tolist()
                                                     
        #
        TTPCurrentList=[]
        #  Found examples of relationships that point to revoked TTP
        #  e.g. G0095 and T1032
        #  So remove revoked TTP from final list
        for TTP in TTP_list:

            # Check to see if this TTP is revoked
            # Returns a Panda Series
            TTP_revoked=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID']  == TTP)]['Revoked']
            TTP_x_mitre_deprecated=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID']  == TTP)]['TTP_x_mitre_deprecated']
            TTP_x_mitre_SubTech=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID']  == TTP)]['TTP_x_mitre_is_subtechnique']
       
            # Should only be one row but fo safety take top row only 
            for Revoked in TTP_revoked:
                FRevoked = Revoked
                break
                
            for Deprecate in TTP_x_mitre_deprecated:
                FDeprecate = Deprecate
                break
                
            for SubTech in TTP_x_mitre_SubTech:
                FSubTech = SubTech
                break
                
            if ((FRevoked == False) & (FDeprecate == False)):
                 if (Level == "Top") & (FSubTech == False):
                     TTPCurrentList.append(TTP)
                 if (Level == "Top") & (FSubTech == True):
                     if GetTopFromSub == "Y":
                         # Extract the top TTP from the name
                         # NOTE: This may be more thorough done as a relationship look up
                         TTPTop=TTP.split(".")[0]
                         if TTPTop not in TTPCurrentList:
                             TTPCurrentList.append(TTPTop)
                 if (Level == "Sub") & (FSubTech == True):
                     TTPCurrentList.append(TTP)
                
  
        # Create a list (pandas Series) of TTP_IDs
        #TTP_ID_List=ATTACK_Index_Segment['TTP_EXT_ID']
        
    
        return(TTPCurrentList)
        
    def GetTTPDomain(self, 
                                ThisTTP):
                                        
        TypeList=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID'] == ThisTTP)]['TTP_Type'].tolist()
        
        for ThisType in TypeList:
            if ThisType == "attack-pattern":
                        DomainList=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID'] == ThisTTP)]['TTP_Domain'].tolist()
                        for ThisDomain in DomainList:
                            Domain=ThisDomain
            if ThisType == "malware":
                Domain = "malware"
            if ThisType == "tool":
                Domain = "tool"
            break
                            
        return(Domain)
        
    def GetTTPType(self, 
                                ThisTTP):
                                        
        TypeList=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID'] == ThisTTP)]['TTP_Type'].tolist()
        
        for ThisType in TypeList:
            if ThisType == "attack-pattern":
                Type = "attack-pattern"
            if ThisType == "malware":
                Type = "malware"
            if ThisType == "tool":
                Type = "tool"
            break
                            
        return(Type)
    
    # Return "Top" or "Sub" to indicate level of technique            
    def GetTTPLevel(self, 
                            ThisTTP):
                                        
        SubList=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID'] == ThisTTP)]['TTP_x_mitre_is_subtechnique'].tolist()
        
        for ThisSub in SubList:
            if ThisSub == True:
                Level="Sub"
            else:
                Level="Top"
            break
                            
        return(Level)
        
    def GetTacticsForGROUP(self, 
                                    group_name):
                                        
        GroupTTPList=self.GetTTPForGROUP(group_name)
        
        GroupTacticList=[]
        
        for ThisTTP in GroupTTPList:
            TacticList=self.ATTACK_TECH_TO_TACTIC_Index[(self.ATTACK_TECH_TO_TACTIC_Index['TTP_EXT_ID'] == ThisTTP)]['TACTIC_EXT_ID'].tolist()
            
            for ThisTactic in TacticList:
                
                if ThisTactic not in GroupTacticList:
                    GroupTacticList.append(ThisTactic)
                                               
        return(GroupTacticList)
        
    def GetTacticsForTTP(self, 
                                    ThisTTP):
                                        
        #print("Getting tactics for " + str(ThisTTP))
                                        
        TacticList=self.ATTACK_TECH_TO_TACTIC_Index[(self.ATTACK_TECH_TO_TACTIC_Index['TTP_EXT_ID'] == ThisTTP)]['TACTIC_EXT_ID'].tolist()
        
        #print("Tactics are " + str(TacticList))
                                               
        return(TacticList)
        
    def GetTacticName(self, 
                                    Tactic_Ext_Id):
                                        
        TacticNameList=self.ATTACK_TACTIC_Index[(self.ATTACK_TACTIC_Index['TACTIC_EXT_ID'] == Tactic_Ext_Id)]['TACTIC_NAME'].tolist()
        
        # No duplicates in table but belt and braces single loop
        for ThisTacticName in TacticNameList:
            TacticName=ThisTacticName
            break
            
        return(TacticName)
        
    def GetTacticDomain(self, 
                                    Tactic_Ext_Id):
                                        
        #
        # One of:
        #     enterprise-attack
        #     mobile-attack
        #     pre-attack
        #
                                        
        TacticDomainList=self.ATTACK_TACTIC_Index[(self.ATTACK_TACTIC_Index['TACTIC_EXT_ID'] == Tactic_Ext_Id)]['TACTIC_MATRIX'].tolist()
        
        # No duplicates in table but belt and braces single loop
        for ThisTacticDomain in TacticDomainList:
            TacticDomain=ThisTacticDomain
            break
            
        return(TacticDomain)
        
    def GetTTPDetails(self, 
                                    TTP_id):
        
        # Get TTP details from the main index
        
        TTP_Details=""
    
        if (self.ATTACK_TTP_Index['TTP_EXT_ID'] == TTP_id).any():
            TTP_Details=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID'] == TTP_id)].to_dict(orient='records')[0]
        else:
            if self.Verbose:
                print("Missing TTP ID: " + TTP_id)
   
        return(TTP_Details)
    
    def GetTOOL_TTPDetails(self, 
                                    TOOL_id):
        
        # Get TOOL TTP details from the sub index
        
        TOOL_TTP_Details=""
    
        if (self.ATTACK_TTP_Index['TTP_EXT_ID'] == TOOL_id).any():
            TOOL_TTP_Details=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID'] == TOOL_id)].to_dict(orient='records')[0]
        else:
            if self.Verbose:
                print("Missing TOOL ID: " + TOOL_id)
   
        return(TOOL_TTP_Details)
        
    def GetGROUPDetails(self, 
                                    GROUP_id):
        
        # Get GROUP details from the main index

        GROUP_Details="No data for " + str(GROUP_id)
            
        if (self.ATTACK_GROUP_Index['Group'] == GROUP_id).any():
             GROUP_Details=self.ATTACK_GROUP_Index[(self.ATTACK_GROUP_Index['Group'] == GROUP_id)].to_dict(orient='records')[0]
        else:
             if self.Verbose:
                 print("Missing GROUP ID: " + GROUP_id)
   
        return(GROUP_Details)
        
    def GetGroupTTPRelDate(self, 
                                    GROUP_NAME, 
                                    TTP):
        
        # Get the date when a TTP was related to a Group
        
        Rel_date_list=self.ATTACK_REL_Index[(self.ATTACK_REL_Index['REL_SOURCE_NAME'] == GROUP_NAME) & 
                                   (self.ATTACK_REL_Index['REL_TARGET_EXT_ID'] == TTP)]['REL_Date_Created'].tolist()

        for rel_date in Rel_date_list:
            
            return(rel_date)
            
    def GetToolTTPRelDate(self, 
                                    Tool_TTP, 
                                    TTP):
        
        # Get the date when a TTP was related to a Tool
        
        Rel_date_list=self.ATTACK_REL_Index[(self.ATTACK_REL_Index['REL_SOURCE_EXT_ID'] == Tool_TTP) & 
                                   (self.ATTACK_REL_Index['REL_TARGET_EXT_ID'] == TTP)]['REL_Date_Created'].tolist()
        

        for rel_date in Rel_date_list:
                        return(rel_date)
                        
    def SelectTOOL_TTP(self, 
                                    TTPList):
        
        # Return only those TTP that may be associated with a tool or malware
        
        TOOL_TTPList=[]
    
        for TTPNext in TTPList:
            if not TTPNext.startswith("T"):
                str("T" + TTPNext)
            if(self.ATTACK_REL_Index[((self.ATTACK_REL_Index['REL_SOURCE_TYPE'] == "tool") |
                                                 (self.ATTACK_REL_Index['REL_SOURCE_TYPE'] == "malware"))  & 
                                                      (self.ATTACK_REL_Index['REL_TARGET_EXT_ID'] == TTPNext)].any()):
                TOOL_TTPList.append(TTPNext)
           # if (self.ATTACK_TOOL_TTP_Index['TTP_EXT_ID'] == TTPNext).any():
                #TOOL_TTPList.append(TTPNext)
   
        return(TOOL_TTPList)
        
    def GetTTPForTOOL(self, 
                                    tool_name):
                                        
        TTP_List=self.ATTACK_REL_Index[(self.ATTACK_REL_Index['REL_SOURCE_EXT_ID'] == tool_name) & 
                                                      (self.ATTACK_REL_Index['REL_TARGET_TYPE'] == "attack-pattern")]['REL_TARGET_EXT_ID'].tolist()
      
        #TTP_List=self.ATTACK_TOOL_TTP_Index[(self.ATTACK_TOOL_TTP_Index['TOOL_EXT_ID'] == tool_name)]['TTP_EXT_ID'].tolist()
  
        # Create a list (pandas Series) of TTP_IDs
        #TTP_ID_List=ATTACK_Index_Segment['TTP_EXT_ID']
        
#Added later marked while testing#########
        TTPCurrentList=[]
        #  Found examples of relationships that point to revoked TTP
        #  e.g. G0095 and T1032
        #  So remove revoked TTP from final list
        for TTP in TTP_List:

            # Check to see if this TTP is revoked
            # Returns a Panda Series
            TTP_revoked=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID']  == TTP)]['Revoked']
            TTP_x_mitre_deprecated=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID']  == TTP)]['TTP_x_mitre_deprecated']
            #TTP_x_mitre_SubTech=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['TTP_EXT_ID']  == TTP)]['TTP_x_mitre_is_subtechnique']
       
            # Should only be one row but fo safety take top row only 
            for Revoked in TTP_revoked:
                FRevoked = Revoked
                break
                
            for Deprecate in TTP_x_mitre_deprecated:
                FDeprecate = Deprecate
                break
                
            #for SubTech in TTP_x_mitre_SubTech:
                #FSubTech = SubTech
                #break
                
            if ((FRevoked == False) & (FDeprecate == False)):
                 TTPCurrentList.append(TTP)
  
        # Create a list (pandas Series) of TTP_IDs
        #TTP_ID_List=ATTACK_Index_Segment['TTP_EXT_ID']
        
    
        return(TTPCurrentList)

##########
    
#        return(TTP_List)   
        
    def GetGROUPForTTP(self,  TTP_id):
        
        # Look for Groups that use this TTP.
        Group_List=self.ATTACK_Index[(self.ATTACK_Index['TTP_EXT_ID'] == str(TTP_id))]['Group'].tolist()
            
        # Create a lit of unique items in Group list
        Group_Set=set(Group_List)
        Group_List_Unique=list(Group_Set)
        # Sort the list
        Group_List_Unique.sort()
            
        return(Group_List_Unique)
        
    def GetATTACK_TTP_BIN_TOTALS(self):
      
        res=self.ATTACK_TTP_BIN_Index.sum(axis = 0, skipna = True) 
            
        return(res)
        
    def GetATTACK_TTP_BIN_TOTALS_MATRIX(self, 
                                                                           Matrix):
        
        # Return counts of Technique use by groups in domain
        # One of:
        #      Enterprise
        #      Mobile
        #       Pre-ATT&CK
        
        # Get a list of techniques for this matrix
        TechList=self.GetListOfTTPInDomain(Matrix)        
        
        # Get these columns from the TTP Bin index        
        RefCount=self.ATTACK_TTP_BIN_Index[TechList].sum(axis = 0, skipna = True)
        
        return(RefCount)
        
    def GetATTACK_TTP_BIN_TOTALS_TACTIC(self, 
                                                                        Tactic):
        
        # Return counts of Technique use by groups in domain
        # One of:
        #      Enterprise
        #      Mobile
        #       Pre-ATT&CK
        
        # Get a list of techniques for this matrix
        TechList=self.GetTechForTactic(Tactic)        
        
        # Get these columns from the TTP Bin index        
        RefCount=self.ATTACK_TTP_BIN_Index[TechList].sum(axis = 0, skipna = True)
        
        return(RefCount)
                        
#   def GetGROUPForTTP(self,  TTP_id):
        
#       Look for Groups that use this TTP.
#        Group_List=self.ATTACK_Index[(self.ATTACK_Index['TTP_EXT_ID'] == str(TTP_id))]['Group'].tolist()
            
#        Create a lit of unique items in Group list
#        Group_Set=set(Group_List)
#        Group_List_Unique=list(Group_Set)
        # Sort the list
#       Group_List_Unique.sort()
            
#        return(Group_List_Unique)
#
#  TO BE REPLACED AS SOON AS POSSIBLE
    def Manually_Add_Group_Details(self, 
                                                           group_name, 
                                                           TTP_List):
#  List of Groups.
#    ATTACK_GROUP_Index = pd.DataFrame(columns=['Group',
#                                                            'GID', 
#                                                            'Aliases',
#                                                           'Description', 
#                                                            'Attribution', 
#                                                            'Group_Create_Date', 
#                                                            'ISet_ID', 
#                                                            'Revoked'
#                                                            ])

#         Add a group
        TheseRowsList=[]
        GROUP_ROW_Dict={'Group': group_name, 
                                      'GID': "G_" + group_name, 
                                      'Aliases': group_name + "_ALIAS", 
                                      'Description': group_name + " GROUP", 
                                      'Attribution': "Fantasia", 
                                      'Group_Create_Date': "2021-01-14 19:41:27.845000+00:00", 
                                      'ISet_ID': group_name + "_SET_ID", 
                                      'Revoked': False}
        
        TheseRowsList.append(GROUP_ROW_Dict)
                                      
        self.ATTACK_GROUP_Index = self.ATTACK_GROUP_Index.append(pd.DataFrame(TheseRowsList), sort=True)

#    ATTACK_REL_Index = pd.DataFrame(columns=[
#                                                            'REL_SOURCE_ID', 
#                                                            'REL_SOURCE_TYPE', 
#                                                            'REL_SOURCE_NAME', 
#                                                           'REL_SOURCE_EXT_ID', 
#                                                           'REL_TARGET_ID', 
#                                                            'REL_TARGET_TYPE', 
#                                                            'REL_TARGET_NAME', 
#                                                            'REL_TARGET_EXT_ID', 
#                                                            'REL_TYPE', 
#                                                            'REL_ID'
#                                                            'REL_Date_Created', 
#                                                            'Revoked'
#                                                            ])

#         Add relatonship rows for TTP.
#        This can ber used to drive this later

                                     
        RawRowDataList=[]
                         
        for nextTech in TTP_List:
            NextRow=["XX","intrusion-set", 
                                     group_name, "G_" + group_name,
                                     "XX", "attack-pattern", 
                                     "DUMMY_NAME", nextTech,  
                                     "uses", "XX",
                                     "2021-01-14 19:41:27.845000+00:00"] 
            RawRowDataList.append(NextRow)
                                     
        TheseRowsList=[]
        
        for nextDictData in RawRowDataList:
            
            #print("Adding for " + nextDictData[7])

            REL_ROW_Dict={
                                        'REL_SOURCE_ID':nextDictData[0], 
                                        'REL_SOURCE_TYPE':nextDictData[1], 
                                        'REL_SOURCE_NAME':nextDictData[2], 
                                        'REL_SOURCE_EXT_ID':nextDictData[3], 
                                        'REL_TARGET_ID':nextDictData[4], 
                                        'REL_TARGET_TYPE':nextDictData[5], 
                                        'REL_TARGET_NAME':nextDictData[6], 
                                        'REL_TARGET_EXT_ID':nextDictData[7], 
                                        'REL_TYPE':nextDictData[8], 
                                        'REL_ID':nextDictData[9], 
                                        'REL_Date_Created':nextDictData[10], 
                                        'Revoked':False
                                        }
                                        
            TheseRowsList.append(REL_ROW_Dict)
            
        self.ATTACK_REL_Index  = self.ATTACK_REL_Index .append(pd.DataFrame(TheseRowsList), sort=True)

#         Add binary TTP line for this group

        TTPTotalList=self.ATTACK_TTP_Index[(self.ATTACK_TTP_Index['Revoked'] == False) &
                                                      (self.ATTACK_TTP_Index['TTP_x_mitre_deprecated'] == False )]['TTP_EXT_ID'].tolist()
        TTPTotalList=self.ATTACK_TTP_BIN_Index.columns.tolist()                                              
        ThisTTPBin=[0]*len(TTPTotalList)
        
        #fname="C:/Users/chris/OneDrive - University of Southampton/MAFpt_PROLOG/DELETETHIS_PRE.csv"
        #self.ATTACK_TTP_BIN_Index.to_csv(path_or_buf=fname, index=True)   
        
        RowList=[]
        for nextTTP in TTP_List:
            TTPPos=TTPTotalList.index(nextTTP)
            ThisTTPBin[TTPPos]=1
        RowList.append(ThisTTPBin)
            
        DFRow=pd.DataFrame(RowList)
        DFRow.columns = TTPTotalList
        DFRow.index=[group_name]
            
        self.ATTACK_TTP_BIN_Index  = self.ATTACK_TTP_BIN_Index .append(DFRow, sort=False)
    
        #fname="C:/Users/chris/OneDrive - University of Southampton/MAFpt_PROLOG/DELETETHIS.csv"
        #self.ATTACK_TTP_BIN_Index.to_csv(path_or_buf=fname, index=True)   
      
            
        return(0)
        
    # This would be better as a Prolog approach
    #        
    def MatchEvidenceSet(self, 
                                         EvidenceSet):
                                             
        MatchedGroups=[]
                                             
        for ThisTTP in EvidenceSet:
            # Get a list of groups who use this TTP.
            ThisTTPCol = self.ATTACK_TTP_BIN_Index.loc[ :  ,  ThisTTP]
            for index, nextval in ThisTTPCol.items():
                #for index, value in s.items():
                
                    
                if nextval == 1:
                    # Is tbis group in the list
                    AlreadyThere=0
                    
                    for NextDict in MatchedGroups:
                        if NextDict.get('GroupName') == index:
                            TTPCount=NextDict.get('TTPCount')
                            NextDict.update({'TTPCount':TTPCount+1})                            
                            NextDict.get('TTPList').append(ThisTTP)
                            AlreadyThere=1
                            
                    if AlreadyThere==0:
                        ThisDict={'GroupName': index, 
                                        'TTPCount':1, 
                                       'TTPList':[]}
                        ThisDict.get('TTPList').append(ThisTTP)
                    
                        MatchedGroups.append(ThisDict)
                        
        newlist = sorted(MatchedGroups, key=lambda k: k['TTPCount'],   reverse=True) 
                        
        return(newlist)
        
    # This would be better as a Prolog approach
    #        
    def MatchEvidenceSet_v2(self, 
                                         EvidenceSet):
                                             
        MatchedGroups=[]
        
        # For each TTP in Evidence Set
                                             
        for ThisTTP in EvidenceSet:
            # Get a list of groups who use this TTP.
            # Do this by using the binary index
            #    each row is a group
            #     each column is a TTP
            #   Get the column for this TTP
            ThisTTPCol = self.ATTACK_TTP_BIN_Index.loc[ :  ,  ThisTTP]
            # Iterate down the column checking for 1s
            for index, nextval in ThisTTPCol.items():
                #for index, value in s.items():
                
                    
                if nextval == 1:
                    # Is tbis group in the list
                    AlreadyThere=0
                    
                    # Go through the dicts already creeated to see if we already have a dict for this group
                    for NextDict in MatchedGroups:
                        if NextDict.get('GroupName') == index:
                            TTPCount=NextDict.get('TTPCount')
                            NextDict.update({'TTPCount':TTPCount+1})                            
                            NextDict.get('TTPList').append(ThisTTP)
                            AlreadyThere=1
                            
                    if AlreadyThere==0:
                        ThisDict={'GroupName': index, 
                                        'TTPCount':1, 
                                       'TTPList':[]}
                        ThisDict.get('TTPList').append(ThisTTP)
                    
                        MatchedGroups.append(ThisDict)
         
        # Sort the list in descending number of matches order         
        newlist = sorted(MatchedGroups, key=lambda k: k['TTPCount'],   reverse=True) 
                        
        return(newlist)
        
     
    def GetSchema(self):
        print("Displaying Schema")
        
        print("<<< Describe ATTACK_META_MODEL")        
        #print(self.ATTACK_META_MODEL.describe())
        for col in self.ATTACK_META_MODEL_Index.columns: 
            print(col) 
        print("<<< Describe ATTACK_GROUP_Index")  
        #print(self.ATTACK_GROUP_Index.describe())
        for col in self.ATTACK_GROUP_Index.columns: 
            print(col) 
        print("<<< Describe ATTACK_TTP_Index")  
        #print(self.ATTACK_TTP_Index.describe())
        for col in self.ATTACK_TTP_Index.columns: 
            print(col) 
        print("<<< Describe ATTACK_REL_Index")  
        #print(self.ATTACK_REL_Index.describe())
        for col in self.ATTACK_REL_Index.columns: 
            print(col) 
        print("<<< Describe ATTACK_CVE_REF_Index")  
        #print(self.ATTACK_CVE_REF_Index.describe())
        for col in self.ATTACK_CVE_REF_Index.columns: 
            print(col) 
        print("<<< Describe ATTACK_TTP_TO_CAPEC_Index")  
        #print(self.ATTACK_TTP_TO_CAPEC_Index.describe())
        for col in self.ATTACK_TTP_REF_Index.columns: 
            print(col) 
        print("<<< Describe ATTACK_TACTIC_Index")  
        #print(self.ATTACK_TACTIC_Index.describe())
        for col in self.ATTACK_TACTIC_Index.columns: 
            print(col) 
        print("<<< Describe ATTACK_TECH_TO_TACTIC_Index")  
        #print(self.ATTACK_TECH_TO_TACTIC_Index.describe())
        for col in self.ATTACK_TECH_TO_TACTIC_Index.columns: 
            print(col) 
            
    def GetReportsForTech(self, 
                                          TTP, 
                                          Include_MIT_Refs):
        #print("Getting reports for " + TTP)
#        ATTACK_TTP_REF_Index = pd.DataFrame(columns=['TTP_EXT_ID',
#                                                            'source_name', 
#                                                            'url',
#                                                            'external_id', 
#                                                            'description'
#                                                            ])    
        Report_List=self.ATTACK_TTP_REF_Index[(self.ATTACK_TTP_REF_Index['TTP_EXT_ID'] == str(TTP))]['url'].tolist()
        
        if Include_MIT_Refs == "Y":

        #         Find the Mitigations for this technique
        #
            MITList=self.GetMitigationsForTech(TTP)
        
            for ThisMIT in MITList:
                #print("Checking for " + str(ThisMIT))

                #
                # Look for the relationship that links the TTP and Mitigation
                RelList=self.ATTACK_REL_Index[(self.ATTACK_REL_Index['REL_SOURCE_EXT_ID'] == str(ThisMIT)) &
                                                              (self.ATTACK_REL_Index['REL_TARGET_EXT_ID'] == str(TTP) )]['REL_ID'].tolist()
                                                              
                for ThisRel in RelList:

                    RefList=self.ATTACK_REL_REF_Index[(self.ATTACK_REL_REF_Index['REL_ID'] == str(ThisRel))]['url'].tolist()
                    for ThisRef in RefList:
                        #print("Adding report " + ThisRef)
                        Report_List.append(ThisRef)
                
                    # There should only be one realtionship line
                    break
            
        return(Report_List)
        
    def GetMitigationsForTech(self, 
                                          TTP):
        #print("Getting mitigations for " + TTP)
#  Mitigation Index
#                                                                                                                          
#    ATTACK_MITIGATION_Index = pd.DataFrame(columns=[
#                                                            'TTP_EXT_ID', 
#                                                           'COA_EXT_ID', 
#                                                            'COA_ID', 
#                                                            'COA_NAME', 
#                                                            'COA_REL_DESC', 
#                                                            'COA_DESC', 
#                                                            'COA_Create_Desc'
#                                                            ])
        MIT_List=self.ATTACK_MITIGATION_Index[(self.ATTACK_MITIGATION_Index['TTP_EXT_ID'] == str(TTP))]['COA_EXT_ID'].tolist()
        return(MIT_List)
        
    def GetTechForRef(self, 
                                          REF):
        #print("Getting TTP for " + REF)
#  #                                                                                                                          
#        ATTACK_TTP_REF_Index = pd.DataFrame(columns=['TTP_EXT_ID',
#                                                            'source_name', 
#                                                            'url',
#                                                            'external_id', 
#                                                            'description'
#                                                            ])
        TTP_List=self.ATTACK_TTP_REF_Index[(self.ATTACK_TTP_REF_Index['url'] == str(REF))]['TTP_EXT_ID'].tolist()
        return(TTP_List)
        
    def GetTTPRefCounts(self):
        #print("Getting TTP for " + REF)
#  #                                                                                                                          
#        ATTACK_TTP_REF_Index = pd.DataFrame(columns=['TTP_EXT_ID',
#                                                            'source_name', 
#                                                            'url',
#                                                            'external_id', 
#                                                            'description'
#                                                            ])
        RefCount=self.ATTACK_TTP_REF_BIN_Index.sum(axis = 0, skipna = True)
        #print(RefCount)
        return(RefCount)
        
    def GetListOfTTPReferences(self):
        
        # Return a list of all TTP
        
        TTP_list=self.ATTACK_TTP_REF_Index['url'].unique().tolist()
        
        return(TTP_list)
        
    def GetTacticForRef(self, 
                                          REF):
        #print("Getting mitigations for " + TTP)

        # Get the TTP that reference REF
        TTP_List=self.GetTechForRef(REF)
        
        #Build a unique list of tactics associated with the TTPs
        
        FullTacticList=[]
        
        for NextTTP in TTP_List:
            Tactic_List=self.GetTacticsForTTP(NextTTP)
            
            for NextTactic in Tactic_List:
                if not NextTactic in FullTacticList:
                    FullTacticList.append(NextTactic)
                    
        return(FullTacticList)
        
    def GetProcTextForTTP(self, 
                                          TTP_EXT_ID):
        #print("Getting mitigations for " + TTP)
 
        FullTextList=self.ATTACK_TTP_TO_PROCEDURE_Index[(self.ATTACK_TTP_TO_PROCEDURE_Index['TTP_EXT_ID'] == TTP_EXT_ID) &
                                                                                     (self.ATTACK_TTP_TO_PROCEDURE_Index['USER_TYPE'] != "course-of-action")]['Procedure_Desc'].tolist()
                    
        return(FullTextList)
        
    def GetCountsTacticUseByGroup(self, 
                                          Matrix):
        # [enterprise-attack, mobile-attack, pre-attack] DomainName)
        #print("Getting mitigations for " + TTP)
 
        # Get a list of tactics for this matrix
        TacticsList=self.GetListOfTacticsInDomain(Matrix)
        
        # Get these columns from the Tactic Bin index
        RefCount=self.ATTACK_TACTIC_BIN_Index[TacticsList].sum(axis = 0, skipna = True)
        
        return(RefCount)
        
    def GetCountsTechUseByTactic(self, 
                                          Matrix):
        # [enterprise-attack, mobile-attack, pre-attack] DomainName)
        #print("Getting mitigations for " + TTP)
 
        # Get a list of tactics for this matrix
        TacticsList=self.GetListOfTacticsInDomain(Matrix)
        
        SumList=[] 
        
        for ThisTactic in TacticsList:
            #Get a list of technique for this tactic
            SumList.append(len(self.GetTechForTactic(ThisTactic)))
            
        # Get these columns from the Tactic Bin index
        RefCount=pd.Series(data=SumList,  index=TacticsList)
        
        return(RefCount)
        
    def GetCountsMainTechUseByTactic(self, 
                                          Matrix):
        # [enterprise-attack, mobile-attack, pre-attack] DomainName)
        #print("Getting mitigations for " + TTP)
 
        # Get a list of tactics for this matrix
        TacticsList=self.GetListOfTacticsInDomain(Matrix)
        
        SumList=[] 
        
        for ThisTactic in TacticsList:
            #Get a list of technique for this tactic
            SumList.append(len(self.GetMainTechForTactic(ThisTactic)))
            
        # Get these columns from the Tactic Bin index
        RefCount=pd.Series(data=SumList,  index=TacticsList)
        
        return(RefCount)
        
    def GetTechForTactic(self, 
                                          Tactic):
        #print("Getting TTP for " + REF)
#  Tech To Tactic Index
#                                                                                                                          
#    ATTACK_TECH_TO_TACTIC_Index = pd.DataFrame(columns=[
#                                                            'TTP_NAME', 
#                                                            'TTP_EXT_ID', 
#                                                            'KILL_CHAIN_NAME', 
#                                                            'TACTIC_NAME', 
#                                                            'TACTIC_EXT_ID', 
#                                                            'REL_Date_Created'
#                                                            ])
        TTP_List=self.ATTACK_TECH_TO_TACTIC_Index[(self.ATTACK_TECH_TO_TACTIC_Index['TACTIC_EXT_ID'] == str(Tactic))]['TTP_EXT_ID'].tolist()
        return(TTP_List)
        
    def GetMainTechForTactic(self, 
                                          Tactic):
        #print("Getting TTP for " + REF)
#  Tech To Tactic Index
#                                                                                                                          
#    ATTACK_TECH_TO_TACTIC_Index = pd.DataFrame(columns=[
#                                                            'TTP_NAME', 
#                                                            'TTP_EXT_ID', 
#                                                            'KILL_CHAIN_NAME', 
#                                                            'TACTIC_NAME', 
#                                                            'TACTIC_EXT_ID', 
#                                                            'REL_Date_Created'
#                                                            ])
        TTP_List=self.ATTACK_TECH_TO_TACTIC_Index[(self.ATTACK_TECH_TO_TACTIC_Index['TACTIC_EXT_ID'] == str(Tactic))]['TTP_EXT_ID'].tolist()
        
        TTPMainList=[]
        
        for NextTTP in TTP_List:
            if self.GetTTPLevel(NextTTP) == "Top":
                TTPMainList.append(NextTTP)
                
        return(TTPMainList)
        
    def GetRelUsedByGroup(self, 
                                          GID):
        #print("Getting TTP for " + REF)
#  GetRelUsedByGroup
#                                                                                                                             
        RELDictList=self.ATTACK_REL_Index[(self.ATTACK_REL_Index['REL_SOURCE_EXT_ID'] == GID) ].to_dict('records')
            
        return(RELDictList)
        
    def GetRelLineDetails(self, 
                                          REL_SOURCE_EXT_ID, 
                                          REL_TARGET_EXT_ID, 
                                          REL_TYPE):
        #                                                                                                                
        RELDictList=self.ATTACK_REL_Index[(self.ATTACK_REL_Index['REL_SOURCE_EXT_ID'] == REL_SOURCE_EXT_ID) &
                                                              (self.ATTACK_REL_Index['REL_TARGET_EXT_ID'] == REL_TARGET_EXT_ID) &
                                                              (self.ATTACK_REL_Index['REL_TYPE'] == REL_TYPE)].to_dict('records')
                                                              
        # Should only be one returned but ...
        for NextRELDict in RELDictList:
            ThisRELDict=NextRELDict
            break
            
        return(ThisRELDict)
            
    def GetTacticDetails(self, 
                                          TACTIC_EXT_ID):
        #                                                                                                               
        TacticDictList=self.ATTACK_TACTIC_Index[(self.ATTACK_TACTIC_Index['TACTIC_EXT_ID'] == TACTIC_EXT_ID) ].to_dict('records')
                                                              
        # Should only be one returned but ...
        for NextTacticDict in TacticDictList:
            ThisTacticDict=NextTacticDict
            break
            
        return(ThisTacticDict)
        
    def HasCARAnalytic(self, 
                                          TTP_EXT_ID):
        #                                                                                                               
        CARDictList=self.ATTACK_CAR_COVERAGE_Index[(self.ATTACK_CAR_COVERAGE_Index['Technique'] == TTP_EXT_ID) ].to_dict('records')
                                                              
        # Should only be one returned but ...
        HasAnalytic=False
        
        for NextCARDict in CARDictList:
            ThisCARDict=NextCARDict
            if ThisCARDict['Tot_Num'] > 0:
                HasAnalytic=True
            break
            
        return(HasAnalytic)
        
    def GetGroupForTTP(self, 
                                          TTP_EXT_ID):
        #                                                                                                               
        GroupList=self.ATTACK_REL_Index[(self.ATTACK_REL_Index['REL_TARGET_TYPE'] == "attack-pattern") &
                                                              (self.ATTACK_REL_Index['REL_TARGET_EXT_ID'] == TTP_EXT_ID) &
                                                              (self.ATTACK_REL_Index['REL_SOURCE_TYPE'] == "intrusion-set")
                                                              ]['REL_SOURCE_EXT_ID'].to_list()
                                                              
        FinalGroupList=[]
        
        for Group in GroupList:
            
            if Group == "G0118":
                print("Have found the G0118 relationship")

            # Check to see if group can be found in the 
            # e.g. for G0118 have found relationships to revoked groups
            #     intrusion-set--dc5e2999-ca1a-47d4-8d12-a6984b138a1b
            # 
            CheckList=self.ATTACK_GROUP_Index[(self.ATTACK_GROUP_Index['GID']  == Group)]['Group']
            # if not found this should not return anything
            Found=False
            for item in CheckList:
                FinalGroupList.append(Group)
                Found=True
            if Found==False:
                print("<<Group " + Group + " not found")
                
            
        return(FinalGroupList)

#   Get Group Techniques organised by Tactic
#   List of dicts{TacticId, TechList}
#  Currently limited to Enterprise
    def GetGroupTechChain(self, 
                                          GID):
        #print("Getting TTP for " + REF)
        #  GetRelUsedByGroup
        #
        
        # Get Tech for this Group      
        group_name=self.GetGroupName(GID)
        
        TechList= self.GetTTPForGROUP(group_name) 
        
        # Get a list of all Tactics
        TacticList=self.GetListOfTacticsInDomain("enterprise-attack")
        
        GroupTechChain=[]
        
        # Build an empty chain        
        for ThisTactic in TacticList:
            TacticDict={"Tactic":ThisTactic, 
                              "TechList":[]}
            GroupTechChain.append(TacticDict)
                    
    
        for ThisTech in TechList:
            # Get the Tactic list for this Technique (techs may be in multiple tactics)
                       
            TechTacticList=self.GetTacticsForTTP(ThisTech)
            
            # Add this tech to relevant tactic tech lists
            for ThisTechTactic in TechTacticList:
                for ThisChainDict in GroupTechChain:
                    if ThisChainDict["Tactic"] == ThisTechTactic:
                        ThisChainDict["TechList"].append(ThisTech)
                        break # You can stop here
            
        return(GroupTechChain)
        
    def GetTechForSub(self, 
                                          TTP_EXT_ID):
        
                
        RELDictList=self.ATTACK_REL_Index[(self.ATTACK_REL_Index['REL_SOURCE_EXT_ID'] == TTP_EXT_ID) &
                                                                (self.ATTACK_REL_Index['REL_TYPE'] == "subtechnique-of")].to_dict('records')
                                                              
        # Should only be one returned but ...
        for NextRELDict in RELDictList:
            ParentTech=NextRELDict["REL_TARGET_EXT_ID"]
            break
            
        return(ParentTech)
        
    def GetRefsForGroupTech(self, 
                                          GID, 
                                          TTP_EXT_ID):
                                              
        # Get the relationship between the Tech and the Group
        RELDictList=self.ATTACK_REL_Index[(self.ATTACK_REL_Index['REL_SOURCE_EXT_ID'] == GID) &
                                                                (self.ATTACK_REL_Index['REL_TYPE'] == "uses") &
                                                                (self.ATTACK_REL_Index['REL_TARGET_EXT_ID'] == TTP_EXT_ID)].to_dict('records')
                                                              
        # Should only be one returned but ...
        RelID=""
        for NextRELDict in RELDictList:
            #print("Another Dict")
            RelID=NextRELDict["REL_ID"]
            break
# #  References associated with relationships
#                                                            
#    ATTACK_REL_REF_Index = pd.DataFrame(columns=['REL_ID',
#                                                            'source_name', 
#                                                            'url',
#                                                            'external_id', 
#                                                            'description'
 #                                                           ]) 
        #print("The REL ID is " + RelID)
        RELRefList=[]
        if not RelID == "" :
           RELRefList=self.ATTACK_REL_REF_Index[(self.ATTACK_REL_REF_Index["REL_ID"] ==RelID) ].to_dict('records')
        
        RefList=[]
        for NextRELRef in RELRefList:
            RefList.append(NextRELRef["url"])
        
        return(RefList)
        
    def GetTacticBinIndex(self):
        # Just return Tactic Bin Index as a DF
                
        return(self.ATTACK_TACTIC_BIN_Index)
        
  # End ATTACK_DB class and methods
#########################
