%Script for plotting Figure 4 (BCF as function of top-of-ocean depth).
%Each run of this script generates one subplot at specified Jovian System-III east latitude.

%Convert longitude list to 0-360 convention
lon_list(lon_list<0) = lon_list(lon_list<0) + 360;
lon = lon_list(1); %Pick any element in lon_list, because figure only contain subflow meridian.

%Pick System-III longitude to plot.
lambda3 = 248;

%Set up plot parameters for choice of System-III Longitude
if lambda3 == 158
    Gpos = 1; %Cell position in Plotting Cell
    lambda3_string = '158^{o}/338^{o}'; %System-III longitude string name (158/338 are together due to identical data)
    subplot_letter = '(b)'; %Subplot letter labelling
    BCFmin = 0; %Minimum BCF y-axis limit
    BCFmax = 80; %Maximum BCF y-axis limit
    textx = 0.5; %starting x-coordinate of subplot's text label
    texty = 75; %starting y-coordinate of subplot's text label
    legend_loc = 'southeast'; %location of plot legend
elseif lambda3 == 248
    Gpos = 2;
    lambda3_string = '248^{o}';
    subplot_letter = '(a)';
    BCFmin = 0;
    BCFmax = 80;
    textx = 0.35;
    texty = 75;
    legend_loc = 'east';
elseif lambda3 == 338
    Gpos = 3;
    lambda3_string = '158^{o}/338^{o}';
    subplot_letter = '(b)';
    BCFmin = 0;
    BCFmax = 80;
    textx = 0.5;
    texty = 75;
    legend_loc = 'southeast';
else
    return
end

%Obtain plotting arrays by reformatting 3D BCF field strength array into a
%cell each containing 1D array of BCF over ocean depth range.
for k = 1:length(lat_list)
    for w = 1:length(Dfcell{Gpos})
        BCF_plotarray = BCFcell{Gpos};
        BCF_plot{k}(w) = BCF_plotarray(k,lon_list==lon,w);
        lat_num{k} = lat_list(k);
    end
end

%Set vertical lines marking outer ice shell limit and top of ocean depths.
ln1_range = linspace(BCFmin,BCFmax,(BCFmax-BCFmin)+1);
sh_ln1 = 0.70*ones(size(ln1_range));
ln2_range = linspace(BCFmin,BCFmax,(BCFmax-BCFmin)+1);
sh_ln2 = 0.95*ones(size(ln1_range));

%Generate plot for each System-III longitude.
%Set up figure.
fig = figure();
xlabel('R_{sp} (R_G)','Interpreter','tex');
xlim([0 1.00]) %Define x-axis.
fontsize(gca,24,'Points')
set(gca,'XDir','reverse');
ylabel('B_{CF} (nT)','Interpreter','tex'); %Define y-axis.
ylim([BCFmin BCFmax])
yticks([0 20 40 60 80])
yticklabels({'0','20','40','60','80'})
fontsize(gca,24,'Points')
set(gca,'XMinorTick','on','YMinorTick','on') %Set tick formats
set(gca,'TickLength',[0.02,0.05])

%Loop over cells to plot evolution lines at different Ganymedean latitudes.
hold on
for m = 1:numel(BCF_plot)
    name = strcat(sprintf('\\Lambda = \\pm%d^\\circ',lat_num{m}));
    p(m) = plot(Dfcell{Gpos},BCF_plot{m},'Linewidth',2,'DisplayName',name);
end
%Plot vertical lines for outer ice shell limit and top of ocean depths.
border_line = plot(sh_ln1,ln1_range,'k--','Linewidth',1.5);
toc_line = plot(sh_ln2,ln2_range,'k--','Linewidth',1.5);
hold off

%Format legend and plot labels.
lgd = legend(p([1 2 3 4]),'Location',legend_loc);
fontsize(lgd,18,'points')
text(textx,texty,strcat(subplot_letter,' \lambda_{III} = ',string(lambda3_string)),'FontSize',24,'FontWeight','bold')
text(0.68,texty-10,'Outer Ice Shell Ends','FontSize',18,'FontWeight','normal')
text(0.93,texty,'Ocean Estimate','FontSize',18,'FontWeight','normal')

%Save figure.
Dfres = round(360./(length(Dfcell{Gpos})-1),1); %Set ocean depth as string name. 
filename = strcat('BCFdepth_lIII',string(lambda3),'_lr',string(Dfres),'_y',string(ymax),'z',string(zmax),'_MPr',string(res),'.png'); %Set file name.
removeToolbarExplorationButtons(fig) %Remove Matlab toolbars
exportgraphics(fig,filename,'Resolution',500) %Export figure.