%Script for plotting Figure 5 (Chapman-Ferraro magnetic field in cross-magnetopause
%planes at specific GphiO Z-coordinate and Jovian System-III east longitude).

%Set up figure with 9 subplots (3x3).
fig = figure('Name','BCF_Plane');
set(fig,'Position',[50,50,900,750]);
ha = tight_subplot(3,3,[.15 .00],[.08 .06],[.03 .08]);

%Set up list of Galileo positions for subplots d-i.
%Subplot a-c used for non-Galileo plots.
xSC = [0.73,0.51,1.14,-1.42,-1.24,0.89]; %GphiO X-coordinates
ySC = [-1.5,1.5,1.5,0,0,-1.5]; %GphiO Y-coordinates
Gnum = [1,2,7,8,28,29]; %Orbit numbers.


%Use loop to plot from left to right, top to bottom
for i = 1:9
    %Identify subplot row and column for this loop.
    row_ind = ceil(i/3);
    col_ind = rem(i,3);
    if col_ind == 0 %no remainder => 3rd column
        col_ind = 3;
    end
    
    %Set letter labels
    alc = 'abcdefghijklmnopqrstuvwxyz';
    label = sprintf('(%s)',alc(i));

    %Obtain GphiO-Z and System-III longitude for the subplot.
    Z0 = Z0cell{row_ind,col_ind};
    Z0str = string(Z0);
    lambda3 = lambda3cell{row_ind,col_ind};
    lambda3str = string(lambda3);

    %Make circle outline for Ganymede at corresponding GphiO-Z coordinate.
    %Circle becomes dot if radius is zero (|Z| > 1 RG).
    ang = linspace(0,2*pi,361);
    rad = sqrt(1-Z0.^2);
    if isreal(rad)
        xcir = rad*cos(ang);
        ycir = rad*sin(ang);
    else
        xdot = 0.0;
        ydot = 0.0;
    end

    %Generate subplot.
    axes(ha(i)) %Pick correct subplot axis.
    subfig = pcolor(Xmcell{row_ind,col_ind},Ymcell{row_ind,col_ind},BCFcell{row_ind,col_ind}); %Plot data.
    set(subfig,'EdgeColor','none'); %Remove grid edges.
    clim([0 100]) %Set color bar limits.
    
    %Overlay Ganymede circle/dot.
    hold on
    if isreal(rad)
        plot(xcir,ycir, 'Color','w','Linewidth',2); %grey color line
    else
        scatter(xdot,ydot,30,'w','filled');
    end

    %Overlay Galileo positions in subplots d-i.
    if i >= 4
        scatter(xSC(i-3),ySC(i-3),40,'r','filled')
        text(xSC(i-3)+0.12,ySC(i-3),strcat('G',string(Gnum(i-3))),'Color','r','FontWeight','bold','FontSize',18)
    end
    hold off
    axis equal

    %Set subplot axes, labels and ticks.
    xlim([-2 2]); %x-axis limits.
    ylim([-2 2]); %y-axis limits.
    set(gca,'XMinorTick','on','YMinorTick','on') %Use minor ticks.
    fontsize(gca,18,'Points') %Set tick fontsize.
    xlabel('X (R_G)','Interpreter','tex','FontSize',20); %Set x-axis label.
    ylabel('Y (R_G)','Interpreter','tex','FontSize',20); %Set y-axis label.
    set(gca,'TickLength',[0.02,0.05]) %Format ticks.
    title(sprintf('\\lambda_{III} = %s^{o} (Z=%s)',lambda3str,Z0str),'Interpreter','tex','FontSize',20) %Set subplot title.
    text(-1.9,1.75,label,'Color','white','FontWeight','bold','FontSize',22) %Add subplot letter label.

end

%Produce color bar.
c = colorbar;
c.Position = [0.90 0.08, 0.01 0.86]; %Set color bar position.
c.Label.String = 'B_{CF} (nT)'; %Set color bar label.
c.FontSize = 20; %Set color bar label font size.

%Save figure.
filename = strcat('BCFplane','_y',string(ymax),'z',string(zmax),'_Galileo','.png'); %Set file name.
removeToolbarExplorationButtons(fig) %Remove Matlab toolbars
exportgraphics(fig,filename,'Resolution',500) %Export figure.
   
