%Function for extending magnetopause surface from the half-cylindrical section 
%toward downstream of Ganymede.

%Take as input half-cylindrical magnetopause produced in jCF_from_MP.m
%Recommend same flank resolution as in half-cylindrical but not enforced.
function [Xex_aj,Yex_aj,Zex_aj,Xex_sj,Yex_sj,Zex_sj,grad_aj,grad_sj] = MPex(Xgrid,Ygrid,Zgrid,Nex,res_ex)

%Define final extension meshgrids
Xex_aj = NaN(length(Xgrid(:,1)),Nex);
Yex_aj = NaN(length(Xgrid(:,1)),Nex);
Zex_aj = NaN(length(Xgrid(:,1)),Nex);
Xex_sj = NaN(length(Xgrid(:,1)),Nex);
Yex_sj = NaN(length(Xgrid(:,1)),Nex);
Zex_sj = NaN(length(Xgrid(:,1)),Nex);
grad_aj = NaN(length(Xgrid(:,1)),1);
grad_sj = NaN(length(Xgrid(:,1)),1);


%Compute X-Y plane gradient at each magnetopause Z value to extend the grid
for i = 1:length(Xgrid(:,1))

    %Take each row of magnetopause grid corresponding to each Z value
    x = Xgrid(i,:);
    y = Ygrid(i,:);
    z = Zgrid(i,:);

    %Filter out the NaNs
    xfil = x(isnan(x) == 0);
    yfil = y(isnan(y) == 0);
    zfil = z(isnan(z) == 0);

    %First element as anti-Jovian (aj) flank, last element as sub-Jovian (sj) flank
    %(Y positive toward Jupiter)
    
    %Calculate gradient using two edge elements
    x_calc_aj = xfil(1:2); %first two
    y_calc_aj = yfil(1:2);
    x_calc_sj = xfil(end-1:end); %last two
    y_calc_sj = yfil(end-1:end);
    grad_aj_row = (x_calc_aj(1)-x_calc_aj(2))./(y_calc_aj(1)-y_calc_aj(2)); %toward the edge for both flanks
    grad_sj_row = (x_calc_sj(2)-x_calc_sj(1))./(y_calc_sj(2)-y_calc_sj(1));

    %Record extension arrays
    for m = 1:Nex
        Xex_aj(i,m) = x_calc_aj(1) - m*grad_aj_row*res_ex; %x in +ve dir
        Yex_aj(i,m) = y_calc_aj(1) - m*res_ex; %y in -ve direction
        Zex_aj(i,m) = zfil(1); %take first element (constant z)
        %-ve part of grad_aj is attributed to y, not x

        Xex_sj(i,m) = x_calc_sj(2) + m*grad_sj_row*res_ex;
        Yex_sj(i,m) = y_calc_sj(2) + m*res_ex; %y in +ve direction
        Zex_sj(i,m) = zfil(end); %can be index 1 too
    end

    %Record gradient arrays
    grad_aj(i) = grad_aj_row;
    grad_sj(i) = grad_sj_row;
    
end

%Invert row for anti-Jovian arrays for consistency
Xex_aj = flip(Xex_aj,2);
Yex_aj = flip(Yex_aj,2);

end