%Script for generating surface area grids for Biot-Savart integration
%on a half-cylindrical Ganymede magnetopause grid.

%Take as input magnetopause grid produced from jCF_from_MP.m
function [dSgrid] = dSgridKiv98(Ygrid,Zgrid,zmin,zmax,res,lambda3)

%Generate grid point positions in GphiO Z components
zr = zmin+(res./2):res:zmax-(res./2);

%Split magnetopause grids in half using Zgrid < 0 and Zgrid > 0;
YgridS = Ygrid(zr < 0,:);
YgridN = Ygrid(zr > 0,:);
ZgridS = Zgrid(zr < 0,:);
ZgridN = Zgrid(zr > 0,:);

%Rotate magnetopause coordinates from GphiO to GphiB for northern and
%southern magnetopause
[B0y,B0z] = upstream_jovian_field_dir(lambda3); %Compute upstream Jovian magnetic field direction.
theta_rot = atan2d(abs(B0z),B0y)-90; %in degrees
Y_GBN = YgridN.*cosd(theta_rot)-ZgridN.*sind(theta_rot);
Z_GBN = YgridN.*sind(theta_rot)+ZgridN.*cosd(theta_rot);
Y_GBS = YgridS.*cosd(theta_rot)-ZgridS.*sind(theta_rot);
Z_GBS = YgridS.*sind(theta_rot)+ZgridS.*cosd(theta_rot);

%Define constants for Kiv98 half-cylindrical magnetopause grid
a = 2.2;
b = 2.90;
lambda = 0.5;
X0_0 = 0.544;
Y0_max = 0.914;
theta = 0.298;

%Compute Jacobian for northern/southern magnetopause using symbolic variables.
syms Y Z
Yprime = Y.*cosd(theta_rot)-Z.*sind(theta_rot); %GphiO to GphiB
Zprime = Y.*sind(theta_rot)+Z.*cosd(theta_rot);

Y0 = (2/pi)*Y0_max*sind((lambda3-248))*atan(Zprime/lambda); %oval dusk-dawn offset
X0N = X0_0 + Zprime*tan(theta); %Zprime > 0, oval upstream-downstream offset
X0S = X0_0 - Zprime*tan(theta); %Zprime < 0, oval upstream-downstream offset

XN = X0N - a*sqrt(1-((Yprime-Y0)/b).^2); %MP eqn for N hemis
XS = X0S - a*sqrt(1-((Yprime-Y0)/b).^2); %MP eqn for S hemis. Note: XN = -XS

XN_dY = diff(XN,Y); %partial diff w.r.t. Y
XN_dZ = diff(XN,Z); %partial diff w.r.t. Z
XS_dY = diff(XS,Y);
XS_dZ = diff(XS,Z);

JN = sqrt(1 + XN_dY.^2 + XN_dZ.^2); %Jacobian N hemis
JS = sqrt(1 + XS_dY.^2 + XS_dZ.^2); %Jacobian S hemis

%Substitute in Y and Z for numerical solutions for northern and southern magnetopause
Y = YgridN;
Z = ZgridN;
JNgrid = double(subs(JN));
JNgrid = real(JNgrid); %take real part, imaginary effectively zero (manually checked)
Y = YgridS;
Z = ZgridS;
JSgrid = double(subs(JS));
JSgrid = real(JSgrid);

%Join two halves of the Jacobian grid back together
Jgrid = cat(1,JSgrid,JNgrid); %rmb S hemis was the top array

%Find dSgrid for element of size res squared
%dS in unit of res squared
dSgrid = Jgrid.*res.*res;

end