%Function for generating Chapman-Ferraro current densities (in SI units)
%from a half-cylindrical Ganymede magnetopause grid

%Take as input grid parameters (from BCF_surface.m)
function [jCFx,jCFy,jCFz,jCF,Xgrid,Ygrid,Zgrid] = jCF_from_MP(ymin,ymax,zmin,zmax,res,lambda3)

%Generate grid point positions in GphiO Y and Z components
yr = ymin+(res./2):res:ymax-(res./2); 
zr = zmin+(res./2):res:zmax-(res./2);

%Set Jovian upstream conditions.
rhoJ248 = 56; %plasma density at lambda3 = 248 (centre of Jovian plasma sheet) in amu/cm3
pJ248 = 3.8; %plasma pressure at lambda3 = 248 in nPa
BJ248 = 70; %magnetic field at lambda3 = 248 in nT
H = 1.65/sqrt(log(2)); %Plasma sheet scale height
[rhoJ0,pJ0,BJ0] = upstream_parameters(lambda3,rhoJ248,pJ248,BJ248,H); %Find conditions at chosen lambda3 
[Byu,Bzu] = upstream_jovian_field_dir(lambda3); %Project upstream Jovian magnetic field into vector components
BJ0x = 0.0; 
BJ0y = BJ0*Byu;
BJ0z = BJ0*Bzu;
vJ0 = 140; %plasma bulk flow speed in km/s

%Set Ganymede-related parameters
theta_0 = 51; %Co-latitude of open-closed field boundary in degrees. Use 51 for Z = +/-0.63

%Create half-cylindrical fixed magnetopause surface
[Xgrid,Ygrid,Zgrid] = magnetopause_surface_fixed(yr,zr,lambda3,BJ0y,BJ0z);

%Compute Upstream Jovian magnetic pressure
BJx_up = BJ0x*ones(size(Ygrid));
BJy_up = BJ0y*ones(size(Ygrid));
BJz_up = BJ0z*ones(size(Ygrid));
PJb_up = magnetic_pressure(BJx_up,BJy_up,BJz_up);

%Compute flaring angle of plasma flow at each magnetopause grid point (deg)
alpha = flaring_angle(Xgrid,Ygrid,Zgrid,vJ0);

%Compute plasma pressure upstream and at magnetopause.
PJp_up = pJ0*ones(size(Ygrid));
PJp = plasma_pressure(pJ0,alpha);

%Compute Jovian plasma velocity at magnetopause.
[vJx,vJy,vJz,~] = jovian_vel_boundary(vJ0,Xgrid,Ygrid,Zgrid,alpha,yr,zr);

%Compute Jovian plasma mass densities at magnetopause.
rhoJ = jovian_mass_density(rhoJ0,alpha);

%Compute ram pressure upstream and at magnetopause.
%IMPORTANT: ram pressure at boundary DOES NOT act on the magnetopause
[Pr_up, PJr] = dynamical_pressure(rhoJ0,vJ0,rhoJ,vJx,vJy,vJz);

%Compute Jovian magnetic pressure at magnetopause.
PJb = magnetic_pressure_boundary(PJb_up,PJp_up,Pr_up,PJr,PJp);

%Compute Jovian magnetic field at magnetopause.
[BJx,BJy,BJz,~] = jovian_field_boundary(BJx_up,BJy_up,BJz_up,PJb,Xgrid,Ygrid,Zgrid);

%Compute total Jovian pressure acting on the magnetopause
PJact = PJb + PJp;

%Compute Ganymedean magnetic field just inside the boundary required to balance Jovian pressure.
z_ocfb = cosd(theta_0);
[BGx,BGy,BGz,~] = ganymede_field_boundary(PJact,Xgrid,Ygrid,Zgrid,zr,z_ocfb);

%Compute Chapman-Ferraro current density at each magnetopause grid point.
[jCFx,jCFy,jCFz,jCF] = Chapman_Ferraro_current_density(BJx,BJy,BJz,BGx,BGy,BGz,Xgrid,Ygrid,Zgrid);

end