%{

Script to plot evolution of BCF at selected measurement points on Ganymede's 
surface over defined range of System-III Jovian east longitude (equivalent to time).

1) Set up magnetopause grid parameters to produce BCF. For consistency,
surface should be the same as one used in BCF_surface.m.
2) Set range of time through Jovian System-III longitude.
3) Run the script.

%}

%Set up magnetopause grid
ymin = -15; %Minimum GphiO-Y surface limit in RG = 2634 km
ymax = 15; %Maximum GphiO-Y surface limit in RG
zmin = -6; %Minimum GphiO-Z surface limit in RG
zmax = 6; %Maximum GphiO-Y surface limit in RG
res = 0.05; %MP grid resolution in RG.
Nex = 40; %Number of flank extension steps
res_ex = res; %Same Y & Z resolution as previous
Df = 0.95; %Fraction of top-of-ocean radius relative to RG

%Set range of time through Jovian System-III longitude
lambda3_array = 0:2:360; %One synodic period.

%Set Ganymede surface measurement points by range of latitudes and longitudes
lon_list = [-90,0,90,180]; %-180 to 180 range
lat_list = [-90,-55,-35,0,35,55,90]; %-90 to 90 range
[lon_mesh,lat_mesh] = meshgrid(lon_list,lat_list); %Make grid for ease of manipulation.

%Convert spherical measurement points' coordinates to Cartesian (R = 0.95)
%for east (clockwise) Ganymedean longitude with Y-axis as reference
Xm = Df.*cosd(lat_mesh).*sind(lon_mesh);
Ym = Df.*cosd(lat_mesh).*cosd(lon_mesh);
Zm = Df.*sind(lat_mesh);
disp('Measurement points obtained')


%Set up 3D BCF output arrays.
%Each field index corresponds to (lat, lon, lambda3).
BCFx_array = NaN(length(lat_list),length(lon_list),length(lambda3_array));
BCFy_array = NaN(length(lat_list),length(lon_list),length(lambda3_array));
BCFz_array = NaN(length(lat_list),length(lon_list),length(lambda3_array));
BCF_array = NaN(length(lat_list),length(lon_list),length(lambda3_array));


%Calculate BCF over each System-III longitudes, then loop for all longitudes.
for i = 1:length(lambda3_array)
    lambda3 = lambda3_array(i);
    
    %Half-cylindrical BCF
    [jCFx_K98,jCFy_K98,jCFz_K98,jCF_K98,Xgrid,Ygrid,Zgrid] = jCF_from_MP(ymin,ymax,zmin,zmax,res,lambda3);
    [dSgrid_K98] = dSgridKiv98(Ygrid,Zgrid,zmin,zmax,res,lambda3);
    [BCFx_K98,BCFy_K98,BCFz_K98,BCF_K98] = BCF_from_MP(Xgrid,Ygrid,Zgrid,jCFx_K98,jCFy_K98,jCFz_K98,dSgrid_K98,Xm,Ym,Zm);

    %Flank extensions BCF 
    [Xex_aj,Yex_aj,Zex_aj,Xex_sj,Yex_sj,Zex_sj,grad_aj,grad_sj] = MPex(Xgrid,Ygrid,Zgrid,Nex,res_ex);
    [jCFx_aj,jCFy_aj,jCFz_aj,jCF_aj] = jCF_MPex(Xex_aj,Yex_aj,Zex_aj,res_ex,lambda3);
    [jCFx_sj,jCFy_sj,jCFz_sj,jCF_sj] = jCF_MPex(Xex_sj,Yex_sj,Zex_sj,res_ex,lambda3);
    [dSgrid_aj,dSgrid_sj] = dSgridEx(grad_aj,grad_sj,Nex,res_ex);
    [BCFx_aj,BCFy_aj,BCFz_aj,BCF_aj] = BCF_from_MP(Xex_aj,Yex_aj,Zex_aj,jCFx_aj,jCFy_aj,jCFz_aj,dSgrid_aj,Xm,Ym,Zm);
    [BCFx_sj,BCFy_sj,BCFz_sj,BCF_sj] = BCF_from_MP(Xex_sj,Yex_sj,Zex_sj,jCFx_sj,jCFy_sj,jCFz_sj,dSgrid_sj,Xm,Ym,Zm);
    
    %Sum BCF contributions
    BCFx = BCFx_K98 + BCFx_aj + BCFx_sj;
    BCFy = BCFy_K98 + BCFy_aj + BCFy_sj;
    BCFz = BCFz_K98 + BCFz_aj + BCFz_sj;
    BCF = sqrt(BCFx.^2+BCFy.^2+BCFz.^2);

    %Record BCF into output arrays
    for m = 1:length(lat_list)
        for n = 1:length(lon_list)
            BCFx_array(m,n,i) = BCFx(m,n);
            BCFy_array(m,n,i) = BCFy(m,n);
            BCFz_array(m,n,i) = BCFz(m,n);
            BCF_array(m,n,i) = BCF(m,n);
        end
    end

    disp(strcat(string(lambda3),' Jovian longitude finished'))

end    