%Function for computing Biot-Savart integration of Chapman-Ferraro
%magnetic field from Ganymede's magnetopause, taking as input 2D grids of
%current sheet surface and 2D grids of measurement locations. The function
%works for both half-cylindrical and flank magnetopause sections.

function [BCFx,BCFy,BCFz,BCF] = BCF_from_MP(Xj,Yj,Zj,jCFx,jCFy,jCFz,dSgrid,Xm,Ym,Zm)
%j = current sheet, m = measurement
%spatial inputs in RG, dSgrid in RG^2
%current density in A/m

%Set up constants
mu0 = 4*pi*10^-7;
RG = 2634*10^3; %in m 

%Set dSgrid NaNs to zeros for summation purposes
dSgrid(isnan(dSgrid)) = 0;

%Convert dSgrid dimension from RG to meters (SI)
dSgrid = dSgrid.*RG.*RG;

%Set up final BCF arrays
BCFx = NaN(size(Xm));
BCFy = NaN(size(Xm));
BCFz = NaN(size(Xm));
BCF = NaN(size(Xm));

%Compute BCF and append to final arrays
for n = 1:length(Xm(:,1))  %loop over each measurement point in 2D array
    for m = 1:length(Xm(1,:))
        %Compute position vector components between this measurement point
        %and ALL magnetopause grid points
        Rx = Xm(n,m) - Xj;
        Ry = Ym(n,m) - Yj;
        Rz = Zm(n,m) - Zj;
        R = sqrt(Rx.^2 + Ry.^2 + Rz.^2);

        %Convert units to meters
        Rx = Rx.*RG;
        Ry = Ry.*RG;
        Rz = Rz.*RG;
        R = R.*RG;

        %Compute JxR cross product & set all NaN locations to zero
        Vx = jCFy.*Rz - jCFz.*Ry;
        Vy = jCFz.*Rx - jCFx.*Rz;
        Vz = jCFx.*Ry - jCFy.*Rx;

        %Compute dBCF for each current sheet grid face
        Pt_dBCFx = (mu0./(4*pi)).*(dSgrid./R.^3).*Vx;
        Pt_dBCFy = (mu0./(4*pi)).*(dSgrid./R.^3).*Vy;
        Pt_dBCFz = (mu0./(4*pi)).*(dSgrid./R.^3).*Vz;

        %Remove NaN from Pt_dBCF (from previous arrays and divide by 0 in Rsq)
        %have to remove because can't sum up NaNs
        Pt_dBCFx(isnan(Pt_dBCFx)) = 0;
        Pt_dBCFy(isnan(Pt_dBCFy)) = 0;
        Pt_dBCFz(isnan(Pt_dBCFz)) = 0;

        %Sum up all elements for BCF components at this measurement point.
        %Sum gives NaN if there is NaN left in the array.
        Pt_BCFx = sum(Pt_dBCFx,'all');
        Pt_BCFy = sum(Pt_dBCFy,'all');
        Pt_BCFz = sum(Pt_dBCFz,'all');

        %Convert to nT.
        Pt_BCFx = Pt_BCFx*10^9;
        Pt_BCFy = Pt_BCFy*10^9;
        Pt_BCFz = Pt_BCFz*10^9;

        %Record these points in final BCF arrays
        BCFx(n,m) = Pt_BCFx;
        BCFy(n,m) = Pt_BCFy;
        BCFz(n,m) = Pt_BCFz;
        BCF(n,m) = sqrt(Pt_BCFx.^2+Pt_BCFy.^2+Pt_BCFz.^2);

    end    
end


end