%{
Script for computing Chapman-Ferraro magnetic field (BCF) at either 
top-of-ocean sphere or magnetospheric cross-sectional plane.

1) Set up magnetopause grid parameters to produce BCF
2) Choose (by commenting) measurement surface structure
3) Choose (by commenting) appropriate save structure (see comments below)
4) If required, define position in the save cell the data will go into.
3) Run script each for System-III east longitude required.

BCF is computed separately for central half-cylindrical ('Kiv98') section
and flank extension section, then summed together.
%}

%Set up MP grid
ymin = -15; %Minimum GphiO-Y surface limit in RG = 2634 km
ymax = 15; %Maximum GphiO-Y surface limit in RG
zmin = -6; %Minimum GphiO-Z surface limit in RG
zmax = 6; %Maximum GphiO-Y surface limit in RG
res = 0.05; %MP grid resolution in RG.
Nex = 40; %Number of flank extension steps
res_ex = res; %Same Y & Z resolution as previous
Df = 0.95; %Fraction of top-of-ocean radius relative to RG
lambda3 = 248; %Jovian System-III east longitude in degrees
disp('Magnetopause surface finished setup')

%Target 1: Top-of-ocean sphere
%lat_list = linspace(-90,90,181); %Ganymedean latitude range (min,max,res) in degrees 
%lon_list = linspace(0,360,181); %Ganymedean longitude range
%[lon_mesh,lat_mesh] = meshgrid(lon_list,lat_list); %Create measurement grid
%Xm = Df.*cosd(lat_mesh).*sind(lon_mesh); %Calculate GphiO coordinates of measurement points
%Ym = Df.*cosd(lat_mesh).*cosd(lon_mesh);
%Zm = Df.*sind(lat_mesh);
%disp('measurement sphere obtained')

%Target 2: Magnetospheric cross-sectional plane
Xm_range = -2.0:0.05:2.0; %Range of plane in GphiO-X (min,max,res) in RG
Ym_range = -2.0:0.05:2.0; %Range of plane in GphiO-Y in RG
Z0 = 0; %Plane location in GphiO-Z
[Xm,Ym] = meshgrid(Xm_range,Ym_range); %Create measurement grid
Zm = Z0*ones(size(Xm)); %Create GphiO-Z positions for each grid point.
disp('measurement plane obtained')


%%%%%% Kiv98 MP Surface %%%%%

%Obtain Chapman-Ferraro current (jCF) at each measurement point.
[jCFx_K98,jCFy_K98,jCFz_K98,jCF_K98,Xgrid,Ygrid,Zgrid] = jCF_from_MP(ymin,ymax,zmin,zmax,res,lambda3);
disp('jCF obtained (Kiv98)')

%Obtain dS Jacobian projection at each measurement point.
[dSgrid_K98] = dSgridKiv98(Ygrid,Zgrid,zmin,zmax,res,lambda3);
disp('dS obtained (Kiv98)')

%Compute Biot-Savart integration at each measurement point for
%central half-cylindrical BCF contribution.
[BCFx_K98,BCFy_K98,BCFz_K98,BCF_K98] = BCF_from_MP(Xgrid,Ygrid,Zgrid,jCFx_K98,jCFy_K98,jCFz_K98,dSgrid_K98,Xm,Ym,Zm);
disp('BCF obtained (Kiv98)')

%%%%%% MP Extension %%%%%

%Create magnetopause flanks extensions on sub-Jovian and anti-Jovian flanks
[Xex_aj,Yex_aj,Zex_aj,Xex_sj,Yex_sj,Zex_sj,grad_aj,grad_sj] = MPex(Xgrid,Ygrid,Zgrid,Nex,res_ex);

%Obtain Chapman-Ferraro current (jCF) at each measurement point.
[jCFx_aj,jCFy_aj,jCFz_aj,jCF_aj] = jCF_MPex(Xex_aj,Yex_aj,Zex_aj,res_ex,lambda3);
[jCFx_sj,jCFy_sj,jCFz_sj,jCF_sj] = jCF_MPex(Xex_sj,Yex_sj,Zex_sj,res_ex,lambda3);
disp('jCF obtained (Flank Extension)')

%Obtain dS Jacobian projection at each measurement point.
[dSgrid_aj,dSgrid_sj] = dSgridEx(grad_aj,grad_sj,Nex,res_ex);
disp('dS obtained (Flank Extension)')

%Compute Biot-Savart integration at each measurement point
%Anti-Jovian flank contribution
[BCFx_aj,BCFy_aj,BCFz_aj,BCF_aj] = BCF_from_MP(Xex_aj,Yex_aj,Zex_aj,jCFx_aj,jCFy_aj,jCFz_aj,dSgrid_aj,Xm,Ym,Zm);
%Sub-Jovian flank. contribution
[BCFx_sj,BCFy_sj,BCFz_sj,BCF_sj] = BCF_from_MP(Xex_sj,Yex_sj,Zex_sj,jCFx_sj,jCFy_sj,jCFz_sj,dSgrid_sj,Xm,Ym,Zm);
disp('BCF obtained (Flank Extension)')

%%%%% Sum All BCF Contributions %%%%%

BCFx = BCFx_K98 + BCFx_aj + BCFx_sj;
BCFy = BCFy_K98 + BCFy_aj + BCFy_sj;
BCFz = BCFz_K98 + BCFz_aj + BCFz_sj;
BCF = sqrt(BCFx.^2+BCFy.^2+BCFz.^2);

%%%%% Save appropriate data for plotting %%%%%

%For Figure 2 (Spherical BCF plots at extreme System-III longitudes)
%{
if lambda3 == 158
    Gpos = 1;
elseif lambda3 == 248
    Gpos = 2;
elseif lambda3 == 338
    Gpos = 3;
else
    return
end

Xmcell{Gpos} = Xm;
Ymcell{Gpos} = Ym;
Zmcell{Gpos} = Zm;
BCFxcell{Gpos} = BCFx;
BCFycell{Gpos} = BCFy;
BCFzcell{Gpos} = BCFz;
BCFcell{Gpos} = BCF;

%}

%For Figure 5 (3x3 grid at arbitrary System-III longitudes)
%Double check cell position to avoid overwriting.
row = 3;
col = 3;

Xmcell{row,col} = Xm;
Ymcell{row,col} = Ym;
Zmcell{row,col} = Zm;
BCFxcell{row,col} = BCFx;
BCFycell{row,col} = BCFy;
BCFzcell{row,col} = BCFz;
BCFcell{row,col} = BCF;
lambda3cell{row,col} = lambda3;
Z0cell{row,col} = Z0;
