%Function to create fixed half-cylindrical magnetopause based on Kivelson et al., 1998. 
%The code uses GphiO coordinate system. Units are in Ganymede radii and degrees

%Take as input magnetopause surface limits, upstream magnetic field, and
%Jovian System-III east longitude.
function [Xplot2,Yplot2,Zplot2] = magnetopause_surface_fixed(y,z,lambda3,B0y,B0z)
%Make magnetopause surface grid
[Y,Z] = meshgrid(y,z);

%Rotate GphiO to GphiB. Take |B0z| to prevent Z-axis sign change at lambda3 = 248
theta_rot = atan2d(abs(B0z),B0y)-90; 
Y_prime = Y.*cosd(theta_rot)-Z.*sind(theta_rot);
Z_prime = Y.*sind(theta_rot)+Z.*cosd(theta_rot);

%Set up magnetopause surface constants
a = 2.2;
b = 2.90;
lambda = 0.5;
X0_0 = 0.544;
Y0_max = 0.914;
theta = 0.298; %not the same as rotation angle.

%Calculate cooresponding X for each (Y,Z) grid position
X0 = X0_0 + abs(Z_prime)*tan(theta);
Y0 = (2/pi)*Y0_max*sin((lambda3-248)*pi/180)*atan(Z_prime/lambda);

%Take negative solution for upstream half-cylinder
X = X0 - a*sqrt(1-((Y_prime-Y0)/b).^2);

%Fill in final grid arrays
Xplot2 = NaN(length(z),length(y));
Yplot2 = NaN(length(z),length(y));
Zplot2 = NaN(length(z),length(y));

for i = 1:length(z) 
    for j = 1:length(y)
        if imag(X(i,j)) == 0 
            Xplot2(i,j) = real(X(i,j)); %Weed out imaginary values to plot.     
            Yplot2(i,j) = real(Y(i,j));            
            Zplot2(i,j) = real(Z(i,j));
        end
    end
end
end
