import os

from pymatgen.core import Structure, Composition, Molecule
from pymatgen.io.vasp.sets import MPRelaxSet
from pymatgen.io.vasp.inputs import Poscar

#Custom vasp input settings since pymatgen is a little old and meh
# ALGO      = Normal        ! Fast algo gives +10% perf at some matrix inverse failing risks
# ISMEAR    = 0             ! unknown bandstructure
# SIGMA     = 0.05          ! small sigma yields better smearing
# LREAL     = False         ! LREAL = Auto/True is really only needed for large cells
# PREC      = "Accurate"    ! PREC : Accurate (Normal has been determined to be meh for these systems)
# NCORE     = 4             ! For better parr
# EDIFF     = 4E-06         ! Should be sufficent for near high-throughput + this size system size
# LWAVE     = True/False          ! Sick of keeping this flag true for high-throughput but its not an issue for min cases
# EDIFFG    = -0.010        ! Force convergence is important so will make this small ish

base_incar_settings={"ALGO": "Normal", "ISMEAR": 0, "SIGMA": 0.05, "LREAL": False,
                     "PREC": "Accurate", "NCORE": 4, "EDIFF": 1E-07, "EDIFFG": -0.005, "IVDW":0}

init_files = [("o3_r3m.vasp", "o3"), ("o2_r3m.vasp", "o2"), ("p2_r3m.vasp", "p2"), ("fd3m.vasp", "fd")]
m_list = ["Co", "Mn", "Ni"]

for file in init_files:
    structure = Structure.from_file(filename=f"structures/{file[0]}")
    for metal in [("Co", "cobalt"), ("Mn", "manganese"), ("Ni", "nickel")]:
        os.makedirs(f"data/{metal[1]}", exist_ok=True)

        for atom in structure:
            if atom.species_string in m_list:
                atom.species = metal[0]

        incar = MPRelaxSet(structure=structure, user_incar_settings=base_incar_settings).incar
        number_metal = len([x for x in structure if x.species_string in m_list])

        if metal[0] == "Co":
            mm = number_metal * [0.0] + number_metal * [1.0] + 2*number_metal * [0.05]
        elif metal[0] == "Mn":
            mm = number_metal * [0.0] + number_metal * [4.0] + 2*number_metal * [0.05]
        elif metal[0] == "Ni":
            mm = number_metal * [0.0] + number_metal * [1.0] + 2*number_metal * [0.05]

        incar["MAGMOM"] = mm
        folder_string = f"data/{metal[1]}/{file[1]}/full_lithiation/"
        os.makedirs(f"{folder_string}", exist_ok=True)

        incar.write_file(filename=f"{folder_string}/INCAR_base")

        for functional in [(0, "no_IVDW"), (1, "VDW_1"), (11, "VDW_2")]:
            # generate both no IVDW+U and no IVDW+noU
            incar_copy = incar
            incar_copy["IVDW"] = functional[0]
            incar_copy["LDAU"] = True
            f2 = folder_string + "/" + functional[1]

            os.makedirs(f"{f2}", exist_ok=True)
            incar_copy.write_file(filename=f"{f2}/INCAR")
            structure.to(filename=f"{f2}/initial_poscar.vasp", fmt="poscar")
            structure.to(filename=f"{f2}/POSCAR", fmt="poscar")

            if functional[0] == 0:
                incar_copy_2 = incar_copy
                incar_copy_2["LDAU"] = False
                f2 = folder_string + "/noU"
                os.makedirs(f"{f2}", exist_ok=True)
                incar_copy_2.write_file(filename=f"{f2}/INCAR")
                structure.to(filename=f"{f2}/initial_poscar.vasp", fmt="poscar")
                structure.to(filename=f"{f2}/POSCAR", fmt="poscar")
