# notes
# Function for processing hplc outputs alongside sample extraction weights
# data are processed to produce a value for mg / gram (dry weight) for ascorbic, malic, and citric acids

# changes
# changed the "sample" variable name to "sample_name" to avoid conflicts with global functions
# updated experiment_id to work with experiments that don't have a "harvest_date" variable
# converted instances of name == "malic" or "citric" to "malic/citric acid"
 


if (!require("pacman")) install.packages("pacman")

pacman::p_load(tidyverse)

organic_acids <- function(hplc_data, dry_weight, experiment_id ,out_dir, tidy = TRUE
) {
  acids <- list.files(
    path = hplc_data,
    pattern = "*results.csv",
    full.names = TRUE
  ) %>%
    map_df(
      ~ read_csv(.)
    ) %>%
    janitor::clean_names() %>% 
    mutate(
      across(
        where(
          is.character
        ), 
        str_trim
      ),
      sample_name = str_remove(sample_name, "K")
    )
  
  
  dryweight <- list.files(
    path = dry_weight,
    pattern = "*extraction_weights.csv",
    full.names = TRUE
  ) %>%
    map_df(
      ~ read_csv(.)
    ) %>%
    janitor::clean_names() %>%
    mutate(
      sample_name = str_remove(sample_name, "K"),
      harvest_date = case_when(
        str_detect(sample_name, "W1") ~ "06/07/2023",
        str_detect(sample_name, "W2") ~ "13/07/2023",
        str_detect(sample_name, "W3") ~ "20/07/2023"#,
        #TRUE ~ harvest_date
      ),
      across(
        where(
          is.character
        ), 
        str_trim
      )
    )
  
  
  acids_reduce <- acids %>% 
    filter(
      str_detect(
        name, "ascorbic acid|AsA|asa|malic|citric|CRM|IHRM"
      ),
      !str_detect(
        sample_name, "std"
      )
    ) %>%
    select(
      sample_name,
      name,
      concentration,
      injection_volume_ul,
      amount
    )  %>% 
    drop_na()
  
  acids_process <- acids_reduce %>%
    left_join(
      dryweight, by = "sample_name"
    ) %>% 
    group_by(
      name
    ) %>% 
    mutate(
      weight_correction = rep(0.89, times = n()),
      extraction_volume = rep(5, times = n()),
      dilution = rep(0.5, times = n()),
      mg_per_g = (concentration / dilution) / ((weight_grams * weight_correction) / extraction_volume),
      experiment = case_when(
        str_detect(harvest_date, "2023") ~ str_glue("{experiment_id}23"),
        str_detect(harvest_date, "2024") ~ str_glue("{experiment_id}24"),
        is.na(harvest_date) ~ experiment_id
      ),
      sample = str_remove(
        sample_name, "-A|W[0-9]-"
      ),
      measure_var = case_when(
        str_detect(name, "asa|AsA|ascorbic") ~ "ascorbic acid",
        str_detect(name, "malic") ~ "malic acid",
        str_detect(name, "citric") ~ "citric acid"
        )
    ) %>% 
    arrange(
      sample_name
    ) %>% 
    ungroup() %>% 
    select(
      harvest_date, experiment, sample_name, measure_var, mg_per_g
    ) %>% 
    rename(
      date = harvest_date,
      value = mg_per_g
    ) %>% 
    mutate(
      measure_var = str_replace_all(
        measure_var, "\\s", "_"
      )
    ) %>% 
    drop_na(
      value
      ) %>% 
    ungroup()
  
  if (tidy == TRUE) assign("acids_process" ,acids_process, envir = .GlobalEnv)
  if (tidy == FALSE) return(c(acids,dryweight,acids_reduce,acids_process))
  
  write_csv(acids_process, paste0(str_glue("{out_dir}", format(Sys.Date(), "%Y%m%d"), "_organicacids_calcs.csv")))
}