# notes
# Function for processing hplc outputs alongside sample extraction weights
# data are processed to produce a value for mg / gram (dry weight) for ascorbic, malic, and citric acids

# changes
# removed mg_per_ml again as I don't think it is necessary



if (!require("pacman")) install.packages("pacman")

pacman::p_load(tidyverse)

organic_acids <- function(hplc_data, dry_weight, out_dir, tidy = TRUE
) {
  acids <- list.files(
    path = hplc_data,
    pattern = "*results.csv",
    full.names = TRUE
  ) %>%
    map_df(
      ~ read_csv(.)
    ) %>%
    janitor::clean_names() %>% 
    mutate(
      across(
        where(
          is.character
        ), 
        str_trim
      ),
      sample_name = str_remove(sample_name, "K")
    )
  
  
  dryweight <- list.files(
    path = dry_weight,
    pattern = "*extraction_weights.csv",
    full.names = TRUE
  ) %>%
    map_df(
      ~ read_csv(.)
    ) %>%
    janitor::clean_names() %>%
    mutate(
      sample = str_remove(sample, "K"),
      harvest_date = case_when(
        str_detect(sample, "W2") ~ "13/07/2023",
        str_detect(sample, "W3") ~ "20/07/2023",
        TRUE ~ harvest_date
      ),
      across(
        where(
          is.character
        ), 
        str_trim
      )
    )
  
  
  acids_reduce <- acids %>% 
    filter(
      str_detect(
        name, "ascorbic acid|AsA|asa|malic|citric|CRM|IHRM"
      ),
      !str_detect(
        sample_name, "std"
      )
    ) %>% 
    rename(
      sample = sample_name
    ) %>% 
    select(
      sample,
      name,
      concentration,
      injection_volume_ul,
      amount
    )  %>% 
    drop_na()
  
  acids_process <- acids_reduce %>%
    left_join(
      dryweight, by = "sample"
    ) %>% 
    group_by(
      name
    ) %>% 
    mutate(
      weight_correction = rep(0.89, times = n()),
      extraction_volume = rep(5, times = n()),
      dilution = rep(0.5, times = n()),
      mg_per_g = (concentration / dilution) / ((weight_grams * weight_correction) / extraction_volume),
      experiment = if_else(
        str_detect(
          harvest_date, "2023"
        ),
        "VT23", "VT24"
      ),
      sample = str_remove(
        sample, "-A|W[0-9]-"
      ),
      measure_var = str_replace_all(
        name, "asa|AsA", "ascorbic acid"
      )
    ) %>% 
    arrange(
      sample
    ) %>% 
    ungroup() %>% 
    select(
      harvest_date, experiment, sample, measure_var, mg_per_g
    ) %>% 
    # group_by(
    #   sample , name
    # ) %>% 
    # summarise(
    #   across(
    #     mg_per_gram,
    #     ~ mean(.x, na.rm = T)
    #   )
    # ) %>% 
    # ungroup() %>% 
    rename(
      date = harvest_date,
      value = mg_per_g
    ) %>% 
    mutate(
      measure_var = str_replace_all(
        measure_var, "\\s", "_"
      )
    ) %>% 
    drop_na(
    ) %>% 
    ungroup()
  
  if (tidy == TRUE) assign("acids_process" ,acids_process, envir = .GlobalEnv)
  if (tidy == FALSE) return(c(acids,dryweight,acids_reduce,acids_process))
  
  write_csv(acids_process, paste0(str_glue("{out_dir}", format(Sys.Date(), "%Y%m%d"), "_organicacids_calcs.csv")))
}