import os
import pandas as pd
import openpyxl

def float_to_fixed_point(inputValue):
    # 定义参数
    num_bits = 24
    sign_bit = 1
    int_bits = 7
    frac_bits = 16

    # 限定整数和小数位的最大范围
    max_int_value = (1 << int_bits) - 1
    min_int_value = -(1 << int_bits)
    max_frac_value = 1 << frac_bits

    # 提取符号
    sign = 0 if inputValue >= 0 else 1
    abs_value = abs(inputValue)

    # 将数值乘以 2**16
    fixed_value = int(round(abs_value * max_frac_value))

    # 获取整数和小数部分
    integer_part = fixed_value >> frac_bits
    fractional_part = fixed_value & (max_frac_value - 1)

    # 检查整数部分是否溢出
    if integer_part > max_int_value:
        raise ValueError(f"Integer part {integer_part} exceeds limit of {max_int_value}")

    # 创建二进制字符串
    binary_str = f"{sign:1b}{integer_part:05b}{fractional_part:010b}"

    # 将二进制字符串转换为24位整数
    fixed_point_value = int(binary_str, 2)

    # 转换为十六进制字符串
    hex_str = f"{fixed_point_value:06X}"

    return hex_str

# 加载 Excel 文件和工作表
excel_path = "vco_matrix_64_12.xlsx"
sheet_name = "Matrix Data"
wb = openpyxl.load_workbook(excel_path)
ws = wb[sheet_name]

# 创建目标文件夹
output_folder = "inputData"
if not os.path.exists(output_folder):
    os.makedirs(output_folder)

# 获取合并单元格的值
def get_merged_cell_value_by_index(ws, col_idx, row):
    """
    获取合并单元格的实际值，根据列索引。未合并的单元格则直接返回其值。
    """
    for merged_range in ws.merged_cells.ranges:
        min_row, min_col, max_row, max_col = merged_range.bounds
        if min_row <= row <= max_row and min_col == col_idx + 1:  # 索引从0开始，openpyxl列从1开始
            return ws.cell(row=min_row, column=min_col).value
    return ws.cell(row=row, column=col_idx + 1).value

# 定义批次大小
batch_size = 16
start_idx = 2  # 从第2行开始
end_idx = ws.max_row  # 最大行数

# 自定义函数，将每一行数据转换为按需删除 NA 的字符串
def process_row(row):
    """
    删除每一行中 'NA' 值的单元格以及该单元格前的制表符。
    """
    processed_cells = [cell for cell in row if cell != ""]
    return processed_cells

# 逐个生成文件
for batch_number, start_row in enumerate(range(start_idx, end_idx + 1, batch_size), start=1):
    current_batch_start = start_row
    current_batch_end = min(start_row + batch_size - 1, end_idx)

    # 获取第 1 列和第 18 列的值（分别是列索引 0 和 17）
    a_col_value = get_merged_cell_value_by_index(ws, 0, current_batch_start) or "DefaultA"
    r_col_value = get_merged_cell_value_by_index(ws, 17, current_batch_start) or "DefaultR"

    # 使用 Pandas 读取 B 列到 Q 列数据（即第2列到第17列）
    df = pd.DataFrame(ws.iter_rows(min_row=current_batch_start, max_row=current_batch_end, min_col=2, max_col=17, values_only=True))

    # 替换 'NA' 为空字符串
    df_clean = df.replace("NA", "", regex=True)

    # 处理每一行并删除空单元格
    processed_rows = [process_row(row) for row in df_clean.values]

    # 创建新的数据框并删除全为空的行和列
    processed_df = pd.DataFrame(processed_rows).dropna(axis=0, how='all').dropna(axis=1, how='all')

    # 检查处理后的形状，如果满足条件则进行转置
    if processed_df.shape == (16, 15):
        # 转置为 15 行 16 列
        transposed_df = processed_df.T
        print(f"批次 {batch_number}: 转置为 15 行 16 列")
    else:
        transposed_df = processed_df
        print(f"批次 {batch_number}: 保留原始数据（{processed_df.shape[0]} 行，{processed_df.shape[1]} 列）")

    # 为每个批次生成唯一文件名并存储在目标文件夹
    file_name = f"{a_col_value}_{r_col_value}_Batch{batch_number}.txt"
    file_path = os.path.join(output_folder, file_name)

    # 将数据写入文件
    with open(file_path, "w", encoding="utf-8") as f:
        for row in transposed_df.values:
            # f.write("\t".join(map(str, row)) + "\n")
            fixed_point_values = [float_to_fixed_point(float(value)) for value in row]
            f.write("\t".join(fixed_point_values) + "\n")

    print(f"批次 {batch_number} 数据已写入文件：{file_path}")
