import asyncio
import tkinter as tk
from tkinter import messagebox, filedialog
from tkinter import ttk, font
from bleak import BleakClient, BleakScanner, BleakError
import pandas as pd
import sys
import time
import math
from datetime import datetime  # Added datetime for timestamp

# Device details
DEVICE_NAME = "Skin Capacitance System"
SERVICE_UUID = "12345678-1234-1234-1234-1234567890AB"
CHARACTERISTIC_UUID = "ABCD1234-5678-1234-5678-1234567890CD"

# Capacitance calculation parameters
DIVIDER = 1  # clock divider is set to 1
TwoPow28 = 2**28
FREF = 40*(10**6)  # frequency of 40 MHz
Csensor = 47  # in pF
Lsensor = 470  # in uH
sensorArea = 0.384  # for G100/200/400

class BLEApp:
    def __init__(self, master):
        self.master = master
        master.title("BLE Device Interface")

        self.client = None
        self.data = []
        self.loop = asyncio.get_event_loop()

        self.is_collecting = False

        # Create main frame
        self.main_frame = tk.Frame(master)
        self.main_frame.pack(pady=10)

        # Create left and right frames
        self.left_frame = tk.Frame(self.main_frame)
        self.left_frame.pack(side=tk.LEFT, padx=10)

        self.right_frame = tk.Frame(self.main_frame)
        self.right_frame.pack(side=tk.LEFT, padx=10)

        # Bold font
        bold_font = font.Font(weight="bold")

        # GUI Elements in left frame
        self.connect_button = tk.Button(self.left_frame, text="Connect", command=self.connect_device)
        self.connect_button.pack(pady=5)

        # Status label with bold font and initial color red
        self.status_label = tk.Label(self.left_frame, text="Status: Disconnected", font=bold_font, fg="red")
        self.status_label.pack(pady=5)

        self.start_button = tk.Button(self.left_frame, text="Start", command=self.start_collection, state=tk.DISABLED)
        self.start_button.pack(pady=5)

        self.stop_button = tk.Button(self.left_frame, text="Stop", command=self.stop_collection, state=tk.DISABLED)
        self.stop_button.pack(pady=5)

        self.disconnect_button = tk.Button(self.left_frame, text="Disconnect", command=self.disconnect_device, state=tk.DISABLED)
        self.disconnect_button.pack(pady=5)

        self.save_button = tk.Button(self.left_frame, text="Save Data", command=self.save_data, state=tk.DISABLED)
        self.save_button.pack(pady=5)

        self.clear_button = tk.Button(self.left_frame, text="Clear Data", command=self.clear_data, state=tk.DISABLED)
        self.clear_button.pack(pady=5)

        # GUI Elements in right frame
        self.person_label = tk.Label(self.right_frame, text="Select patient, control or volunteer:", font=bold_font)
        self.person_label.pack(pady=5)

        self.person_var = tk.StringVar(value="Patient")
        self.person_dropdown = tk.OptionMenu(self.right_frame, self.person_var, "Patient", "Volunteer", "Control")
        self.person_dropdown.config(font=bold_font)  # Make the OptionMenu button text bold
        self.person_dropdown['menu'].config(font=bold_font)  # Make the dropdown options bold
        self.person_dropdown.pack(pady=5)

        # Study number entry
        # Study ID entry setup in right frame
        self.study_id_label = tk.Label(self.right_frame, text="Enter study ID:", font=bold_font)
        self.study_id_label.pack(pady=5)

        # Frame for the study ID entries
        self.study_id_frame = tk.Frame(self.right_frame)
        self.study_id_frame.pack(pady=5)
        # Default study number format
        # Label and entry for each part of the study ID
        tk.Label(self.study_id_frame, text="S -", font=bold_font).pack(side=tk.LEFT)

        self.study_part1_var = tk.StringVar(value="00")
        self.study_part1_entry = tk.Entry(self.study_id_frame, textvariable=self.study_part1_var, font=bold_font,
                                          width=3)
        self.study_part1_entry.pack(side=tk.LEFT)

        tk.Label(self.study_id_frame, text="-").pack(side=tk.LEFT)

        self.study_part2_var = tk.StringVar(value="XX")  # Patient number part
        self.study_part2_entry = tk.Entry(self.study_id_frame, textvariable=self.study_part2_var, font=bold_font,
                                          width=3)
        self.study_part2_entry.pack(side=tk.LEFT)

        tk.Label(self.study_id_frame, text="-").pack(side=tk.LEFT)

        self.study_part3_var = tk.StringVar(value="000")
        self.study_part3_entry = tk.Entry(self.study_id_frame, textvariable=self.study_part3_var, font=bold_font,
                                          width=4)
        self.study_part3_entry.pack(side=tk.LEFT)

        tk.Label(self.study_id_frame, text="-").pack(side=tk.LEFT)

        self.study_part4_var = tk.StringVar(value="00")
        self.study_part4_entry = tk.Entry(self.study_id_frame, textvariable=self.study_part4_var, font=bold_font,
                                          width=3)
        self.study_part4_entry.pack(side=tk.LEFT)

        # Lesional or non-lesional entry
        self.skin_type_label = tk.Label(self.right_frame, text="Select skin type for measurement:", font=bold_font)
        self.skin_type_label.pack(pady=5)

        self.skin_type_var = tk.StringVar(value="Lesional")
        self.skin_type_dropdown = tk.OptionMenu(self.right_frame, self.skin_type_var, "Lesional", "Non-Lesional")
        self.skin_type_dropdown.config(font=bold_font)  # Make the OptionMenu button text bold
        self.skin_type_dropdown['menu'].config(font=bold_font)  # Make the dropdown options bold
        self.skin_type_dropdown.pack(pady=5)

        self.skin_type_label = tk.Label(self.right_frame, text="Enter test number:", font=bold_font)
        self.skin_type_label.pack(pady=5)

        self.test_number_var = tk.StringVar(value="1")
        self.test_number_entry = tk.Entry(self.right_frame, textvariable=self.test_number_var, font=bold_font, width=3)
        self.test_number_entry.pack(pady=5)

        # Label for "Remove Sensor" notification
        self.notification_label = tk.Label(master, text="", fg="red")
        self.notification_label.pack(pady=5)

        # Treeview Table for data display with columns
        columns = ('Pressure Level', 'Pressure Value', 'Raw Data', 'Sensor Capacitance (pF)')
                   # 'Normalized Capacitance (pF/cm2)')
        self.tree = ttk.Treeview(master, columns=columns, show='headings', height=15)
        self.tree.heading('Pressure Level', text='Pressure Level:')
        self.tree.heading('Pressure Value', text='Pressure Value')
        self.tree.heading('Raw Data', text='Raw Data')
        self.tree.heading('Sensor Capacitance (pF)', text='Sensor Capacitance (pF)')
        # self.tree.heading('Normalized Capacitance (pF/cm2)', text='Normalized Capacitance (pF/cm\u00B2)')

        # Define column widths
        self.tree.column('Pressure Level', width=150, stretch=True)
        self.tree.column('Pressure Value', width=100, stretch=True)
        self.tree.column('Raw Data', width=100, stretch=True)
        self.tree.column('Sensor Capacitance (pF)', width=200, stretch=True)
        # self.tree.column('Normalized Capacitance (pF/cm2)', width=200, stretch=True)

        self.tree.pack(pady=10, expand=True, fill='both')

        # Scrollbar for the Treeview
        scrollbar = ttk.Scrollbar(master, orient=tk.VERTICAL, command=self.tree.yview)
        self.tree.configure(yscrollcommand=scrollbar.set)
        scrollbar.pack(side=tk.RIGHT, fill=tk.Y)

        # Start the asyncio loop integration
        self.master.after(100, self.process_asyncio_events)

    def process_asyncio_events(self):
        try:
            self.loop.call_soon(self.loop.stop)
            self.loop.run_forever()
        except Exception as e:
            print(f"Error in asyncio loop: {e}", file=sys.stderr)
        self.master.after(100, self.process_asyncio_events)

    async def connect(self):
        try:
            # Scanning for devices
            self.status_label.config(text="Status: Connecting", fg="orange")
            devices = await BleakScanner.discover()
            for d in devices:
                if d.name == DEVICE_NAME or d.address == DEVICE_NAME:
                    self.client = BleakClient(d.address)
                    try:
                        await self.client.connect()
                        self.status_label.config(text="Status: Connected", fg="green")
                        self.start_button.config(state=tk.NORMAL)
                        self.disconnect_button.config(state=tk.NORMAL)
                        self.connect_button.config(state=tk.DISABLED)

                        # Set up disconnection callback
                        self.client.set_disconnected_callback(self.on_disconnect)
                    except Exception as e:
                        messagebox.showerror("Connection Error", str(e))
                    return
            messagebox.showerror("Device Not Found", f"Could not find device {DEVICE_NAME}")
        except Exception as e:
            messagebox.showerror("Error", str(e))

    def connect_device(self):
        self.loop.create_task(self.connect())

    async def disconnect(self):
        if self.client and self.client.is_connected:
            try:
                await self.client.disconnect()
                self.status_label.config(text="Status: Disconnected", fg="red")
                self.start_button.config(state=tk.DISABLED)
                self.disconnect_button.config(state=tk.DISABLED)
                self.connect_button.config(state=tk.NORMAL)

                # Stop data collection if it's running
                if self.is_collecting:
                    await self.stop_notify()
            except Exception as e:
                messagebox.showerror("Disconnection Error", str(e))

    def disconnect_device(self):
        self.loop.create_task(self.disconnect())

    def on_disconnect(self, client):
        self.loop.call_soon_threadsafe(self.handle_disconnect)

    def handle_disconnect(self):
        self.status_label.config(text="Status: Disconnected", fg="red")
        self.start_button.config(state=tk.DISABLED)
        self.disconnect_button.config(state=tk.DISABLED)
        self.connect_button.config(state=tk.NORMAL)

        # If collecting data, update the buttons
        if self.is_collecting:
            self.is_collecting = False
            self.stop_button.config(state=tk.DISABLED)
            self.start_button.config(state=tk.DISABLED)
            self.save_button.config(state=tk.NORMAL)
            self.clear_button.config(state=tk.NORMAL)

    async def start_notify(self):
        if self.client is None or not self.client.is_connected:
            messagebox.showwarning("Warning", "Device not connected")
            return

        self.is_collecting = True
        self.stop_button.config(state=tk.NORMAL)
        self.start_button.config(state=tk.DISABLED)

        def notification_handler(sender, data):
            # Assume data is received as a string, decode it
            decoded_data = data.decode("utf-8").strip()

            # If the special message is received, update the label and stop parsing
            if "30 readings! Remove sensor" in decoded_data:
                self.notification_label.config(text="30 readings received! Please remove sensor.")
                return  # Do not process this data into the table

            # Parse the string "PressureLevel,1000,5000000"
            try:
                pressure_level, pressure_value, sensor_data = decoded_data.split(',')

                # Capacitance calculations
                fsensor = FREF * (int(sensor_data) / TwoPow28)
                totalcap = (10**12) * (1 / (Lsensor * (10**-6) * ((2 * math.pi * fsensor) ** 2)))
                skincap = round(totalcap - Csensor, 6)
                # normskincap = round(skincap / sensorArea, 6)

                # timestampe = datetime.now().strftime("%H:%M:%S")

                self.data.append({
                    'PressureLevel': pressure_level,
                    'PressureValue': int(pressure_value),
                    'SensorData': int(sensor_data),
                    'Total Capacitance (pF)': skincap,
                    # 'Normalized Capacitance (pF/cm$^2$)': normskincap,
                    # 'Timestamp (HH:MM:SS)': timestampe
                    # 'Skin Type': self.skin_type_var.get()  # Added skin type
                })

                # Insert data into Treeview
                self.tree.insert('', tk.END, values=(
                    pressure_level,
                    pressure_value,
                    sensor_data,
                    skincap,
                    # normskincap
                    # timestampe
                ))

                # Auto-scroll to the bottom of the Treeview
                self.tree.yview_moveto(1.0)

            except ValueError:
                print(f"Error parsing data: {decoded_data}")

        try:
            await self.client.start_notify(CHARACTERISTIC_UUID, notification_handler)
        except Exception as e:
            messagebox.showerror("Error", str(e))

    def start_collection(self):
        self.loop.create_task(self.start_notify())

        # Enable Clear Data button
        self.clear_button.config(state=tk.NORMAL)

    async def stop_notify(self):
        if self.client:
            try:
                await self.client.stop_notify(CHARACTERISTIC_UUID)
            except BleakError as e:
                messagebox.showerror("Error", str(e))
        self.is_collecting = False
        self.stop_button.config(state=tk.DISABLED)
        self.start_button.config(state=tk.NORMAL)
        self.save_button.config(state=tk.NORMAL)
        # Enable Clear Data button
        self.clear_button.config(state=tk.NORMAL)

    def stop_collection(self):
        self.loop.create_task(self.stop_notify())

    def save_data(self):
        # Get the skin type
        skin_type = self.skin_type_var.get()
        person_type = self.person_var.get()
        test_number = self.test_number_var.get()
        if person_type == "Volunteer":
            person_abbv = "V"
        elif person_type == "Control":
            person_abbv = "C"
        else:
            person_abbv = "P"
        study_number = self.study_part2_var.get()

        # Get the current timestamp
        timestamp = datetime.now().strftime("%H%M_%d_%m_%Y")

        # Create the default filename
        default_filename = f"{person_abbv}{study_number}_{skin_type}{test_number}_{timestamp}.csv"

        file_path = filedialog.asksaveasfilename(
            defaultextension=".csv",
            initialfile=default_filename,
            filetypes=[("CSV files", "*.csv")]
        )
        if file_path:
            df = pd.DataFrame(self.data)
            df.to_csv(file_path, index=False)
            messagebox.showinfo("Save Data", "Data saved successfully")

    def clear_data(self):
        self.data.clear()

        # Clear the Treeview table
        for item in self.tree.get_children():
            self.tree.delete(item)

        # Reset notification for readings taken
        self.notification_label.config(text="")

        # Disable Save and Clear buttons since there's no data
        self.save_button.config(state=tk.DISABLED)
        self.clear_button.config(state=tk.DISABLED)


def main():
    # Set the event loop policy to WindowsSelectorEventLoopPolicy
    if sys.platform.startswith('win'):
        asyncio.set_event_loop_policy(asyncio.WindowsSelectorEventLoopPolicy())

    root = tk.Tk()
    app = BLEApp(root)
    root.mainloop()


if __name__ == "__main__":
    main()
