import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
import seaborn as sns
from cycler import cycler
from scipy.stats import pearsonr
from sklearn.metrics import roc_curve, roc_auc_score

plt.rcParams.update({
    'font.size': 14,  # Default font size for all text
    'font.weight': 'bold',  # Make all text bold
    'font.style': 'normal',  # Set font style to italic
    'font.family': 'Arial',  # Set default font family (you can change to 'sans-serif', 'monospace', etc.)
    'axes.titlesize': 18,  # Title font size
    'axes.titleweight': 'bold',  # Title font weight
    'axes.labelsize': 16,  # X and Y labels font size
    'axes.labelweight': 'bold',  # X and Y labels font weight
    # Added the Origin Pro colors from their built-in palette -
    # gray, red, blue, green,
    # purple, ochre, cyan, brown,
    # olive, orange, sky-blue
    'axes.prop_cycle': cycler('color', ['#515151', '#f14040', '#1a6fdf', '#37ad6b',
                                        '#b177de', '#cc9900', '#00cbcc', '#7d4e4e',
                                        '#8e8e00', '#fb6501', '#6699cc'])
})

def calculate_statistics(data):
    means = []
    stddevs = []
    patients = []
    for patient, values in data.items():
        means.append(np.mean(values))
        stddevs.append(np.std(values))
        patients.append(patient)
    return patients, means, stddevs

# Load up data 
lesional_sensor = pd.read_csv('lesional_sensor.csv').iloc[:, 1:].to_dict(orient='list')
nonlesional_sensor = pd.read_csv('nonlesional_sensor.csv').iloc[:, 1:].to_dict(orient='list')

# Clean NaN values
lesional_sensor = {k: [v for v in vals if pd.notna(v)] for k, vals in lesional_sensor.items()}
nonlesional_sensor = {k: [v for v in vals if pd.notna(v)] for k, vals in nonlesional_sensor.items()}

# Calculate means 
lesional_patients, lesional_means, lesional_stddevs = calculate_statistics(lesional_sensor)
nonlesional_patients, nonlesional_means, nonlesional_stddevs = calculate_statistics(nonlesional_sensor)


# Scatter plot with error bars
plt.figure(figsize=(10, 6))
plt.errorbar(lesional_patients, lesional_means, yerr=lesional_stddevs, fmt='o', label='Lesional', capsize=5)
plt.errorbar(nonlesional_patients, nonlesional_means, yerr=nonlesional_stddevs, fmt='o', label='Non-Lesional', capsize=5)

# Customize plot
plt.xlabel('Patient Number')
plt.ylabel('Mean Sensor Capacitance (pF)')
plt.title("Sensor Capacitance Values: Lesional vs Non-Lesional Skin of Patients")
plt.xticks(range(len(lesional_patients)), range(1, len(lesional_patients) + 1))
# plt.xticks(rotation=45)
plt.grid(True, linestyle='--', alpha=0.7)
plt.legend()
plt.tight_layout()
# plt.savefig('SensorCapLvsNL_allPat.pdf', format='pdf')

plt.show()